# move a polynomial f from oscar to HC
# -------------  Input:
# f              an oscar polynomial
# varstring      HC variables
# -------------  Output:
# a polynomial in HC

function oscar_to_HC_Q(f::QQMPolyRingElem, varstring::Vector{Variable})::Expression
    
    coeffs = convert.(Rational{Int64},collect(Oscar.coefficients(f)))
    exps = collect(Oscar.exponents(f))
    sum([coeffs[i]*prod(varstring.^exps[i]) for i = 1:length(coeffs)])

end

# Translates HC expressions to Oscar polynomials

# ------------------- Input:
# f                   Expression
# varstring           Vector{QQMPolyRingElem}  - Oscar ring gens
# HC_vars             Vector{Variable}         - HC variables
# ------------------- Output:
# f                   QQMPolyRingElem  

function HC_to_oscar(f::Expression, varstring::Vector{QQMPolyRingElem}, HC_vars::Vector{Variable})
    E,C = exponents_coefficients(f,HC_vars);
    sum([Rational.(C[i])*prod(varstring.^E[:,i]) for i = 1:length(C)])
end

# For m create a matrix representing point on M_0,m
# -------------  Input:
# m              an integer >= 4 ::Int64
# -------------  Output:
# sigma          a matrix ::AbstractAlgebra.Generic.MatSpaceElem{ZZMPolyRingElem}
# R              a ring of sigma ::ZZMPolyRing
# x              vars of R ::Vector{ZZMPolyRingElem}

function create_point_of_M0m(m::Int64; gauge = false)
    
    if gauge
        R, x = polynomial_ring(Oscar.QQ, ["x$i" for i = 3:m-1])

        row1 = [ones(Int, m-1); 0]
        row2 = vcat([zero(R), one(R)], x, [one(R)])
    
        sigma = AbstractAlgebra.matrix(R,[transpose(row1); transpose(row2)])

        return sigma, R, x
    end

    R, x = polynomial_ring(Oscar.QQ, ["x$i" for i = 1:m])

    row1 = ones(Int, m);
    row2 = x
    
    sigma = AbstractAlgebra.matrix(R, [transpose(row1); transpose(row2)])

    return sigma, R, x
end

# For a linear system return the coefficient matrix A, and the kernel basis
# -------------  Input:
# eqns           linear system in Oscar or HC
# -------------  Output:
# A              coefficients matrix
#                matrix with columns vectors forming basis of the kernel of A

function get_coeffs_kernel(s, eqns::Vector{Expression})

    if typeof(eqns) == Vector{Expression}

        coeff = []
        for f in eqns
            A0, M0 = exponents_coefficients(f, s)
            push!(coeff, A0*M0)
        end

        A = transpose(hcat(coeff...))
        return A, nullspace(A)
    end
end

# the same in Oscar

function get_coeffs_kernel_Oscar(eqns::Union{Vector{Expression}, Vector{QQMPolyRingElem}})

    if typeof(eqns) == Vector{Expression}
        R, var_oscar = polynomial_ring(QQ, :x => 1:length(variables(eqns)))
        eqns = [HC_to_oscar(f, var_oscar, variables(eqns)) for f in eqns]
    end
    
    coeffs = []
    for i in 1:length(eqns)
        if eqns[i] != 0
            push!(coeffs, sum(collect(Oscar.exponents(eqns[i])), dims=1))
        end
    end
    A = Float64.(hcat(vcat(coeffs...)...))'

    for i in 1:size(A,1)
        A[i, findall(x -> x==1, A[i, :])] = Rational.(collect(Oscar.coefficients(eqns[i])))
    end
    
    S = matrix_space(QQ, size(A, 1), size(A, 2))
    return Float64.(A), Float64.(nullspace(S(Rational.(A)))[2])
end

# get a product of minors corresponding to edges of a graph
# -------------  Input:
# n              Int64
# zeros          Vector{Tuple{Any, Any}} of nonedges of the graph
# x              HC variables
# -------------  Output:
# product of minors corresponding to edges of a graph

function get_prod_minors(n::Int64, zeros, x)
    minors = []
    for j in 2:n
        for i in 1:(j-1)
            if !((i,j) in zeros)
                push!(minors,x[j]-x[i])
            end
        end
    end
    return prod(minors)
end

# get a list of all binom(n,2) pairs (i,j)
# -------------  Input:
# n              Int64
# -------------  Output:
# a list of all binom(n,2) pairs (i,j)

function get_all_tuples(n::Int64)
    alltuples = []
    for j in 2:n
        for i in 1:(j-1)
            push!(alltuples, (i,j))
        end
    end
    return alltuples
end

# get edges of G from nonedges

function get_edges(n, NE)
    allpairs::Vector{Vector{Int64}} = [[i,j] for i in 1:n-1 for j in (i+1):n]
    return [e for e in allpairs if(e ∉ NE)]
end

# get coefficient matrix of linear equations in homotopy continuation

function get_coeff_mat_HC(eqs, var_HC)
    E, C = HomotopyContinuation.exponents_coefficients(eqs[1], var_HC);
    coeff_mat = transpose(E*C)
    for eq in eqs[2:end]
        E, C = HomotopyContinuation.exponents_coefficients(eq, var_HC);
        coeff_mat = vcat(coeff_mat, transpose(E*C))
    end
    return coeff_mat
end

function read_m2_list_of_lists(filename::String)
    text = read(filename, String);

    # Remove BOM if present
    text = replace(text, r"^\ufeff" => "");

    # Remove whitespace
    text = replace(text, r"\s+" => "");

    # Split by top-level "}," to separate the inner lists 
    #chunks = split(text, "},") # If comma at end of each linea

    # Split by top-level "}" to separate the inner lists 
    chunks = split(text, "}"); #(if no comma at the end)

    result = []
    for chunk in chunks
        # Remove braces { and }
        clean = replace(chunk, r"[{}]" => "")

        # Split by commas into "sij"
        symbols = split(clean, ",")
        symbols = filter(!isempty, symbols)

        # Parse each sij into [i,j]
        push!(result, [[parse(Int, s[2]), parse(Int, s[3])] for s in symbols])
    end

    return result
end