# get edges of a complete graph

function get_complete_graph(components)
    completegraph = []
    for c in components
        push!(completegraph, collect(powerset(c,2,2)))
    end

    return completegraph
end

# check whether a set of 2-tuples is disjoint
# -------------  Input:
# edge_subset    Vector{Vector{Int64}}
# -------------  Output:
# bool

function is_union_of_completegraphs(n, edges)
    G = Oscar.graph_from_edges(edges, n)
    components =  sort(connected_components(G))

    for c in components
        for v in c
            if Oscar.degree(G, v) != length(c) - 1
                return false
            end
        end
    end

    return true
end

# det dimension of a fiber over a boundary stratum of M_G
# -------------  Input:
# n              Int64
# subset         Vector{Tuple{Any, Any}} of nonedges that form union of cliques in the complement of G
# components     Vector{Vector} of vertices representing a connected component of this subset
# -------------  Output:
# dimension of the kinematic fiber 

function get_dim_fiber(n, subset, components)

    completegraph = get_complete_graph(components)
        
    s, eqns = momentum_conservation(n, completegraph, HC = true)
    x, s, I = ScatteringGraphs.likelihoodIdeal_complete(n, HC = true)
            
    ideal_nonedges = [s[Int((j^2-3*j+2)//2 + i)] for (i,j) in subset]
    new_components = [[j for i in 1:length(components[j])] for j in 1:length(components)]
            
    new_x = Expression.(x)
    for j in 1:length(components)
        new_x[components[j]] = new_components[j]
    end

    S = vcat(ideal_nonedges, eqns, HomotopyContinuation.evaluate(I, x => new_x))
    _, M = get_coeffs_kernel(s, S)

    return(size(M,2))
end

# finds ML degree via alternative sum of euler characteristics OVER COMPLETE GRAPHS
# -------------  Input:
# n              Int64
# nonedges       Vector{Tuple{Any, Any}} of nonedges of the graph
# -------------  Output:
# ML degree

function ML_deg_formula(n::Int64, nonedges)
    all_subsets = collect(Combinatorics.powerset(nonedges))[2:end]

    euler_dict = Dict()
    euler_dict[[]] = (-1)^(n-3)*factorial(n-3)
    for subset in all_subsets
        edges = [[e[1],e[2]] for e in subset]
        
        if is_union_of_completegraphs(n, edges)
            G = Oscar.graph_from_edges(Directed, edges)
            G1 = Oscar.graph_from_edges(edges, n)
            components =  sort(connected_components(G1))

            k = rank(matrix(QQ, signed_incidence_matrix(G)))

            #println(edges, ":   k = ", k, "   fiber: ", get_dim_fiber(n, subset, components))

            if (n-k-3 >= 0) 
                euler_dict[components] = (-1)^(n-k-3)*factorial(n-k-3)
            end
        end
    end

    euler_vector = collect(values(euler_dict))
    println("Euler:   ", euler_vector)
    return (sum(euler_vector))
end

# filter a list of given graphs to get graphs compatible with momentum conservation
# -------------  Input:
# n              Int64
# all_graphs     Vector{Tuple{Any, Any}} list of graphs given by nonedges 
# -------------  Output:
# graph_nonedges Vector{Tuple{Any, Any}} list of nonedges of compatible graphs

function get_compatible_graphs(n, all_graphs)

    alltuples = get_all_tuples(n)
    
    graph_nonedges = []
    for nonedges in all_graphs

        edges = setdiff(alltuples,nonedges)
        a = get_data(n, nonedges)

        if length(nonedges) == count(isapprox(0, atol = 10^(-8)), a)
            push!(graph_nonedges, nonedges)
        end
    end
    return graph_nonedges
end

# one of the matroidally copious conditions
function check_condition_18(n, nonedges)
    s, eqns = momentum_conservation(n, nonedges, HC = true)
    M_K, _ = get_coeffs_kernel(s, eqns::Vector{Expression})
    
    kG = rank(M_K)
    dimKG =  binomial(n,2) - length(nonedges) - kG

    x, s, eqns, F = get_scattering_eqns(n, nonedges)
    x0 = randn(ComplexF64, n)
    S = vcat(eqns, HomotopyContinuation.evaluate(F, x => x0))
    _, M = get_coeffs_kernel(s, S)
    
    for i in 1:20
        x0 = randn(ComplexF64, n)
        S = vcat(eqns, HomotopyContinuation.evaluate(F, x => x0))
        A, M_S = get_coeffs_kernel(s, S)

        M = hcat(M, M_S)
    end

    if rank(M) - length(nonedges) != dimKG 
        return false
    end

    return true
end

# check if the graph is copious
function is_copious(n, nonedges)

    nedges = binomial(n,2) - length(nonedges)
    s, eqns = momentum_conservation(n, nonedges, HC = true)
    M_K, _ = get_coeffs_kernel(s, eqns::Vector{Expression})

    x, s,eqns, F = get_scattering_eqns(n, nonedges)

    kG = rank(M_K)
    
    x0 = randn(ComplexF64, n)
    S = vcat(eqns, HomotopyContinuation.evaluate(F, x => x0))
    M_S, _ = get_coeffs_kernel(s, S)
    
    if rank(M_S) != (kG + n - 3)
        println("Condition 17 fails")
        return false
    end

    for j in 1:size(M_S,2)
        L = M_S[:,setdiff(1:size(M_S,2),j)]
        if rank(L) != (kG + n - 3)
            println("Condition candidate fails")
            return false
        end
    end
        
    if check_condition_18(n, nonedges)
        return true
    else
        println("Condition 18 fails")
        return false
    end
end