
# get I_G^1 for a complete graph

function likelihoodIdeal_complete(n::Int64; HC = false)
    
    if HC
        S = Matrix{Expression}(undef, n, n)
        for i in 1:n, j in 1:n
            if (i == j) 
                S[i,j] = 0
            elseif j > i
                S[i,j] = Variable(:s, i, j)
            else 
                S[i,j] = Variable(:s, j, i)
            end
        end
    
        @var x[1:n]
    
        ideal = Expression[]
    
        for k in 3:(n-1)
            poly_supp = []
            
            for j in 2:n, i in 1:j-1
                push!(poly_supp, S[i,j]*sum([x[i]^(l)*x[j]^(k-1-l) for l in 0:(k-1)]))
            end
            
            push!(ideal, sum(poly_supp))
        end
    
        return x, variables(S), ideal
    end

    xnames = ["x_$i" for i in 1:n]
    snames = String[]
    index_map = Dict{Tuple{Int,Int}, Int}()
    var_count = 0
    for j in 2:n
        for i in 1:(j-1)
            var_count += 1
            name = "s_$(i)$(j)"
            push!(snames, name)
            index_map[(i,j)] = var_count
            index_map[(j,i)] = var_count  # symmetric
        end
    end
    
    names = vcat(xnames, snames)

    R, var = polynomial_ring(QQ, names)

    x_vars = var[1:n]
    s_vars = var[n+1:end]
    
    S = Matrix{Any}(undef, n, n)
    for i in 1:n, j in 1:n
        if (i == j)
            S[i,j] = zero(R)
        else
            idx = index_map[(i,j)]
            S[i,j] = s_vars[idx]
        end
    end

    ideal = []

    for k in 3:(n-1)
        poly_supp = []
            
        for j in 2:n, i in 1:j-1
            push!(poly_supp, S[i,j]*sum([x_vars[i]^(l)*x_vars[j]^(k-1-l) for l in 0:(k-1)]))
        end
            
        push!(ideal, sum(poly_supp))
    end
    
    
    return R, x_vars, s_vars, ideal  
end

####################################### MULTIDEGREE CONPUTATION #####################################################

# Create bigraded ring Q[a,b,c..., s12,..,], with s_ij iff ij \not\in NE
# Output
# R     polynomial ring
# x     x variables from 1 to n
# S     S variables, one for each edges
# s     s[[i,j]] = s variable corresponding to edge [i,j]  

function bigraded_ring(n, NE) 
    E = get_edges(n, NE)
    vars = vcat(["x$i" for i in 1:n],["s$j" for j in E])
    #deg = vcat([[1,0] for i in vars[1:n]],[[0,1] for i in vars[n+1:end]]) # deg(x)=(0,1), deg(s[(i,j)])=(1,0)
    #R, X = graded_polynomial_ring(QQ, vars, deg )
    R, X = polynomial_ring(QQ, vars )
    x = X[1:n]
    S = X[n+1:end]
    s = Dict(E[k] => S[k] for k = 1:length(E)) # allow to call s_ij as s[[i,j]]
    return (R, x, S, s)
end

# Create momentum conservation equations in the bigraded ring

function get_momentum_conservation(n, NE)  
    (R, x, S, s) = bigraded_ring(n, NE)
    E = get_edges(n, NE)
    isolated_vertices = findall( i-> [s[a] for a in E if a[1] == i || a[2] == i] ==[], 1:n)
    mom_cons_nonzero = [sum(s[a] for a in E if a[1] == i || a[2] == i) for i in 1:n if (i∉ isolated_vertices)  ]
    return mom_cons_nonzero
end

# Compute the scattering equations of G with cleared denominators

function get_cleared_scatt_eqs(n, NE)
    E = get_edges(n, NE)
    (R, x, S, s) = bigraded_ring(n, NE)
    scatt_eqs_no_den = [] #vector with scattering equations after clearing denominator
    for i=1:n
        N = [e for e in E if i∈e ] #neighbors of i
        prod_neighb = prod([x[e[2]]-x[e[1]] for e in N])
        scatt = 0
        for e in N 
            scatt = scatt + prod_neighb * s[e] * derivative( x[e[2]]-x[e[1]], x[i])/ (x[e[2]]-x[e[1]])
        end
        push!(scatt_eqs_no_den, scatt)
    end
    return scatt_eqs_no_den
end

# compute scattering equations in HC variables as rational functions

function get_scattering_eqns_HC(n, NE)
    E = get_edges(n, NE)

    x_HC = Variable[]
    for i = 1:n
        push!(x_HC, Variable(:x, i))
    end

    s_HC = Variable[]
    for e in E
        push!(s_HC, Variable(:s, e[1], e[2]))
    end
    hyperplanes = [x_HC[e[2]]-x_HC[e[1]] for e in E]
   
    L = sum([s_HC[i]*log(hyperplanes[i]) for i = 1:length(s_HC)])
    F = differentiate(L,x_HC)
    
    return x_HC, s_HC, F
end

# symmetric functions to generate in I_G^1

function theta(n, NE, h)
    E = get_edges(n,NE)
    (R, x, S, s) = bigraded_ring(n, NE)
    G = gens(R)  # all generators
    # loop over edges in E
    terms = [ s[e] * sum(Oscar.monomials((G[e[1]] + G[e[2]])^h)) for e in E]
    return sum(terms)
end


function get_codimension_kinematic_space(n, NE)
    (R, x, S, s) = bigraded_ring(n, NE)
    mom_cons_oscar = get_momentum_conservation(n, NE)
    Mat_mom_cons, _ =  get_coeffs_kernel_Oscar(mom_cons_oscar);
    k = rank(Mat_mom_cons)
    r = size(Mat_mom_cons, 1)
    return k, r
end

# Compute start solution (start_x,start_s) and start parameters L and M. 
# Use it for monodromy after setting coefficients of L and M as parameters

function get_start_sol_multideg(n, NE, i)
    (R, x, S, s) = bigraded_ring(n, NE)
    mom_cons_oscar = get_momentum_conservation(n, NE)
    Mat_mom_cons, _ =  get_coeffs_kernel_Oscar(mom_cons_oscar);
    k = rank(Mat_mom_cons)
    c = n + k - 3  
   
    start_x = randn(Float64, n)
    E = get_edges(n, NE)

    x_HC, s_HC, scatt_eqs_HC = get_scattering_eqns_HC(n, NE)
    
    eqs_oscar = vcat(mom_cons_oscar,  [theta(n, NE, h) for h = 2:n-2]);
    
    mom_cons_HC = [oscar_to_HC_Q(eq, vcat(x_HC, s_HC)) for eq in mom_cons_oscar];
    
    eqs_HC = vcat(mom_cons_HC, scatt_eqs_HC)
    eqs_HC_ev = [HomotopyContinuation.evaluate(eq, x_HC => start_x) for eq in eqs_HC]; 
    
    M = get_coeff_mat_HC(eqs_HC_ev, s_HC);
    Ker = nullspace(M)
    m = size(Ker, 2)
    coeffs = randn(Float64, m);
    start_s = Ker*coeffs;
    
    L0 = (transpose(nullspace(transpose(start_s))));
    L0 = transpose(randn(ComplexF64,  size(L0,1), length(E)-1-i ,)) * L0;

    L1 = (transpose(nullspace(transpose( vcat(start_s,[1])))));
    L1 = transpose(randn(ComplexF64,  size(L1,1),  1,)) * L1 # 1 affine eq in s

    L = vcat( hcat(L0, zeros(size(L0,1))), L1);

    M0 = (transpose(nullspace((transpose(start_x)))));
    M0 = transpose(randn(ComplexF64,  size(M0,1), n-1-(c-i),)) * M0;
    
    M1 = (transpose(nullspace((transpose( vcat(start_x,[1])) ) )));
    M1 = transpose(randn(ComplexF64,  size(M1,1),1 )) * M1

    M = vcat( hcat(M0, zeros(size(M0,1))), M1) 

    return start_x, start_s,  L,  M
end

# Compute number of solutions in V(I) ∩ L ∩ M, where L and M are subspaces of P^|G|-1 and P^n-1 

function get_system_HC(n, NE, i)
    (R, x, S, s) = bigraded_ring(n, NE)
    mom_cons_oscar = get_momentum_conservation(n, NE)
    Mat_mom_cons, _ =  get_coeffs_kernel_Oscar(mom_cons_oscar);
    k = rank(Mat_mom_cons)
    c = n + k - 3  
   
    start_x, start_s,  L_start, M_start  = get_start_sol_multideg(n, NE, i)
    E = get_edges(n, NE)
    
    x_HC, s_HC, scatt_eqs_HC = get_scattering_eqns_HC(n, NE)

    @var l[1:(length(E)-1-i + 1), 1:length(s_HC)]
    @var l1

    @var m[1:(n-1-(c-i)+1), 1:n] 
    @var m1

    L_param = hcat(l, zeros(size(l,1))) 
    L_param[end,end] = l1
    
    M_param = hcat(m, zeros(size(m,1))) 
    M_param[end,end] = m1

    mom_cons_HC = [oscar_to_HC_Q(eq, vcat(x_HC, s_HC)) for eq in mom_cons_oscar];
    
    eqs_HC = vcat(mom_cons_HC, scatt_eqs_HC,  L_param*vcat(s_HC,[1]), M_param*(vcat(x_HC,[1])))
    
    start_sol = vcat(start_x, start_s);
    start_param = vcat(L_start[:,1:end-1]..., L_start[end,end], (M_start)[:,1:end-1]..., M_start[end,end]...) ;

    L_target = randn(ComplexF64, length(E) - 1 - i + 1, length(s_HC) + 1 );
    L_target[1:end-1,end] = zeros(size(L_target,1)-1)

    M_target = randn( ComplexF64, n - 1 - (c - i) + 1, length(x_HC) + 1);
    M_target[1:end-1,end] = zeros(size(M_target,1)-1)

    target_param = vcat(L_target[:,1:end-1]..., L_target[end,end], (M_target)[:,1:end-1]..., M_target[end,end]...) 
    
    F_HC = System( eqs_HC, variables = vcat(x_HC,s_HC), parameters = vcat(l...,l1, m..., m1))
    
    return F_HC, start_sol, start_param, target_param
end

function get_numerical_multidegree(n, NE)

    NE = [[e[1],e[2]] for e in NE]
    (R, x, S, _ ) = bigraded_ring(n, NE)
    mom_cons_oscar = get_momentum_conservation(n, NE)
    Mat_mom_cons, _ =  get_coeffs_kernel_Oscar(mom_cons_oscar);
    k = rank(Mat_mom_cons)
    
    coeff_multideg = []
    for i = k:k+n-4 
        F_HC, start_sol, start_param, target_param = get_system_HC(n, NE, i);
        #norm(F_HC( start_sol , start_param)) 
        @time res = HomotopyContinuation.monodromy_solve(F_HC, start_sol, start_param ) # THIS WORKS
        push!(coeff_multideg, length(solutions(res))) 
    end

    CR, (t, s) = polynomial_ring(QQ,["t","s"])
    multideg = (transpose(coeff_multideg) * [s^i*t^(n+k-3-i) for i=k:k+n-4]) + s^(k+n-3)

    return multideg
end





