/*******************************************************************************
 * Copyright (C) 2017-2025 Theodore Chang
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/

#include "IntegrationPlan.h"

arma::mat generate_points(const unsigned dimension, const std::pair<arma::vec, arma::vec>& points) {
    const auto& [location, weight] = points;

    if(1 == dimension) return join_rows(location, weight);

    arma::mat int_pts;

    const auto order = location.n_rows;

    auto counter = 0;

    if(2 == dimension) {
        int_pts.set_size(order * order, 3);
        for(auto i = 0llu; i < order; ++i)
            for(auto j = 0llu; j < order; ++j) {
                int_pts(counter, 0) = location(i);
                int_pts(counter, 1) = location(j);
                int_pts(counter++, 2) = weight(i) * weight(j);
            }
        return int_pts;
    }

    if(3 == dimension) {
        int_pts.set_size(order * order * order, 4);
        for(auto i = 0llu; i < order; ++i)
            for(auto j = 0llu; j < order; ++j)
                for(auto k = 0llu; k < order; ++k) {
                    int_pts(counter, 0) = location(i);
                    int_pts(counter, 1) = location(j);
                    int_pts(counter, 2) = location(k);
                    int_pts(counter++, 3) = weight(i) * weight(j) * weight(k);
                }
        return int_pts;
    }

    throw std::invalid_argument("not supported");
}

template<IntegrationType> std::pair<arma::vec, arma::vec> generate_seeds(unsigned) { throw std::invalid_argument("not supported"); }

template<> std::pair<arma::vec, arma::vec> generate_seeds<IntegrationType::GAUSS>(const unsigned order) {
    arma::vec PTL(order, arma::fill::none), PTW(order, arma::fill::none);

    switch(order) {
    case 1: {
        PTL(0) = 0.0000000000000000e+00;

        PTW(0) = 2.0000000000000000e+00;

        break;
    }
    case 2: {
        PTL(0) = -5.7735026918962573e-01;

        PTW(0) = 1.0000000000000000e+00;

        break;
    }
    case 3: {
        PTL(0) = -7.7459666924148340e-01;
        PTL(1) = 0.0000000000000000e+00;

        PTW(0) = 5.5555555555555580e-01;
        PTW(1) = 8.8888888888888828e-01;

        break;
    }
    case 4: {
        PTL(0) = -8.6113631159405257e-01;
        PTL(1) = -3.3998104358485631e-01;

        PTW(0) = 3.4785484513745379e-01;
        PTW(1) = 6.5214515486254621e-01;

        break;
    }
    case 5: {
        PTL(0) = -9.0617984593866396e-01;
        PTL(1) = -5.3846931010568311e-01;
        PTL(2) = 0.0000000000000000e+00;

        PTW(0) = 2.3692688505618897e-01;
        PTW(1) = 4.7862867049936653e-01;
        PTW(2) = 5.6888888888888900e-01;

        break;
    }
    case 6: {
        PTL(0) = -9.3246951420315205e-01;
        PTL(1) = -6.6120938646626448e-01;
        PTL(2) = -2.3861918608319693e-01;

        PTW(0) = 1.7132449237917016e-01;
        PTW(1) = 3.6076157304813855e-01;
        PTW(2) = 4.6791393457269120e-01;

        break;
    }
    case 7: {
        PTL(0) = -9.4910791234275838e-01;
        PTL(1) = -7.4153118559939446e-01;
        PTL(2) = -4.0584515137739718e-01;
        PTL(3) = 0.0000000000000000e+00;

        PTW(0) = 1.2948496616886995e-01;
        PTW(1) = 2.7970539148927670e-01;
        PTW(2) = 3.8183005050511876e-01;
        PTW(3) = 4.1795918367346907e-01;

        break;
    }
    case 8: {
        PTL(0) = -9.6028985649753629e-01;
        PTL(1) = -7.9666647741362673e-01;
        PTL(2) = -5.2553240991632899e-01;
        PTL(3) = -1.8343464249564984e-01;

        PTW(0) = 1.0122853629037562e-01;
        PTW(1) = 2.2238103445337473e-01;
        PTW(2) = 3.1370664587788760e-01;
        PTW(3) = 3.6268378337836205e-01;

        break;
    }
    case 9: {
        PTL(0) = -9.6816023950762609e-01;
        PTL(1) = -8.3603110732663577e-01;
        PTL(2) = -6.1337143270059036e-01;
        PTL(3) = -3.2425342340380892e-01;
        PTL(4) = 0.0000000000000000e+00;

        PTW(0) = 8.1274388361574135e-02;
        PTW(1) = 1.8064816069485759e-01;
        PTW(2) = 2.6061069640293549e-01;
        PTW(3) = 3.1234707704000275e-01;
        PTW(4) = 3.3023935500125962e-01;

        break;
    }
    case 10: {
        PTL(0) = -9.7390652851717174e-01;
        PTL(1) = -8.6506336668898443e-01;
        PTL(2) = -6.7940956829902444e-01;
        PTL(3) = -4.3339539412924721e-01;
        PTL(4) = -1.4887433898163116e-01;

        PTW(0) = 6.6671344308687139e-02;
        PTW(1) = 1.4945134915058053e-01;
        PTW(2) = 2.1908636251598224e-01;
        PTW(3) = 2.6926671930999674e-01;
        PTW(4) = 2.9552422471475331e-01;

        break;
    }
    case 11: {
        PTL(0) = -9.7822865814605697e-01;
        PTL(1) = -8.8706259976809532e-01;
        PTL(2) = -7.3015200557404936e-01;
        PTL(3) = -5.1909612920681192e-01;
        PTL(4) = -2.6954315595234496e-01;
        PTL(5) = 0.0000000000000000e+00;

        PTW(0) = 5.5668567116174308e-02;
        PTW(1) = 1.2558036946490389e-01;
        PTW(2) = 1.8629021092773451e-01;
        PTW(3) = 2.3319376459199045e-01;
        PTW(4) = 2.6280454451024660e-01;
        PTW(5) = 2.7292508677790056e-01;

        break;
    }
    case 12: {
        PTL(0) = -9.8156063424671924e-01;
        PTL(1) = -9.0411725637047491e-01;
        PTL(2) = -7.6990267419430469e-01;
        PTL(3) = -5.8731795428661748e-01;
        PTL(4) = -3.6783149899818013e-01;
        PTL(5) = -1.2523340851146897e-01;

        PTW(0) = 4.7175336386513188e-02;
        PTW(1) = 1.0693932599531782e-01;
        PTW(2) = 1.6007832854334608e-01;
        PTW(3) = 2.0316742672306573e-01;
        PTW(4) = 2.3349253653835469e-01;
        PTW(5) = 2.4914704581340261e-01;

        break;
    }
    case 13: {
        PTL(0) = -9.8418305471858814e-01;
        PTL(1) = -9.1759839922297792e-01;
        PTL(2) = -8.0157809073330988e-01;
        PTL(3) = -6.4234933944034034e-01;
        PTL(4) = -4.4849275103644687e-01;
        PTL(5) = -2.3045831595513483e-01;
        PTL(6) = 0.0000000000000000e+00;

        PTW(0) = 4.0484004765316141e-02;
        PTW(1) = 9.2121499837727605e-02;
        PTW(2) = 1.3887351021978761e-01;
        PTW(3) = 1.7814598076194554e-01;
        PTW(4) = 2.0781604753688859e-01;
        PTW(5) = 2.2628318026289745e-01;
        PTW(6) = 2.3255155323087401e-01;

        break;
    }
    case 14: {
        PTL(0) = -9.8628380869681220e-01;
        PTL(1) = -9.2843488366357352e-01;
        PTL(2) = -8.2720131506976502e-01;
        PTL(3) = -6.8729290481168548e-01;
        PTL(4) = -5.1524863635815410e-01;
        PTL(5) = -3.1911236892788974e-01;
        PTL(6) = -1.0805494870734363e-01;

        PTW(0) = 3.5119460331752915e-02;
        PTW(1) = 8.0158087159759556e-02;
        PTW(2) = 1.2151857068790295e-01;
        PTW(3) = 1.5720316715819366e-01;
        PTW(4) = 1.8553839747793782e-01;
        PTW(5) = 2.0519846372129555e-01;
        PTW(6) = 2.1526385346315760e-01;

        break;
    }
    case 15: {
        PTL(0) = -9.8799251802048538e-01;
        PTL(1) = -9.3727339240070595e-01;
        PTL(2) = -8.4820658341042721e-01;
        PTL(3) = -7.2441773136017007e-01;
        PTL(4) = -5.7097217260853883e-01;
        PTL(5) = -3.9415134707756339e-01;
        PTL(6) = -2.0119409399743454e-01;
        PTL(7) = 0.0000000000000000e+00;

        PTW(0) = 3.0753241996118154e-02;
        PTW(1) = 7.0366047488107153e-02;
        PTW(2) = 1.0715922046717176e-01;
        PTW(3) = 1.3957067792615432e-01;
        PTW(4) = 1.6626920581699411e-01;
        PTW(5) = 1.8616100001556224e-01;
        PTW(6) = 1.9843148532711163e-01;
        PTW(7) = 2.0257824192556131e-01;

        break;
    }
    case 16: {
        PTL(0) = -9.8940093499164994e-01;
        PTL(1) = -9.4457502307323260e-01;
        PTL(2) = -8.6563120238783176e-01;
        PTL(3) = -7.5540440835500300e-01;
        PTL(4) = -6.1787624440264377e-01;
        PTL(5) = -4.5801677765722743e-01;
        PTL(6) = -2.8160355077925892e-01;
        PTL(7) = -9.5012509837637454e-02;

        PTW(0) = 2.7152459411756466e-02;
        PTW(1) = 6.2253523938647630e-02;
        PTW(2) = 9.5158511682492314e-02;
        PTW(3) = 1.2462897125553363e-01;
        PTW(4) = 1.4959598881657638e-01;
        PTW(5) = 1.6915651939500212e-01;
        PTW(6) = 1.8260341504492328e-01;
        PTW(7) = 1.8945061045506811e-01;

        break;
    }
    case 17: {
        PTL(0) = -9.9057547531441736e-01;
        PTL(1) = -9.5067552176876780e-01;
        PTL(2) = -8.8023915372698602e-01;
        PTL(3) = -7.8151400389680137e-01;
        PTL(4) = -6.5767115921669084e-01;
        PTL(5) = -5.1269053708647694e-01;
        PTL(6) = -3.5123176345387636e-01;
        PTL(7) = -1.7848418149584777e-01;
        PTL(8) = 0.0000000000000000e+00;

        PTW(0) = 2.4148302868547192e-02;
        PTW(1) = 5.5459529373987050e-02;
        PTW(2) = 8.5036148317179164e-02;
        PTW(3) = 1.1188384719340382e-01;
        PTW(4) = 1.3513636846852553e-01;
        PTW(5) = 1.5404576107681064e-01;
        PTW(6) = 1.6800410215645020e-01;
        PTW(7) = 1.7656270536699295e-01;
        PTW(8) = 1.7944647035620676e-01;

        break;
    }
    case 18: {
        PTL(0) = -9.9156516842093090e-01;
        PTL(1) = -9.5582394957139760e-01;
        PTL(2) = -8.9260246649755581e-01;
        PTL(3) = -8.0370495897252314e-01;
        PTL(4) = -6.9168704306035322e-01;
        PTL(5) = -5.5977083107394754e-01;
        PTL(6) = -4.1175116146284269e-01;
        PTL(7) = -2.5188622569150559e-01;
        PTL(8) = -8.4775013041735320e-02;

        PTW(0) = 2.1616013526482385e-02;
        PTW(1) = 4.9714548894970838e-02;
        PTW(2) = 7.6425730254888746e-02;
        PTW(3) = 1.0094204410628754e-01;
        PTW(4) = 1.2255520671147817e-01;
        PTW(5) = 1.4064291467065074e-01;
        PTW(6) = 1.5468467512626527e-01;
        PTW(7) = 1.6427648374583265e-01;
        PTW(8) = 1.6914238296314366e-01;

        break;
    }
    case 19: {
        PTL(0) = -9.9240684384358435e-01;
        PTL(1) = -9.6020815213483002e-01;
        PTL(2) = -9.0315590361481779e-01;
        PTL(3) = -8.2271465653714282e-01;
        PTL(4) = -7.2096617733522939e-01;
        PTL(5) = -6.0054530466168099e-01;
        PTL(6) = -4.6457074137596088e-01;
        PTL(7) = -3.1656409996362989e-01;
        PTL(8) = -1.6035864564022534e-01;
        PTL(9) = 0.0000000000000000e+00;

        PTW(0) = 1.9461788229728973e-02;
        PTW(1) = 4.4814226765699280e-02;
        PTW(2) = 6.9044542737641171e-02;
        PTW(3) = 9.1490021622449694e-02;
        PTW(4) = 1.1156664554733375e-01;
        PTW(5) = 1.2875396253933588e-01;
        PTW(6) = 1.4260670217360638e-01;
        PTW(7) = 1.5276604206585920e-01;
        PTW(8) = 1.5896884339395403e-01;
        PTW(9) = 1.6105444984878325e-01;

        break;
    }
    case 20: {
        PTL(0) = -9.9312859918509488e-01;
        PTL(1) = -9.6397192727791370e-01;
        PTL(2) = -9.1223442825132595e-01;
        PTL(3) = -8.3911697182221889e-01;
        PTL(4) = -7.4633190646015080e-01;
        PTL(5) = -6.3605368072651491e-01;
        PTL(6) = -5.1086700195082702e-01;
        PTL(7) = -3.7370608871541955e-01;
        PTL(8) = -2.2778585114164504e-01;
        PTL(9) = -7.6526521133497297e-02;

        PTW(0) = 1.7614007139152687e-02;
        PTW(1) = 4.0601429800387480e-02;
        PTW(2) = 6.2672048334109332e-02;
        PTW(3) = 8.3276741576704269e-02;
        PTW(4) = 1.0193011981724026e-01;
        PTW(5) = 1.1819453196151841e-01;
        PTW(6) = 1.3168863844917644e-01;
        PTW(7) = 1.4209610931838176e-01;
        PTW(8) = 1.4917298647260360e-01;
        PTW(9) = 1.5275338713072559e-01;

        break;
    }
    default:
        throw std::invalid_argument("not supported");
    }

    for(auto I = 0u, J = order - 1; I < order / 2; ++I, --J) {
        PTL(J) = -PTL(I);
        PTW(J) = PTW(I);
    }

    return {std::move(PTL), std::move(PTW)};
}

template<> std::pair<arma::vec, arma::vec> generate_seeds<IntegrationType::HERMITE>(const unsigned order) {
    arma::vec PTL(order, arma::fill::none), PTW(order, arma::fill::none);

    switch(order) {
    case 2: {
        PTL(0) = -7.0710678118654746e-01;

        PTW(0) = 8.8622692545275794e-01;

        break;
    }
    case 3: {
        PTL(0) = -1.2247448713915889e+00;
        PTL(1) = 0.0000000000000000e+00;

        PTW(0) = 2.9540897515091918e-01;
        PTW(1) = 1.1816359006036774e+00;

        break;
    }
    case 4: {
        PTL(0) = -1.6506801238857842e+00;
        PTL(1) = -5.2464762327529035e-01;

        PTW(0) = 8.1312835447245047e-02;
        PTW(1) = 8.0491409000551295e-01;

        break;
    }
    case 5: {
        PTL(0) = -2.0201828704560851e+00;
        PTL(1) = -9.5857246461381851e-01;
        PTL(2) = 0.0000000000000000e+00;

        PTW(0) = 1.9953242059045886e-02;
        PTW(1) = 3.9361932315224113e-01;
        PTW(2) = 9.4530872048294179e-01;

        break;
    }
    case 6: {
        PTL(0) = -2.3506049736744918e+00;
        PTL(1) = -1.3358490740136968e+00;
        PTL(2) = -4.3607741192761651e-01;

        PTW(0) = 4.5300099055088630e-03;
        PTW(1) = 1.5706732032285656e-01;
        PTW(2) = 7.2462959522439263e-01;

        break;
    }
    case 7: {
        PTL(0) = -2.6519613568352334e+00;
        PTL(1) = -1.6735516287674712e+00;
        PTL(2) = -8.1628788285896459e-01;
        PTL(3) = 0.0000000000000000e+00;

        PTW(0) = 9.7178124509952402e-04;
        PTW(1) = 5.4515582819126940e-02;
        PTW(2) = 4.2560725261012788e-01;
        PTW(3) = 8.1026461755680745e-01;

        break;
    }
    case 8: {
        PTL(0) = -2.9306374202572441e+00;
        PTL(1) = -1.9816567566958427e+00;
        PTL(2) = -1.1571937124467802e+00;
        PTL(3) = -3.8118699020732211e-01;

        PTW(0) = 1.9960407221136621e-04;
        PTW(1) = 1.7077983007413412e-02;
        PTW(2) = 2.0780232581489169e-01;
        PTW(3) = 6.6114701255824149e-01;

        break;
    }
    case 9: {
        PTL(0) = -3.1909932017815272e+00;
        PTL(1) = -2.2665805845318427e+00;
        PTL(2) = -1.4685532892166679e+00;
        PTL(3) = -7.2355101875283745e-01;
        PTL(4) = 0.0000000000000000e+00;

        PTW(0) = 3.9606977263264792e-05;
        PTW(1) = 4.9436242755369602e-03;
        PTW(2) = 8.8474527394376778e-02;
        PTW(3) = 4.3265155900255553e-01;
        PTW(4) = 7.2023521560605075e-01;

        break;
    }
    case 10: {
        PTL(0) = -3.4361591188377374e+00;
        PTL(1) = -2.5327316742327897e+00;
        PTL(2) = -1.7566836492998819e+00;
        PTL(3) = -1.0366108297895136e+00;
        PTL(4) = -3.4290132722370459e-01;

        PTW(0) = 7.6404328552327460e-06;
        PTW(1) = 1.3436457467812350e-03;
        PTW(2) = 3.3874394455481086e-02;
        PTW(3) = 2.4013861108231460e-01;
        PTW(4) = 6.1086263373532579e-01;

        break;
    }
    case 11: {
        PTL(0) = -3.6684708465595826e+00;
        PTL(1) = -2.7832900997816514e+00;
        PTL(2) = -2.0259480158257555e+00;
        PTL(3) = -1.3265570844949328e+00;
        PTL(4) = -6.5680956688209968e-01;
        PTL(5) = 0.0000000000000000e+00;

        PTW(0) = 1.4395603937142234e-06;
        PTW(1) = 3.4681946632334740e-04;
        PTW(2) = 1.1911395444911568e-02;
        PTW(3) = 1.1722787516770829e-01;
        PTW(4) = 4.2935975235612511e-01;
        PTW(5) = 6.5475928691459184e-01;

        break;
    }
    case 12: {
        PTL(0) = -3.8897248978697818e+00;
        PTL(1) = -3.0206370251208896e+00;
        PTL(2) = -2.2795070805010598e+00;
        PTL(3) = -1.5976826351526046e+00;
        PTL(4) = -9.4778839124016367e-01;
        PTL(5) = -3.1424037625435908e-01;

        PTW(0) = 2.6585516843563065e-07;
        PTW(1) = 8.5736870435878466e-05;
        PTW(2) = 3.9053905846290668e-03;
        PTW(3) = 5.1607985615883818e-02;
        PTW(4) = 2.6049231026416136e-01;
        PTW(5) = 5.7013523626247953e-01;

        break;
    }
    case 13: {
        PTL(0) = -4.1013375961786398e+00;
        PTL(1) = -3.2466089783724099e+00;
        PTL(2) = -2.5197356856782376e+00;
        PTL(3) = -1.8531076516015119e+00;
        PTL(4) = -1.2200550365907483e+00;
        PTL(5) = -6.0576387917106000e-01;
        PTL(6) = 0.0000000000000000e+00;

        PTW(0) = 4.8257318500730054e-08;
        PTW(1) = 2.0430360402707409e-05;
        PTW(2) = 1.2074599927193814e-03;
        PTW(3) = 2.0862775296169922e-02;
        PTW(4) = 1.4032332068702352e-01;
        PTW(5) = 4.2161629689854324e-01;
        PTW(6) = 6.0439318792116148e-01;

        break;
    }
    case 14: {
        PTL(0) = -4.3044485704736317e+00;
        PTL(1) = -3.4626569336022701e+00;
        PTL(2) = -2.7484707249854026e+00;
        PTL(3) = -2.0951832585077166e+00;
        PTL(4) = -1.4766827311411408e+00;
        PTL(5) = -8.7871378732939931e-01;
        PTL(6) = -2.9174551067256205e-01;

        PTW(0) = 8.6285911681251173e-09;
        PTW(1) = 4.7164843550188979e-06;
        PTW(2) = 3.5509261355192427e-04;
        PTW(3) = 7.8500547264578648e-03;
        PTW(4) = 6.8505534223465017e-02;
        PTW(5) = 2.7310560906424680e-01;
        PTW(6) = 5.3640590971209012e-01;

        break;
    }
    case 15: {
        PTL(0) = -4.4999907073093910e+00;
        PTL(1) = -3.6699503734044523e+00;
        PTL(2) = -2.9671669279056028e+00;
        PTL(3) = -2.3257324861738575e+00;
        PTL(4) = -1.7199925751864886e+00;
        PTL(5) = -1.1361155852109206e+00;
        PTL(6) = -5.6506958325557566e-01;
        PTL(7) = 0.0000000000000000e+00;

        PTW(0) = 1.5224758042535364e-09;
        PTW(1) = 1.0591155477110472e-06;
        PTW(2) = 1.0000444123249875e-04;
        PTW(3) = 2.7780688429127759e-03;
        PTW(4) = 3.0780033872546110e-02;
        PTW(5) = 1.5848891579593574e-01;
        PTW(6) = 4.1202868749889848e-01;
        PTW(7) = 5.6410030872641759e-01;

        break;
    }
    case 16: {
        PTL(0) = -4.6887389393058179e+00;
        PTL(1) = -3.8694479048601225e+00;
        PTL(2) = -3.1769991619799560e+00;
        PTL(3) = -2.5462021578474809e+00;
        PTL(4) = -1.9517879909162539e+00;
        PTL(5) = -1.3802585391988806e+00;
        PTL(6) = -8.2295144914465590e-01;
        PTL(7) = -2.7348104613815238e-01;

        PTW(0) = 2.6548074740111456e-10;
        PTW(1) = 2.3209808448651755e-07;
        PTW(2) = 2.7118600925378953e-05;
        PTW(3) = 9.3228400862417843e-04;
        PTW(4) = 1.2880311535509965e-02;
        PTW(5) = 8.3810041398985860e-02;
        PTW(6) = 2.8064745852853362e-01;
        PTW(7) = 5.0792947901661356e-01;

        break;
    }
    case 17: {
        PTL(0) = -4.8713451936744026e+00;
        PTL(1) = -4.0619466758754745e+00;
        PTL(2) = -3.3789320911414942e+00;
        PTL(3) = -2.7577629157038883e+00;
        PTL(4) = -2.1735028266666205e+00;
        PTL(5) = -1.6129243142212313e+00;
        PTL(6) = -1.0676487257434504e+00;
        PTL(7) = -5.3163300134265468e-01;
        PTL(8) = 0.0000000000000000e+00;

        PTW(0) = 4.5805789307984591e-11;
        PTW(1) = 4.9770789816309016e-08;
        PTW(2) = 7.1122891400213076e-06;
        PTW(3) = 2.9864328669775236e-04;
        PTW(4) = 5.0673499576275099e-03;
        PTW(5) = 4.0920034149756195e-02;
        PTW(6) = 1.7264829767009676e-01;
        PTW(7) = 4.0182646947041195e-01;
        PTW(8) = 5.3091793762486372e-01;

        break;
    }
    case 18: {
        PTL(0) = -5.0483640088744668e+00;
        PTL(1) = -4.2481178735681269e+00;
        PTL(2) = -3.5737690684862660e+00;
        PTL(3) = -2.9613775055316065e+00;
        PTL(4) = -2.3862990891666858e+00;
        PTL(5) = -1.8355316042616288e+00;
        PTL(6) = -1.3009208583896172e+00;
        PTL(7) = -7.7668291926741162e-01;
        PTL(8) = -2.5826775051909678e-01;

        PTW(0) = 7.8281997721153926e-12;
        PTW(1) = 1.0467205795791976e-08;
        PTW(2) = 1.8106544810934432e-06;
        PTW(3) = 9.1811268679293717e-05;
        PTW(4) = 1.8885226302684020e-03;
        PTW(5) = 1.8640042387544652e-02;
        PTW(6) = 9.7301747641315384e-02;
        PTW(7) = 2.8480728566997959e-01;
        PTW(8) = 4.8349569472545556e-01;

        break;
    }
    case 19: {
        PTL(0) = -5.2202716905374817e+00;
        PTL(1) = -4.4285328066037790e+00;
        PTL(2) = -3.7621873519640197e+00;
        PTL(3) = -3.1578488183476017e+00;
        PTL(4) = -2.5911337897945423e+00;
        PTL(5) = -2.0492317098506190e+00;
        PTL(6) = -1.5241706193935329e+00;
        PTL(7) = -1.0103683871343112e+00;
        PTL(8) = -5.0352016342388817e-01;
        PTL(9) = 0.0000000000000000e+00;

        PTW(0) = 1.3262970944984907e-12;
        PTW(1) = 2.1630510098636488e-09;
        PTW(2) = 4.4882431472231798e-07;
        PTW(3) = 2.7209197763161993e-05;
        PTW(4) = 6.7087752140718691e-04;
        PTW(5) = 7.9888667777230152e-03;
        PTW(6) = 5.0810386909052138e-02;
        PTW(7) = 1.8363270130699688e-01;
        PTW(8) = 3.9160898861303034e-01;
        PTW(9) = 5.0297488827618630e-01;

        break;
    }
    case 20: {
        PTL(0) = -5.3874808900112328e+00;
        PTL(1) = -4.6036824495507442e+00;
        PTL(2) = -3.9447640401156248e+00;
        PTL(3) = -3.3478545673832163e+00;
        PTL(4) = -2.7888060584281305e+00;
        PTL(5) = -2.2549740020892752e+00;
        PTL(6) = -1.7385377121165861e+00;
        PTL(7) = -1.2340762153953229e+00;
        PTL(8) = -7.3747372854539439e-01;
        PTL(9) = -2.4534070830090121e-01;

        PTW(0) = 2.2293936455341036e-13;
        PTW(1) = 4.3993409922730724e-10;
        PTW(2) = 1.0860693707692408e-07;
        PTW(3) = 7.8025564785320294e-06;
        PTW(4) = 2.2833863601635424e-04;
        PTW(5) = 3.2437733422378246e-03;
        PTW(6) = 2.4810520887463574e-02;
        PTW(7) = 1.0901720602002327e-01;
        PTW(8) = 2.8667550536283415e-01;
        PTW(9) = 4.6224366960061031e-01;

        break;
    }
    default:
        throw std::invalid_argument("not supported");
    }

    for(auto I = 0u, J = order - 1; I < order / 2; ++I, --J) {
        PTL(J) = -PTL(I);
        PTW(J) = PTW(I);
    }

    return {std::move(PTL), std::move(PTW)};
}

template<> std::pair<arma::vec, arma::vec> generate_seeds<IntegrationType::CHEBYSHEV>(const unsigned order) {
    arma::vec PTL(order, arma::fill::none), PTW(order, arma::fill::none);

    switch(order) {
    case 2: {
        PTL(0) = -0.5773502691;

        PTW(0) = 1.;

        break;
    }
    case 3: {
        PTL(0) = -0.7071067811;
        PTL(1) = 0.;

        PTW(0) = 2. / 3.;
        PTW(1) = 2. / 3.;

        break;
    }
    case 4: {
        PTL(0) = -0.7946544722;
        PTL(1) = -0.1875924740;

        PTW(0) = .5;
        PTW(1) = .5;

        break;
    }
    case 5: {
        PTL(0) = -0.8324974870;
        PTL(1) = -0.3745414095;
        PTL(2) = 0.;

        PTW(0) = .4;
        PTW(1) = .4;
        PTW(2) = .4;

        break;
    }
    case 6: {
        PTL(0) = -0.8662468181;
        PTL(1) = -0.4225186537;
        PTL(2) = -0.2666354015;

        PTW(0) = 1. / 3.;
        PTW(1) = 1. / 3.;
        PTW(2) = 1. / 3.;

        break;
    }
    case 7: {
        PTL(0) = -0.8838617007;
        PTL(1) = -0.5296567752;
        PTL(2) = -0.3239118105;
        PTL(3) = 0.;

        PTW(0) = 2. / 7.;
        PTW(1) = 2. / 7.;
        PTW(2) = 2. / 7.;
        PTW(3) = 2. / 7.;

        break;
    }
    default:
        throw std::invalid_argument("not supported");
    }

    for(auto I = 0u, J = order - 1; I < order / 2; ++I, --J) {
        PTL(J) = -PTL(I);
        PTW(J) = PTW(I);
    }

    return {std::move(PTL), std::move(PTW)};
}

template<> std::pair<arma::vec, arma::vec> generate_seeds<IntegrationType::LOBATTO>(const unsigned order) {
    arma::vec PTL(order, arma::fill::none), PTW(order, arma::fill::none);

    switch(order) {
    case 3: {
        PTL(0) = -1.;
        PTL(1) = .0;

        PTW(0) = 1. / 3.;
        PTW(1) = 4. / 3.;

        break;
    }
    case 4: {
        PTL(0) = -1.;
        PTL(1) = -sqrt(.2);

        PTW(0) = 1. / 6.;
        PTW(1) = 5. / 6.;

        break;
    }
    case 5: {
        PTL(0) = -1.;
        PTL(1) = -sqrt(3. / 7.);
        PTL(2) = .0;

        PTW(0) = .1;
        PTW(1) = 49. / 90.;
        PTW(2) = 64. / 90.;

        break;
    }
    case 6: {
        const auto TMPA = 2. * sqrt(7.);
        constexpr auto TMPB = 14. / 30.;
        const auto TMPC = sqrt(7.) / 30.;

        PTL(0) = -1.;
        PTL(1) = -sqrt((7. + TMPA) / 21.);
        PTL(2) = -sqrt((7. - TMPA) / 21.);

        PTW(0) = 1. / 15.;
        PTW(1) = TMPB - TMPC;
        PTW(2) = TMPB + TMPC;

        break;
    }
    case 7: {
        PTL(0) = -1.;
        PTL(1) = -.83022389627856693;
        PTL(2) = -.468848793470714214;
        PTL(3) = .0;

        PTW(0) = .0476190476190476191;
        PTW(1) = .276826047361565948;
        PTW(2) = .431745381209862623;
        PTW(3) = .48761904761904762;

        break;
    }
    case 8: {
        PTL(0) = -1.;
        PTL(1) = -.871740148509606615;
        PTL(2) = -.591700181433142302;
        PTL(3) = -.209299217902478869;

        PTW(0) = .0357142857142857143;
        PTW(1) = .210704227143506039;
        PTW(2) = .34112269248350436;
        PTW(3) = .412458794658703882;

        break;
    }
    case 9: {
        PTL(0) = -1.;
        PTL(1) = -.899757995411460157;
        PTL(2) = -.677186279510737753;
        PTL(3) = -.363117463826178159;
        PTL(4) = .0;

        PTW(0) = .0277777777777777778;
        PTW(1) = .165495361560805525;
        PTW(2) = .274538712500161735;
        PTW(3) = .346428510973046345;
        PTW(4) = .371519274376417234;

        break;
    }
    case 10: {
        PTL(0) = -1.;
        PTL(1) = -.919533908166458814;
        PTL(2) = -.738773865105505075;
        PTL(3) = -.477924949810444496;
        PTL(4) = -.165278957666387025;

        PTW(0) = .0222222222222222222;
        PTW(1) = .13330599085107011;
        PTW(2) = .22488934206312645;
        PTW(3) = .292042683679683758;
        PTW(4) = .327539761183897457;

        break;
    }
    case 11: {
        PTL(0) = -1.;
        PTL(1) = -.934001430408059134;
        PTL(2) = -.78448347366314442;
        PTL(3) = -.565235326996205007;
        PTL(4) = -.295758135586939391;
        PTL(5) = .0;

        PTW(0) = .0181818181818181818;
        PTW(1) = .109612273266994865;
        PTW(2) = .187169881780305204;
        PTW(3) = .248048104264028314;
        PTW(4) = .286879124779008089;
        PTW(5) = .30021759545569069;

        break;
    }
    case 12: {
        PTL(0) = -1.;
        PTL(1) = -.944899272222882223;
        PTL(2) = -.819279321644006678;
        PTL(3) = -.632876153031860678;
        PTL(4) = -.399530940965348932;
        PTL(5) = -.136552932854927555;

        PTW(0) = .0151515151515151515;
        PTW(1) = .09168451741319613;
        PTW(2) = .15797470556437012;
        PTW(3) = .212508417761021145;
        PTW(4) = .25127560319920128;
        PTW(5) = .271405240910696177;

        break;
    }
    case 13: {
        PTL(0) = -1.;
        PTL(1) = -.953309846642163912;
        PTL(2) = -.84634756465187232;
        PTL(3) = -.686188469081757426;
        PTL(4) = -.482909821091336202;
        PTL(5) = -.249286930106239993;
        PTL(6) = .0;

        PTW(0) = .0128205128205128205;
        PTW(1) = .07780168674681893;
        PTW(2) = .134981926689608349;
        PTW(3) = .18364686520355009;
        PTW(4) = .220767793566110086;
        PTW(5) = .24401579030667636;
        PTW(6) = .251930849333446736;

        break;
    }
    case 14: {
        PTL(0) = -1.;
        PTL(1) = -.959935045267260901;
        PTL(2) = -.867801053830347251;
        PTL(3) = -.728868599091326141;
        PTL(4) = -.550639402928647055;
        PTL(5) = -.342724013342712845;
        PTL(6) = -.116331868883703868;

        PTW(0) = .010989010989010989;
        PTW(1) = .06683728449768128;
        PTW(2) = .116586655898711652;
        PTW(3) = .160021851762952142;
        PTW(4) = .19482614937341612;
        PTW(5) = .219126253009770755;
        PTW(6) = .231612794468457059;

        break;
    }
    case 15: {
        PTL(0) = -1.;
        PTL(1) = -.965245926503838573;
        PTL(2) = -.885082044222976299;
        PTL(3) = -.763519689951815201;
        PTL(4) = -.606253205469845711;
        PTL(5) = -.420638054713672481;
        PTL(6) = -.215353955363794238;
        PTL(7) = .0;

        PTW(0) = .00952380952380952381;
        PTW(1) = .05802989302860125;
        PTW(2) = .101660070325718068;
        PTW(3) = .14051169980242811;
        PTW(4) = .172789647253600949;
        PTW(5) = .19698723596461336;
        PTW(6) = .21197358592682092;
        PTW(7) = .21704811634881565;

        break;
    }
    case 16: {
        PTL(0) = -1.;
        PTL(1) = -.969568046270217933;
        PTL(2) = -.899200533093472093;
        PTL(3) = -.792008291861815064;
        PTL(4) = -.65238870288249309;
        PTL(5) = -.486059421887137612;
        PTL(6) = -.299830468900763208;
        PTL(7) = -.101326273521949448;

        PTW(0) = .00833333333333333333;
        PTW(1) = .05085036100591991;
        PTW(2) = .089393697325930801;
        PTW(3) = .124255382132514098;
        PTW(4) = .154026980807164281;
        PTW(5) = .177491913391704125;
        PTW(6) = .19369002382520358;
        PTW(7) = .201958308178229872;

        break;
    }
    case 17: {
        PTL(0) = -1.;
        PTL(1) = -.973132176631418314;
        PTL(2) = -.910879995915573596;
        PTL(3) = -.815696251221770307;
        PTL(4) = -.691028980627684705;
        PTL(5) = -.541385399330101539;
        PTL(6) = -.372174433565477042;
        PTL(7) = -.189511973518317388;
        PTL(8) = .0;

        PTW(0) = .00735294117647058824;
        PTW(1) = .04492194054325421;
        PTW(2) = .079198270503687119;
        PTW(3) = .110592909007028161;
        PTW(4) = .137987746201926559;
        PTW(5) = .16039466199762154;
        PTW(6) = .17700425351565787;
        PTW(7) = .18721633967761924;
        PTW(8) = .190661874753469433;

        break;
    }
    case 18: {
        PTL(0) = -1.;
        PTL(1) = -.976105557412198543;
        PTL(2) = -.920649185347533874;
        PTL(3) = -.835593535218090214;
        PTL(4) = -.723679329283242681;
        PTL(5) = -.588504834318661761;
        PTL(6) = -.434415036912123975;
        PTL(7) = -.26636265287828098;
        PTL(8) = -.089749093484652111;

        PTW(0) = .00653594771241830065;
        PTW(1) = .039970628810914066;
        PTW(2) = .070637166885633665;
        PTW(3) = .0990162717175028;
        PTW(4) = .1242105331329671;
        PTW(5) = .145411961573802268;
        PTW(6) = .161939517237602489;
        PTW(7) = .173262109489456226;
        PTW(8) = .17901586343970308;

        break;
    }
    case 19: {
        PTL(0) = -1.;
        PTL(1) = -.978611766222080095;
        PTL(2) = -.928901528152586244;
        PTL(3) = -.852460577796646093;
        PTL(4) = -.751494202552613014;
        PTL(5) = -.628908137265220498;
        PTL(6) = -.488229285680713503;
        PTL(7) = -.33350484782449861;
        PTL(8) = -.169186023409281571;
        PTL(9) = .0;

        PTW(0) = .00584795321637426901;
        PTW(1) = .035793365186176477;
        PTW(2) = .063381891762629737;
        PTW(3) = .0891317570992070845;
        PTW(4) = .11231534147730504;
        PTW(5) = .132267280448750777;
        PTW(6) = .14841394259593889;
        PTW(7) = .160290924044061242;
        PTW(8) = .167556584527142867;
        PTW(9) = .170001919284827235;

        break;
    }
    case 20: {
        PTL(0) = -1.;
        PTL(1) = -.980743704893914172;
        PTL(2) = -.935934498812665436;
        PTL(3) = -.866877978089950141;
        PTL(4) = -.77536826095205587;
        PTL(5) = -.66377640229031129;
        PTL(6) = -.53499286403188626;
        PTL(7) = -.392353183713909299;
        PTL(8) = -.239551705922986495;
        PTL(9) = -.080545937238821838;

        PTW(0) = .00526315789473684211;
        PTW(1) = .032237123188488941;
        PTW(2) = .057181802127566826;
        PTW(3) = .080631763996119603;
        PTW(4) = .10199149969945082;
        PTW(5) = .12070922762867473;
        PTW(6) = .136300482358724185;
        PTW(7) = .148361554070916826;
        PTW(8) = .156580102647475487;
        PTW(9) = .16074328638784575;

        break;
    }
    default:
        throw std::invalid_argument("not supported");
    }

    for(auto I = 0u, J = order - 1; I < order / 2; ++I, --J) {
        PTL(J) = -PTL(I);
        PTW(J) = PTW(I);
    }

    return {std::move(PTL), std::move(PTW)};
}

template<> std::pair<arma::vec, arma::vec> generate_seeds<IntegrationType::RADAU>(const unsigned order) {
    arma::vec PTL(order, arma::fill::none), PTW(order, arma::fill::none);

    switch(order) {
    case 2: {
        PTL(0) = -1.;
        PTL(1) = 1. / 3.;

        PTW(0) = .5;
        PTW(1) = 1.5;

        break;
    }
    case 3: {
        PTL(0) = -1.;
        PTL(1) = -.2898979485;
        PTL(2) = .6898979485;

        PTW(0) = 2. / 9.;
        PTW(1) = 1.0249716523;
        PTW(2) = .7528061254;

        break;
    }
    case 4: {
        PTL(0) = -1.;
        PTL(1) = -.5753189235;
        PTL(2) = .1810662711;
        PTL(3) = .8228240809;

        PTW(0) = .125;
        PTW(1) = .6576886399;
        PTW(2) = .7763869376;
        PTW(3) = .4409244223;

        break;
    }
    case 5: {
        PTL(0) = -1.;
        PTL(1) = -.7204802713;
        PTL(2) = -.1671808647;
        PTL(3) = .44631398727;
        PTL(4) = .8857916077;

        PTW(0) = .08;
        PTW(1) = .4462078021;
        PTW(2) = .6236530459;
        PTW(3) = .5627120302;
        PTW(4) = .2874271215;

        break;
    }
    case 6: {
        PTL(0) = -1.;
        PTL(1) = -.8029298284;
        PTL(2) = -.3909285467;
        PTL(3) = .1240503795;
        PTL(4) = .6039731642;
        PTL(5) = .9203802858;

        PTW(0) = 1. / 18.;
        PTW(1) = .3196407532;
        PTW(2) = .4853871884;
        PTW(3) = .5209267831;
        PTW(4) = .4169013343;
        PTW(5) = .2015883852;

        break;
    }
    case 7: {
        PTL(0) = -1.;
        PTL(1) = -.8538913426;
        PTL(2) = -.5384677240;
        PTL(3) = -.1173430375;
        PTL(4) = .3260306194;
        PTL(5) = .7038428006;
        PTL(6) = .9413671456;

        PTW(0) = 2. / 49.;
        PTW(1) = .2392274892;
        PTW(2) = .3809498736;
        PTW(3) = .4471098290;
        PTW(4) = .4247037790;
        PTW(5) = .3182042314;
        PTW(6) = .1489884711;

        break;
    }
    case 8: {
        PTL(0) = -1.;
        PTL(1) = -.8874748789;
        PTL(2) = -.6395186165;
        PTL(3) = -.2947505657;
        PTL(4) = .0943072526;
        PTL(5) = .4684203544;
        PTL(6) = .7706418936;
        PTL(7) = .9550412271;

        PTW(0) = 1. / 32.;
        PTW(1) = .1853581548;
        PTW(2) = .3041306206;
        PTW(3) = .3765175453;
        PTW(4) = .3915721674;
        PTW(5) = .3470147956;
        PTW(6) = .2496479013;
        PTW(7) = .1145088147;

        break;
    }
    case 9: {
        PTL(0) = -1.;
        PTL(1) = -.9107320894;
        PTL(2) = -.7112674859;
        PTL(3) = -.4263504857;
        PTL(4) = -.0903733696;
        PTL(5) = .2561356708;
        PTL(6) = .5713830412;
        PTL(7) = .8173527842;
        PTL(8) = .9644401697;

        PTW(0) = 2. / 81.;
        PTW(1) = .1476540190;
        PTW(2) = .2471893782;
        PTW(3) = .3168437756;
        PTW(4) = .3482730027;
        PTW(5) = .3376939669;
        PTW(6) = .2863866963;
        PTW(7) = .2005532980;
        PTW(8) = .0907145049;

        break;
    }
    case 10: {
        PTL(0) = -1.;
        PTL(1) = -.9274843742;
        PTL(2) = -.7638420424;
        PTL(3) = -.5256460303;
        PTL(4) = -.2362344693;
        PTL(5) = .0760591978;
        PTL(6) = .3806648401;
        PTL(7) = .6477666876;
        PTL(8) = .8512252205;
        PTL(9) = .9711751807;

        PTW(0) = .02;
        PTW(1) = .1202966705;
        PTW(2) = .2042701318;
        PTW(3) = .2681948378;
        PTW(4) = .3058592877;
        PTW(5) = .3135824572;
        PTW(6) = .2906101648;
        PTW(7) = .2391934317;
        PTW(8) = .1643760127;
        PTW(9) = .0736170054;

        break;
    }
    default:
        throw std::invalid_argument("not supported");
    }

    return {std::move(PTL), std::move(PTW)};
}

template<> std::pair<arma::vec, arma::vec> generate_seeds<IntegrationType::LAGUERRE>(const unsigned order) {
    arma::vec PTL(order, arma::fill::none), PTW(order, arma::fill::none);

    switch(order) {
    case 2: {
        PTL(0) = 5.8578643762690497e-01;
        PTL(1) = 3.4142135623730954e+00;

        PTW(0) = 8.5355339059327373e-01;
        PTW(1) = 1.4644660940672624e-01;

        break;
    }
    case 3: {
        PTL(0) = 4.1577455678347913e-01;
        PTL(1) = 2.2942803602790418e+00;
        PTL(2) = 6.2899450829374777e+00;

        PTW(0) = 7.1109300992917313e-01;
        PTW(1) = 2.7851773356924076e-01;
        PTW(2) = 1.0389256501586133e-02;

        break;
    }
    case 4: {
        PTL(0) = 3.2254768961939229e-01;
        PTL(1) = 1.7457611011583467e+00;
        PTL(2) = 4.5366202969211278e+00;
        PTL(3) = 9.3950709123011329e+00;

        PTW(0) = 6.0315410434163330e-01;
        PTW(1) = 3.5741869243779995e-01;
        PTW(2) = 3.8887908515005412e-02;
        PTW(3) = 5.3929470556132947e-04;

        break;
    }
    case 5: {
        PTL(0) = 2.6356031971814092e-01;
        PTL(1) = 1.4134030591065168e+00;
        PTL(2) = 3.5964257710407219e+00;
        PTL(3) = 7.0858100058588374e+00;
        PTL(4) = 1.2640800844275784e+01;

        PTW(0) = 5.2175561058280873e-01;
        PTW(1) = 3.9866681108317570e-01;
        PTW(2) = 7.5942449681707616e-02;
        PTW(3) = 3.6117586799220545e-03;
        PTW(4) = 2.3369972385776238e-05;

        break;
    }
    case 6: {
        PTL(0) = 2.2284660417926072e-01;
        PTL(1) = 1.1889321016726231e+00;
        PTL(2) = 2.9927363260593141e+00;
        PTL(3) = 5.7751435691045101e+00;
        PTL(4) = 9.8374674183825910e+00;
        PTL(5) = 1.5982873980601701e+01;

        PTW(0) = 4.5896467394996326e-01;
        PTW(1) = 4.1700083077212113e-01;
        PTW(2) = 1.1337338207404495e-01;
        PTW(3) = 1.0399197453149094e-02;
        PTW(4) = 2.6101720281493249e-04;
        PTW(5) = 8.9854790642962145e-07;

        break;
    }
    case 7: {
        PTL(0) = 1.9304367656036242e-01;
        PTL(1) = 1.0266648953391919e+00;
        PTL(2) = 2.5678767449507465e+00;
        PTL(3) = 4.9003530845264844e+00;
        PTL(4) = 8.1821534445628608e+00;
        PTL(5) = 1.2734180291797815e+01;
        PTL(6) = 1.9395727862262540e+01;

        PTW(0) = 4.0931895170127391e-01;
        PTW(1) = 4.2183127786171987e-01;
        PTW(2) = 1.4712634865750529e-01;
        PTW(3) = 2.0633514468716932e-02;
        PTW(4) = 1.0740101432807467e-03;
        PTW(5) = 1.5865464348564213e-05;
        PTW(6) = 3.1703154789955671e-08;

        break;
    }
    case 8: {
        PTL(0) = 1.7027963230510101e-01;
        PTL(1) = 9.0370177679937991e-01;
        PTL(2) = 2.2510866298661307e+00;
        PTL(3) = 4.2667001702876588e+00;
        PTL(4) = 7.0459054023934655e+00;
        PTL(5) = 1.0758516010180996e+01;
        PTL(6) = 1.5740678641278006e+01;
        PTL(7) = 2.2863131736889262e+01;

        PTW(0) = 3.6918858934163784e-01;
        PTW(1) = 4.1878678081434267e-01;
        PTW(2) = 1.7579498663717177e-01;
        PTW(3) = 3.3343492261215656e-02;
        PTW(4) = 2.7945362352256747e-03;
        PTW(5) = 9.0765087733581104e-05;
        PTW(6) = 8.4857467162725409e-07;
        PTW(7) = 1.0480011748715118e-09;

        break;
    }
    case 9: {
        PTL(0) = 1.5232222773180826e-01;
        PTL(1) = 8.0722002274225579e-01;
        PTL(2) = 2.0051351556193473e+00;
        PTL(3) = 3.7834739733312333e+00;
        PTL(4) = 6.2049567778766122e+00;
        PTL(5) = 9.3729852516875773e+00;
        PTL(6) = 1.3466236911092093e+01;
        PTL(7) = 1.8833597788991696e+01;
        PTL(8) = 2.6374071890927375e+01;

        PTW(0) = 3.3612642179796293e-01;
        PTW(1) = 4.1121398042398399e-01;
        PTW(2) = 1.9928752537088557e-01;
        PTW(3) = 4.7460562765651616e-02;
        PTW(4) = 5.5996266107945711e-03;
        PTW(5) = 3.0524976709321062e-04;
        PTW(6) = 6.5921230260753642e-06;
        PTW(7) = 4.1107693303495907e-08;
        PTW(8) = 3.2908740303506877e-11;

        break;
    }
    case 10: {
        PTL(0) = 1.3779347054049243e-01;
        PTL(1) = 7.2945454950317046e-01;
        PTL(2) = 1.8083429017403161e+00;
        PTL(3) = 3.4014336978548996e+00;
        PTL(4) = 5.5524961400638038e+00;
        PTL(5) = 8.3301527467644974e+00;
        PTL(6) = 1.1843785837900066e+01;
        PTL(7) = 1.6279257831378104e+01;
        PTL(8) = 2.1996585811980761e+01;
        PTL(9) = 2.9920697012273891e+01;

        PTW(0) = 3.0844111576502004e-01;
        PTW(1) = 4.0111992915527350e-01;
        PTW(2) = 2.1806828761180955e-01;
        PTW(3) = 6.2087456098677767e-02;
        PTW(4) = 9.5015169751810971e-03;
        PTW(5) = 7.5300838858753834e-04;
        PTW(6) = 2.8259233495995628e-05;
        PTW(7) = 4.2493139849626937e-07;
        PTW(8) = 1.8395648239796337e-09;
        PTW(9) = 9.9118272196090327e-13;

        break;
    }
    case 11: {
        PTL(0) = 1.2579644218796754e-01;
        PTL(1) = 6.6541825583922787e-01;
        PTL(2) = 1.6471505458721694e+00;
        PTL(3) = 3.0911381430352551e+00;
        PTL(4) = 5.0292844015798339e+00;
        PTL(5) = 7.5098878638066164e+00;
        PTL(6) = 1.0605950999546968e+01;
        PTL(7) = 1.4431613758064186e+01;
        PTL(8) = 1.9178857403214678e+01;
        PTL(9) = 2.5217709339677562e+01;
        PTL(10) = 3.3497192847175533e+01;

        PTW(0) = 2.8493321289420076e-01;
        PTW(1) = 3.8972088952784917e-01;
        PTW(2) = 2.3278183184899134e-01;
        PTW(3) = 7.6564453546196787e-02;
        PTW(4) = 1.4393282767350699e-02;
        PTW(5) = 1.5188808464848696e-03;
        PTW(6) = 8.5131224354718894e-05;
        PTW(7) = 2.2924038795744968e-06;
        PTW(8) = 2.4863537027677723e-08;
        PTW(9) = 7.7126269336913902e-11;
        PTW(10) = 2.8837758683235775e-14;

        break;
    }
    case 12: {
        PTL(0) = 1.1572211735802068e-01;
        PTL(1) = 6.1175748451513068e-01;
        PTL(2) = 1.5126102697764188e+00;
        PTL(3) = 2.8337513377435073e+00;
        PTL(4) = 4.5992276394183484e+00;
        PTL(5) = 6.8445254531151773e+00;
        PTL(6) = 9.6213168424568671e+00;
        PTL(7) = 1.3006054993306346e+01;
        PTL(8) = 1.7116855187462253e+01;
        PTL(9) = 2.2151090379397004e+01;
        PTL(10) = 2.8487967250983999e+01;
        PTL(11) = 3.7099121044466919e+01;

        PTW(0) = 2.6473137105544420e-01;
        PTW(1) = 3.7775927587313751e-01;
        PTW(2) = 2.4408201131987708e-01;
        PTW(3) = 9.0449222211680808e-02;
        PTW(4) = 2.0102381154634023e-02;
        PTW(5) = 2.6639735418653122e-03;
        PTW(6) = 2.0323159266299928e-04;
        PTW(7) = 8.3650558568198412e-06;
        PTW(8) = 1.6684938765409072e-07;
        PTW(9) = 1.3423910305149902e-09;
        PTW(10) = 3.0616016350350716e-12;
        PTW(11) = 8.1480774674260206e-16;

        break;
    }
    case 13: {
        PTL(0) = 1.0714238847225230e-01;
        PTL(1) = 5.6613189904040184e-01;
        PTL(2) = 1.3985643364510196e+00;
        PTL(3) = 2.6165971084064115e+00;
        PTL(4) = 4.2388459290170335e+00;
        PTL(5) = 6.2922562711400740e+00;
        PTL(6) = 8.8150019411869778e+00;
        PTL(7) = 1.1861403588811243e+01;
        PTL(8) = 1.5510762037703753e+01;
        PTL(9) = 1.9884635663880228e+01;
        PTL(10) = 2.5185263864677758e+01;
        PTL(11) = 3.1800386301947267e+01;
        PTL(12) = 4.0723008669265582e+01;

        PTW(0) = 2.4718870842996213e-01;
        PTW(1) = 3.6568882290052185e-01;
        PTW(2) = 2.5256242005765855e-01;
        PTW(3) = 1.0347075802418398e-01;
        PTW(4) = 2.6432754415561584e-02;
        PTW(5) = 4.2203960402547574e-03;
        PTW(6) = 4.1188177047273407e-04;
        PTW(7) = 2.3515473981553229e-05;
        PTW(8) = 7.3173116202491101e-07;
        PTW(9) = 1.1088416257039839e-08;
        PTW(10) = 6.7708266922058900e-11;
        PTW(11) = 1.1599799599050753e-13;
        PTW(12) = 2.2450932038927184e-17;

        break;
    }
    case 14: {
        PTL(0) = 9.9747507032597590e-02;
        PTL(1) = 5.2685764885190289e-01;
        PTL(2) = 1.3006291212514964e+00;
        PTL(3) = 2.4308010787308447e+00;
        PTL(4) = 3.9321028222932188e+00;
        PTL(5) = 5.8255362183017088e+00;
        PTL(6) = 8.1402401415651457e+00;
        PTL(7) = 1.0916499507366019e+01;
        PTL(8) = 1.4210805011161288e+01;
        PTL(9) = 1.8104892220218098e+01;
        PTL(10) = 2.2723381628269625e+01;
        PTL(11) = 2.8272981723248204e+01;
        PTL(12) = 3.5149443660592425e+01;
        PTL(13) = 4.4366081711117424e+01;

        PTW(0) = 2.3181557714486570e-01;
        PTW(1) = 3.5378469159754317e-01;
        PTW(2) = 2.5873461024542782e-01;
        PTW(3) = 1.1548289355692307e-01;
        PTW(4) = 3.3192092159337314e-02;
        PTW(5) = 6.1928694370066151e-03;
        PTW(6) = 7.3989037786738440e-04;
        PTW(7) = 5.4907194668416865e-05;
        PTW(8) = 2.4095857640853837e-06;
        PTW(9) = 5.8015439816765169e-08;
        PTW(10) = 6.8193146924849305e-10;
        PTW(11) = 3.2212077518948225e-12;
        PTW(12) = 4.2213524405166307e-15;
        PTW(13) = 6.0523750222891402e-19;

        break;
    }
    case 15: {
        PTL(0) = 9.3307812017281777e-02;
        PTL(1) = 4.9269174030188390e-01;
        PTL(2) = 1.2155954120709496e+00;
        PTL(3) = 2.2699495262037432e+00;
        PTL(4) = 3.6676227217514370e+00;
        PTL(5) = 5.4253366274135528e+00;
        PTL(6) = 7.5659162266130675e+00;
        PTL(7) = 1.0120228568019114e+01;
        PTL(8) = 1.3130282482175724e+01;
        PTL(9) = 1.6654407708329956e+01;
        PTL(10) = 2.0776478899448765e+01;
        PTL(11) = 2.5623894226728780e+01;
        PTL(12) = 3.1407519169753940e+01;
        PTL(13) = 3.8530683306486011e+01;
        PTL(14) = 4.8026085572685794e+01;

        PTW(0) = 2.1823488594008655e-01;
        PTW(1) = 3.4221017792288311e-01;
        PTW(2) = 2.6302757794168063e-01;
        PTW(3) = 1.2642581810593062e-01;
        PTW(4) = 4.0206864921000847e-02;
        PTW(5) = 8.5638778036118413e-03;
        PTW(6) = 1.2124361472142539e-03;
        PTW(7) = 1.1167439234425229e-04;
        PTW(8) = 6.4599267620228911e-06;
        PTW(9) = 2.2263169070962643e-07;
        PTW(10) = 4.2274303849793574e-09;
        PTW(11) = 3.9218972670411083e-11;
        PTW(12) = 1.4565152640731047e-13;
        PTW(13) = 1.4830270511133285e-16;
        PTW(14) = 1.6005949062111185e-20;

        break;
    }
    case 16: {
        PTL(0) = 8.7649410478927839e-02;
        PTL(1) = 4.6269632891508083e-01;
        PTL(2) = 1.1410577748312269e+00;
        PTL(3) = 2.1292836450983810e+00;
        PTL(4) = 3.4370866338932067e+00;
        PTL(5) = 5.0780186145497677e+00;
        PTL(6) = 7.0703385350482346e+00;
        PTL(7) = 9.4383143363919384e+00;
        PTL(8) = 1.2214223368866159e+01;
        PTL(9) = 1.5441527368781617e+01;
        PTL(10) = 1.9180156856753136e+01;
        PTL(11) = 2.3515905693991908e+01;
        PTL(12) = 2.8578729742882139e+01;
        PTL(13) = 3.4583398702286622e+01;
        PTL(14) = 4.1940452647688332e+01;
        PTL(15) = 5.1701160339543321e+01;

        PTW(0) = 2.0615171495780069e-01;
        PTW(1) = 3.3105785495088402e-01;
        PTW(2) = 2.6579577764421436e-01;
        PTW(3) = 1.3629693429637774e-01;
        PTW(4) = 4.7328928694125222e-02;
        PTW(5) = 1.1299900080339450e-02;
        PTW(6) = 1.8490709435263094e-03;
        PTW(7) = 2.0427191530827824e-04;
        PTW(8) = 1.4844586873981333e-05;
        PTW(9) = 6.8283193308712460e-07;
        PTW(10) = 1.8810248410796997e-08;
        PTW(11) = 2.8623502429738586e-10;
        PTW(12) = 2.1270790332240987e-12;
        PTW(13) = 6.2979670025178009e-15;
        PTW(14) = 5.0504737000355466e-18;
        PTW(15) = 4.1614623703728040e-22;

        break;
    }
    case 17: {
        PTL(0) = 8.2638214708947680e-02;
        PTL(1) = 4.3615032355871042e-01;
        PTL(2) = 1.0751765775114286e+00;
        PTL(3) = 2.0051935316492320e+00;
        PTL(4) = 3.2342561240474437e+00;
        PTL(5) = 4.7735135137001974e+00;
        PTL(6) = 6.6378292053649526e+00;
        PTL(7) = 8.8466855111697988e+00;
        PTL(8) = 1.1425529319373352e+01;
        PTL(9) = 1.4407823037481318e+01;
        PTL(10) = 1.7838284730701140e+01;
        PTL(11) = 2.1778268257722264e+01;
        PTL(12) = 2.6315317811248800e+01;
        PTL(13) = 3.1581771680456733e+01;
        PTL(14) = 3.7796093837477102e+01;
        PTL(15) = 4.5375716533988971e+01;
        PTL(16) = 5.5389751789839607e+01;

        PTW(0) = 1.9533220525177239e-01;
        PTW(1) = 3.2037535727453942e-01;
        PTW(2) = 2.6732972635717112e-01;
        PTW(3) = 1.4512985435875814e-01;
        PTW(4) = 5.4436943245338355e-02;
        PTW(5) = 1.4357297766061842e-02;
        PTW(6) = 2.6628247355727653e-03;
        PTW(7) = 3.4367972715630023e-04;
        PTW(8) = 3.0275517837828533e-05;
        PTW(9) = 1.7685150532316702e-06;
        PTW(10) = 6.5762728868103644e-08;
        PTW(11) = 1.4697309321595515e-09;
        PTW(12) = 1.8169103625554485e-11;
        PTW(13) = 1.0954013889286682e-13;
        PTW(14) = 2.6173738822233697e-16;
        PTW(15) = 1.6729356931461285e-19;
        PTW(16) = 1.0656263162740832e-23;

        break;
    }
    case 18: {
        PTL(0) = 7.8169166669705470e-02;
        PTL(1) = 4.1249008525912928e-01;
        PTL(2) = 1.0165201796235397e+00;
        PTL(3) = 1.8948885099697610e+00;
        PTL(4) = 3.0543531132026596e+00;
        PTL(5) = 4.5042055388898925e+00;
        PTL(6) = 6.2567250739491120e+00;
        PTL(7) = 8.3278251566056305e+00;
        PTL(8) = 1.0737990047757609e+01;
        PTL(9) = 1.3513656207555091e+01;
        PTL(10) = 1.6689306281930108e+01;
        PTL(11) = 2.0310767626267744e+01;
        PTL(12) = 2.4440681359283701e+01;
        PTL(13) = 2.9168208662579616e+01;
        PTL(14) = 3.4627927065660174e+01;
        PTL(15) = 4.1041816772808758e+01;
        PTL(16) = 4.8833922716086526e+01;
        PTL(17) = 5.9090546435901253e+01;

        PTW(0) = 1.8558860314691827e-01;
        PTW(1) = 3.1018176637022532e-01;
        PTW(2) = 2.6786656714853668e-01;
        PTW(3) = 1.5297974746807508e-01;
        PTW(4) = 6.1434917860961689e-02;
        PTW(5) = 1.7687213080772944e-02;
        PTW(6) = 3.6601797677599259e-03;
        PTW(7) = 5.4062278700773561e-04;
        PTW(8) = 5.6169650512142644e-05;
        PTW(9) = 4.0153078837011978e-06;
        PTW(10) = 1.9146698566756742e-07;
        PTW(11) = 5.8360952686315988e-09;
        PTW(12) = 1.0717112669553936e-10;
        PTW(13) = 1.0890987138888249e-12;
        PTW(14) = 5.3866647483784063e-15;
        PTW(15) = 1.0498659780357064e-17;
        PTW(16) = 5.4053984516311071e-21;
        PTW(17) = 2.6916532692010392e-25;

        break;
    }
    case 19: {
        PTL(0) = 7.4158783757205082e-02;
        PTL(1) = 3.9126861331999463e-01;
        PTL(2) = 9.6395734399795796e-01;
        PTL(3) = 1.7961755820683281e+00;
        PTL(4) = 2.8936513818737839e+00;
        PTL(5) = 4.2642155396277666e+00;
        PTL(6) = 5.9181415616440480e+00;
        PTL(7) = 7.8686189153347339e+00;
        PTL(8) = 1.0132423716815266e+01;
        PTL(9) = 1.2730881463842397e+01;
        PTL(10) = 1.5691278339835888e+01;
        PTL(11) = 1.9048993209823550e+01;
        PTL(12) = 2.2850849760829483e+01;
        PTL(13) = 2.7160669327411448e+01;
        PTL(14) = 3.2069122251862240e+01;
        PTL(15) = 3.7712905801219648e+01;
        PTL(16) = 4.4317362795831500e+01;
        PTL(17) = 5.2312902457404384e+01;
        PTL(18) = 6.2802423153500378e+01;

        PTW(0) = 1.7676847491591330e-01;
        PTW(1) = 3.0047814360725467e-01;
        PTW(2) = 2.6759954703817462e-01;
        PTW(3) = 1.5991337213557974e-01;
        PTW(4) = 6.8249379976149022e-02;
        PTW(5) = 2.1239307606544335e-02;
        PTW(6) = 4.8416273511483867e-03;
        PTW(7) = 8.0491274738136737e-04;
        PTW(8) = 9.6524720931534766e-05;
        PTW(9) = 8.2073052580509969e-06;
        PTW(10) = 4.8305667247308033e-07;
        PTW(11) = 1.9049913611232707e-08;
        PTW(12) = 4.8166846309280258e-10;
        PTW(13) = 7.3482588395511407e-12;
        PTW(14) = 6.2022753875725485e-14;
        PTW(15) = 2.5414308430154390e-16;
        PTW(16) = 4.0788612968257481e-19;
        PTW(17) = 1.7077501875939025e-22;
        PTW(18) = 6.7150646499079086e-27;

        break;
    }
    case 20: {
        PTL(0) = 7.0539889691988739e-02;
        PTL(1) = 3.7212681800161146e-01;
        PTL(2) = 9.1658210248327354e-01;
        PTL(3) = 1.7073065310283440e+00;
        PTL(4) = 2.7491992553094322e+00;
        PTL(5) = 4.0489253138508872e+00;
        PTL(6) = 5.6151749708616165e+00;
        PTL(7) = 7.4590174536710627e+00;
        PTL(8) = 9.5943928695810978e+00;
        PTL(9) = 1.2038802546964316e+01;
        PTL(10) = 1.4814293442630740e+01;
        PTL(11) = 1.7948895520519375e+01;
        PTL(12) = 2.1478788240285013e+01;
        PTL(13) = 2.5451702793186904e+01;
        PTL(14) = 2.9932554631700611e+01;
        PTL(15) = 3.5013434240479000e+01;
        PTL(16) = 4.0833057056728570e+01;
        PTL(17) = 4.7619994047346502e+01;
        PTL(18) = 5.5810795750063896e+01;
        PTL(19) = 6.6524416525615749e+01;

        PTW(0) = 1.6874680185111224e-01;
        PTW(1) = 2.9125436200606919e-01;
        PTW(2) = 2.6668610286700123e-01;
        PTW(3) = 1.6600245326950741e-01;
        PTW(4) = 7.4826064668792505e-02;
        PTW(5) = 2.4964417309283272e-02;
        PTW(6) = 6.2025508445722761e-03;
        PTW(7) = 1.1449623864769102e-03;
        PTW(8) = 1.5574177302781267e-04;
        PTW(9) = 1.5401440865224990e-05;
        PTW(10) = 1.0864863665179856e-06;
        PTW(11) = 5.3301209095567636e-08;
        PTW(12) = 1.7579811790506002e-09;
        PTW(13) = 3.7255024025122663e-11;
        PTW(14) = 4.7675292515781856e-13;
        PTW(15) = 3.3728442433625577e-15;
        PTW(16) = 1.1550143395004038e-17;
        PTW(17) = 1.5395221405823465e-20;
        PTW(18) = 5.2864427255690809e-24;
        PTW(19) = 1.6564566124991287e-28;

        break;
    }
    default:
        throw std::invalid_argument("not supported");
    }

    return {std::move(PTL), std::move(PTW)};
}

IntegrationPlan::IntegrationPlan(const unsigned D, const unsigned O, const IntegrationType T) {
    switch(T) {
    case IntegrationType::GAUSS:
        generate<IntegrationType::GAUSS>(D, O);
        break;
    case IntegrationType::HERMITE:
        generate<IntegrationType::HERMITE>(D, O);
        break;
    case IntegrationType::CHEBYSHEV:
        generate<IntegrationType::CHEBYSHEV>(D, O);
        break;
    case IntegrationType::LOBATTO:
        generate<IntegrationType::LOBATTO>(D, O);
        break;
    case IntegrationType::RADAU:
        generate<IntegrationType::RADAU>(D, O);
        break;
    case IntegrationType::LAGUERRE:
        generate<IntegrationType::LAGUERRE>(D, O);
        break;
    case IntegrationType::IRONS:
        generate<IntegrationType::IRONS>(D, O);
        break;
    case IntegrationType::TRIANGLE:
        generate<IntegrationType::TRIANGLE>(D, O);
        break;
    }
}

const arma::mat& IntegrationPlan::get_data() const { return int_pts; }

double IntegrationPlan::operator()(const unsigned i, const unsigned j) const { return int_pts(i, j); }

void IntegrationPlan::print() const {
    for(unsigned i = 0; i < n_rows; ++i) {
        printf("Node %u\t", i + 1);
        for(unsigned j = 0; j < n_cols - 1; ++j) printf("%+.6E\t", int_pts(i, j));
        printf("Weight\t%+.6E\n", int_pts(i, n_cols - 1));
    }
    printf("\n");
}

template<> void IntegrationPlan::generate<IntegrationType::GAUSS>(const unsigned D, const unsigned O) {
    int_pts = generate_points(D, generate_seeds<IntegrationType::GAUSS>(std::min(std::max(O, 1u), 20u)));
    arma::access::rw(n_rows) = static_cast<unsigned>(int_pts.n_rows);
    arma::access::rw(n_cols) = static_cast<unsigned>(int_pts.n_cols);
}

template<> void IntegrationPlan::generate<IntegrationType::HERMITE>(const unsigned D, const unsigned O) {
    int_pts = generate_points(D, generate_seeds<IntegrationType::HERMITE>(std::min(std::max(O, 2u), 20u)));
    arma::access::rw(n_rows) = static_cast<unsigned>(int_pts.n_rows);
    arma::access::rw(n_cols) = static_cast<unsigned>(int_pts.n_cols);
}

template<> void IntegrationPlan::generate<IntegrationType::CHEBYSHEV>(const unsigned D, const unsigned O) {
    int_pts = generate_points(D, generate_seeds<IntegrationType::CHEBYSHEV>(std::min(std::max(O, 2u), 20u)));
    arma::access::rw(n_rows) = static_cast<unsigned>(int_pts.n_rows);
    arma::access::rw(n_cols) = static_cast<unsigned>(int_pts.n_cols);
}

template<> void IntegrationPlan::generate<IntegrationType::LOBATTO>(const unsigned D, const unsigned O) {
    int_pts = generate_points(D, generate_seeds<IntegrationType::LOBATTO>(std::min(std::max(O, 3u), 20u)));
    arma::access::rw(n_rows) = static_cast<unsigned>(int_pts.n_rows);
    arma::access::rw(n_cols) = static_cast<unsigned>(int_pts.n_cols);
}

template<> void IntegrationPlan::generate<IntegrationType::RADAU>(const unsigned D, const unsigned O) {
    int_pts = generate_points(D, generate_seeds<IntegrationType::RADAU>(std::min(std::max(O, 2u), 10u)));
    arma::access::rw(n_rows) = static_cast<unsigned>(int_pts.n_rows);
    arma::access::rw(n_cols) = static_cast<unsigned>(int_pts.n_cols);
}

template<> void IntegrationPlan::generate<IntegrationType::LAGUERRE>(const unsigned D, const unsigned O) {
    int_pts = generate_points(D, generate_seeds<IntegrationType::LAGUERRE>(std::min(std::max(O, 2u), 20u)));
    arma::access::rw(n_rows) = static_cast<unsigned>(int_pts.n_rows);
    arma::access::rw(n_cols) = static_cast<unsigned>(int_pts.n_cols);
}

template<> void IntegrationPlan::generate<IntegrationType::IRONS>(const unsigned D, const unsigned O) {
    arma::access::rw(n_cols) = D + 1;

    if(3 == D && 2 == O) {
        arma::access::rw(n_rows) = 6;

        int_pts.set_size(n_rows, n_cols);

        constexpr auto WB = 4. / 3.;
        for(auto I = 0; I < 6; ++I) int_pts(I, 3) = WB;
        int_pts(0, 0) = int_pts(2, 1) = int_pts(4, 2) = -(int_pts(1, 0) = int_pts(3, 1) = int_pts(5, 2) = 1.);
        int_pts(0, 1) = int_pts(0, 2) = int_pts(1, 1) = int_pts(1, 2) = int_pts(2, 0) = int_pts(2, 2) = int_pts(3, 0) = int_pts(3, 2) = int_pts(4, 0) = int_pts(4, 1) = int_pts(5, 0) = int_pts(5, 1) = 0.;

        return;
    }

    if(3 == D && 3 == O) {
        arma::access::rw(n_rows) = 14;

        int_pts.set_size(n_rows, n_cols);

        constexpr auto WB = .886426593;
        constexpr auto WC = .335180055;
        constexpr auto LB = .795822426;
        constexpr auto LC = .758786911;
        for(auto I = 0; I < 6; ++I) int_pts(I, 3) = WB;
        for(auto I = 6; I < 14; ++I) int_pts(I, 3) = WC;
        int_pts(0, 0) = int_pts(2, 1) = int_pts(4, 2) = -(int_pts(1, 0) = int_pts(3, 1) = int_pts(5, 2) = LB);
        int_pts(0, 1) = int_pts(0, 2) = int_pts(1, 1) = int_pts(1, 2) = int_pts(2, 0) = int_pts(2, 2) = int_pts(3, 0) = int_pts(3, 2) = int_pts(4, 0) = int_pts(4, 1) = int_pts(5, 0) = int_pts(5, 1) = 0.;
        int_pts(6, 0) = int_pts(6, 1) = int_pts(6, 2) = int_pts(7, 1) = int_pts(7, 2) = int_pts(8, 2) = int_pts(9, 0) = int_pts(9, 2) = int_pts(10, 0) = int_pts(10, 1) = int_pts(11, 1) = int_pts(13, 0) = -LC;
        int_pts(7, 0) = int_pts(8, 0) = int_pts(8, 1) = int_pts(9, 1) = int_pts(10, 2) = int_pts(11, 0) = int_pts(11, 2) = int_pts(12, 0) = int_pts(12, 1) = int_pts(12, 2) = int_pts(13, 1) = int_pts(13, 2) = LC;

        return;
    }

    if(2 == D && 2 == O) {
        arma::access::rw(n_rows) = 5;

        int_pts.set_size(n_rows, n_cols);

        constexpr auto WB = 2. / 3.;
        for(auto I = 0; I < 5; ++I) {
            for(auto J = 0; J < 2; ++J) int_pts(I, J) = 0.;
            int_pts(I, 2) = WB;
        }
        int_pts(0, 2) *= 2.;
        int_pts(1, 0) = int_pts(2, 1) = -1.;
        int_pts(3, 0) = int_pts(4, 1) = 1.;

        return;
    }

    throw std::invalid_argument("not supported");
}

template<> void IntegrationPlan::generate<IntegrationType::TRIANGLE>(const unsigned D, const unsigned O) {
    if(3 == D) throw std::invalid_argument("not supported");

    arma::access::rw(n_cols) = 4;

    if(1 == O) {
        arma::access::rw(n_rows) = 1;

        int_pts.set_size(n_rows, n_cols);

        int_pts(0, 0) = int_pts(0, 1) = int_pts(0, 2) = 1. / 3.;
        int_pts(0, 3) = 1.;

        return;
    }

    if(2 == O) {
        arma::access::rw(n_rows) = 3;

        int_pts.set_size(n_rows, n_cols);

        int_pts(0, 0) = int_pts(0, 1) = int_pts(1, 1) = int_pts(1, 2) = int_pts(2, 0) = int_pts(2, 2) = .5;
        int_pts(0, 2) = int_pts(1, 0) = int_pts(2, 1) = 0.;
        int_pts(0, 3) = int_pts(1, 3) = int_pts(2, 3) = 1. / 3.;

        return;
    }

    if(3 == O) {
        arma::access::rw(n_rows) = 4;

        int_pts.set_size(n_rows, n_cols);

        int_pts(0, 0) = int_pts(0, 1) = int_pts(0, 2) = 1. / 3.;
        int_pts(0, 3) = -27. / 48.;

        int_pts(1, 1) = int_pts(1, 2) = int_pts(2, 0) = int_pts(2, 2) = int_pts(3, 0) = int_pts(3, 1) = .2;
        int_pts(1, 0) = int_pts(2, 1) = int_pts(3, 2) = .6;
        int_pts(1, 3) = int_pts(2, 3) = int_pts(3, 3) = 25. / 48.;

        return;
    }

    if(5 == O) {
        arma::access::rw(n_rows) = 7;

        int_pts.set_size(n_rows, n_cols);

        int_pts(0, 0) = int_pts(0, 1) = int_pts(0, 2) = 1. / 3.;
        int_pts(0, 3) = .225;

        int_pts(1, 1) = int_pts(1, 2) = int_pts(2, 0) = int_pts(2, 2) = int_pts(3, 0) = int_pts(3, 1) = .4701420641;
        int_pts(1, 0) = int_pts(2, 1) = int_pts(3, 2) = .0597158717;
        int_pts(1, 3) = int_pts(2, 3) = int_pts(3, 3) = .1323941527;

        int_pts(4, 1) = int_pts(4, 2) = int_pts(5, 0) = int_pts(5, 2) = int_pts(6, 0) = int_pts(6, 1) = .1012865073;
        int_pts(4, 0) = int_pts(5, 1) = int_pts(6, 2) = .7974269853;
        int_pts(4, 3) = int_pts(5, 3) = int_pts(6, 3) = .1259391805;

        return;
    }

    throw std::invalid_argument("not supported");
}
