/*******************************************************************************
 * Copyright (C) 2017-2025 Theodore Chang
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
/**
 * @class ArmstrongFrederick
 * @brief The ArmstrongFrederick class defines a nonlinear hardening material with mixed
 * hardening (isotropic and kinematic) based on J2 plasticity rule.
 *
 * The isotropic hardening is defined as an exponential function.
 *
 * The kinematic hardening consists of multiple Armstrong--Frederick type back stresses.
 *
 * algorithm verified at 06 October 2019 by tlc
 *
 * @author tlc
 * @date 06/10/2019
 * @version 1.0.0
 * @file ArmstrongFrederick.h
 * @addtogroup Material-3D
 * @{
 */

#ifndef ARMSTRONGFREDERICK_H
#define ARMSTRONGFREDERICK_H

#include <Material/Material3D/Material3D.h>

struct DataArmstrongFrederick {
    const double elastic_modulus; // elastic modulus
    const double poissons_ratio;  // poisson's ratio
    const double yield;           // yield stress
    const double hardening;       // linear isotropic hardening modulus
    const double saturation;      // saturation stress
    const double ms;              // saturation rate
    const vec a, b;
};

class ArmstrongFrederick final : protected DataArmstrongFrederick, public Material3D {
    static constexpr unsigned max_iteration = 20u;
    static const double root_three_two;
    static const mat unit_dev_tensor;

    const unsigned size = static_cast<unsigned>(a.size());

    const double shear = elastic_modulus / (2. + 2. * poissons_ratio); // shear modulus
    const double double_shear = 2. * shear;
    const double three_shear = 3. * shear;
    const double root_six_shear = sqrt(6.) * shear;

public:
    ArmstrongFrederick(
        unsigned,                 // tag
        DataArmstrongFrederick&&, // material data
        double = 0.               // density
    );

    int initialize(const shared_ptr<DomainBase>&) override;

    unique_ptr<Material> get_copy() override;

    [[nodiscard]] double get_parameter(ParameterType) const override;

    int update_trial_status(const vec&) override;

    int clear_status() override;
    int commit_status() override;
    int reset_status() override;

    void print() override;
};

#endif

//! @}
