%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
%           USER INTERFACE FOR MAYOTTE SURROGATE MODELS                   %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Created by Clea Denamiel
% For information contact: Clea Denamiel at cdenami@irb.hr
% last modifications: 
%
% The user friendly interface is used to run the Mayotte submarine lanslide
% surrogate models, display the results and save the output in PTHA_results.mat containing:
%
% PTHA:
%    struct array with fields: 
% 
%    PTHA.input_parameters 
%         zone:       string     
%         area:       string
%         distance:   [1x1 struct] 
%         volume:     [1x1 struct] 
%         friction:   [1x1 struct] 
%    PTHA.max_ele 
%         value: Petite Terre [211x1000 double] => 211 coastal locations and 1000 samples generated by surrogate model 
%             or Grande Terre [287x1000 double] => 287 coastal locations and 1000 samples generated by the surrogate model  
%         units: 'm' 
%         long_name: 'Maximum Tsunami Elevation including tide' 
%    PTHA.max_vel 
%         value: Petite Terre [211x1000 double] => 211 coastal locations and 1000 samples generated by surrogate model 
%             or Grande Terre [287x1000 double] => 287 coastal locations and 1000 samples generated by the surrogate model  
%         units: 'm/s' 
%         long_name: 'Maximum Tsunami Current including tide' 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Initialisation
%--------------------------------------------------------------------------
function varargout = mayotte_landslide_tsunami_surrogate_model(varargin)
gui_Singleton = 1;
gui_State = struct('gui_Name',       mfilename, ...
                   'gui_Singleton',  gui_Singleton, ...
                   'gui_OpeningFcn', @mayotte_landslide_tsunami_surrogate_model_OpeningFcn, ...
                   'gui_OutputFcn',  @mayotte_landslide_tsunami_surrogate_model_OutputFcn, ...
                   'gui_LayoutFcn',  [] , ...
                   'gui_Callback',   []);
if nargin && ischar(varargin{1})
    gui_State.gui_Callback = str2func(varargin{1});
end

if nargout
    [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
else
    gui_mainfcn(gui_State, varargin{:});
end
path(path,'TMD_FUNCTIONS');
%--------------------------------------------------------------------------
%% Default display settings
%--------------------------------------------------------------------------
function mayotte_landslide_tsunami_surrogate_model_OpeningFcn(hObject, eventdata, handles, varargin)
% Choose default command line output for statistics
handles.output = hObject;
% Update handles structure
guidata(hObject, handles);
set(handles.min_Distance,'Enable','Off')
set(handles.min_Volume,'Enable','Off')
set(handles.min_Friction,'Enable','Off')
set(handles.max_Distance,'Enable','Off')
set(handles.max_Volume,'Enable','Off')
set(handles.max_Friction,'Enable','Off')
%
axes(handles.Map)
fig_eq = imread('map_mayotte_landslide_zones.png');
fig_eq_res = imresize(fig_eq,1.8);
imshow(fig_eq_res)
% Update maps
set(handles.Plot_1,'Visible','off')
set(handles.Plot_2,'Visible','off')
%
set(handles.Plot_3,'Visible','on')
axes(handles.Plot_3)
fig_eq = imread('splash.png');
fig_eq_res = imresize(fig_eq,1.8);
imshow(fig_eq_res)

function varargout = mayotte_landslide_tsunami_surrogate_model_OutputFcn(hObject, eventdata, handles) 
varargout{1} = handles.output;
%--------------------------------------------------------------------------
%% Distance
%--------------------------------------------------------------------------
function min_Distance_Callback(hObject, eventdata, handles) %#ok<*DEFNU>
set(handles.Infos,'String','Press Download')
function min_Distance_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function max_Distance_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function max_Distance_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function imposed_Distance_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
max_dist_val=evalin('base','max_dist_val');
valeur=get(handles.imposed_Distance,'Value');
if valeur == 0
    set(handles.min_Distance,'Enable','on')
    set(handles.max_Distance,'Enable','on')
else 
    set(handles.min_Distance,'Enable','on')
    set(handles.max_Distance,'Enable','off')
    set(handles.max_Distance,'String',num2str(max_dist_val))
end
%--------------------------------------------------------------------------
%% Volume
%--------------------------------------------------------------------------
function min_Volume_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function min_Volume_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function max_Volume_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function max_Volume_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function imposed_Volume_Callback(hObject, eventdata, handles)
max_volu_val=evalin('base','max_volu_val');
valeur=get(handles.imposed_Volume,'Value');
set(handles.Infos,'String','Press Download')
if valeur == 0
    set(handles.min_Volume,'Enable','on')
    set(handles.max_Volume,'Enable','on')
else 
    set(handles.min_Volume,'Enable','on')
    set(handles.max_Volume,'Enable','off')
    set(handles.max_Volume,'String',num2str(max_volu_val))
end
%--------------------------------------------------------------------------
%% Friction
%--------------------------------------------------------------------------
function min_Friction_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function min_Friction_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function max_Friction_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function max_Friction_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function imposed_Friction_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
valeur=get(handles.imposed_Friction,'Value');
if valeur == 0
    set(handles.min_Friction,'Enable','on')
    set(handles.max_Friction,'Enable','on')
else 
    set(handles.min_Friction,'Enable','on')
    set(handles.max_Friction,'Enable','off')
    set(handles.max_Friction,'String','12')
end
%--------------------------------------------------------------------------
%% Date and Tide
%--------------------------------------------------------------------------
function year_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function year_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function month_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function month_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function day_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function day_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function hour_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
function hour_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function imposed_Tide_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Download')
valeur=get(handles.imposed_Tide,'Value');
if valeur == 1
    set(handles.year,'Enable','on')
    set(handles.month,'Enable','on')
    set(handles.day,'Enable','on')
    set(handles.hour,'Enable','on')
else 
    set(handles.year,'Enable','off')
    set(handles.month,'Enable','off')
    set(handles.day,'Enable','off')
    set(handles.hour,'Enable','off')
    set(handles.year,'String','2025')
    set(handles.month,'String','1')
    set(handles.day,'String','1')
    set(handles.hour,'String','0')
end
%--------------------------------------------------------------------------
%% Selection of the zone
%--------------------------------------------------------------------------
% Informations
function Infos_Callback(hObject, eventdata, handles) %#ok<*INUSD>
function Infos_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
% zone
function Select_Callback(hObject, eventdata, handles)
zone=get(handles.Zone,'Value');
area=get(handles.Area,'Value');
if area==1
    set(handles.Infos,'String','Select a Coastal Area then Press Select')
else
load('../data/location.mat')
load('../data/zone.mat');
if zone==1
    set(handles.Infos,'String','Select a Landslide Zone then Press Select')
elseif zone==2
    load('../data/Piton_DGP_dim_3_order_6_coeff.mat');
    min_dist_val=0;
    max_dist_val=919;
    min_volu_val=1;
    max_volu_val=200;
    zone_name = 'Piton';
    zone_x=x_piton;
    zone_y=y_piton;
    zone_dist=dist_piton;
    if area==2
        coeff_hat=coeff_hat_petite;
        coast_x=x_petite;
        coast_y=y_petite;
        coast_lon=lon_petite;
        coast_lat=lat_petite;     
        area_name = 'Petite Terre';
    end
    if area==3
        coeff_hat=coeff_hat_grande;
        coast_x=x_grande;
        coast_y=y_grande;
        coast_lon=lon_grande;
        coast_lat=lat_grande;  
        area_name = 'Grande Terre';
    end
elseif zone==3
    load('../data/Piton_Offshore_DGP_dim_3_order_6_coeff.mat');    
    min_dist_val=0;
    max_dist_val=1190;
    min_volu_val=1;
    max_volu_val=100;
    zone_name = 'Piton Offshore';
    zone_x=x_piton_offshore;
    zone_y=y_piton_offshore;    
    zone_dist=dist_piton_offshore;    
    if area==2
        coeff_hat=coeff_hat_petite;
        coast_x=x_petite;
        coast_y=y_petite;
        coast_lon=lon_petite;
        coast_lat=lat_petite;  
        area_name = 'Petite Terre';
    end
    if area==3
        coeff_hat=coeff_hat_grande;
        coast_x=x_grande;
        coast_y=y_grande;
        coast_lon=lon_grande;
        coast_lat=lat_grande;  
        area_name = 'Grande Terre';
    end
elseif zone==4
    load('../data/Piton_South_DGP_dim_3_order_6_coeff.mat');    
    min_dist_val=0;
    max_dist_val=5045;
    min_volu_val=1;
    max_volu_val=50;
    zone_name = 'Piton South';
    zone_x=x_piton_south;
    zone_y=y_piton_south;    
    zone_dist=dist_piton_south;    
    if area==2
        coeff_hat=coeff_hat_petite;
        coast_x=x_petite;
        coast_y=y_petite;
        coast_lon=lon_petite;
        coast_lat=lat_petite; 
        area_name = 'Petite Terre';
    end
    if area==3
        coeff_hat=coeff_hat_grande;
        coast_x=x_grande;
        coast_y=y_grande;
        coast_lon=lon_grande;
        coast_lat=lat_grande;  
        area_name = 'Grande Terre';
    end    
elseif zone==5
    load('../data/Mayotte_South_DGP_dim_3_order_6_coeff.mat');    
    min_dist_val=0;
    max_dist_val=6071;
    min_volu_val=1;
    max_volu_val=50;
    zone_name = 'Mayotte South';
    zone_x=x_mayotte_south;
    zone_y=y_mayotte_south;    
    zone_dist=dist_mayotte_south;    
    if area==2
        coeff_hat=coeff_hat_petite;
        coast_x=x_petite;
        coast_y=y_petite;
        coast_lon=lon_petite;
        coast_lat=lat_petite;   
        area_name = 'Petite Terre';
    end
    if area==3
        coeff_hat=coeff_hat_grande;
        coast_x=x_grande;
        coast_y=y_grande;
        coast_lon=lon_grande;
        coast_lat=lat_grande; 
        area_name = 'Grande Terre';
    end    
elseif zone==6
    load('../data/Mayotte_Offshore_DGP_dim_3_order_6_coeff.mat');    
    min_dist_val=0;
    max_dist_val=3568;
    min_volu_val=1;
    max_volu_val=100;
    zone_name = 'Mayotte Offshore';
    zone_x=x_mayotte_offshore;
    zone_y=y_mayotte_offshore; 
    zone_dist=dist_mayotte_offshore; 
    if area==2
        coeff_hat=coeff_hat_petite;
        coast_x=x_petite;
        coast_y=y_petite;
        coast_lon=lon_petite;
        coast_lat=lat_petite;   
        area_name = 'Petite Terre';
    end
    if area==3
        coeff_hat=coeff_hat_grande;
        coast_x=x_grande;
        coast_y=y_grande;
        coast_lon=lon_grande;
        coast_lat=lat_grande;                        
        area_name = 'Grande Terre';
    end    
end
end
assignin('base','area_name',area_name);
assignin('base','zone_name',zone_name);
assignin('base','zone_x',zone_x);
assignin('base','zone_y',zone_y);
assignin('base','zone_dist',zone_dist);
assignin('base','coast_x',coast_x);
assignin('base','coast_y',coast_y);
assignin('base','coast_lon',coast_lon);
assignin('base','coast_lat',coast_lat);
assignin('base','coeff_hat',coeff_hat);
assignin('base','min_dist_val',min_dist_val);
assignin('base','max_dist_val',max_dist_val);
assignin('base','min_volu_val',min_volu_val);
assignin('base','max_volu_val',max_volu_val);
if zone~=1 && area~=1
    load('../data/topo_bath.mat')
    axes(handles.Map)
    cla(handles.Map,'reset')
    axes(handles.Map)
    pcolor(topobath_x,topobath_y,topobath_z); 
    shading interp;
    colormap([m_colmap('blues',100);m_colmap('gland',50)]);  
    caxis([-1000 500]);
    colorbar
    axis equal
    xlim([min([zone_x(:);coast_x(:)])-50 max([zone_x(:);coast_x(:)+2000])])
    ylim([min([zone_y(:);coast_y(:)])-50 max([zone_y(:);coast_y(:)+2000])])
    hold on
    hzone=plot(zone_x,zone_y,'r','LineWidth',5);
    set(hzone, 'ButtonDownFcn', {@selectStartEndCallback, handles, zone_x, zone_y, zone_dist, min_dist_val, max_dist_val, min_volu_val, max_volu_val});
    hold off
end
set(handles.Infos,'String','Click on the Map (red line) to select Landslide Locations (min/max: 2 points or imposed: 1 point)')
set(handles.min_Distance,'Enable','on')
set(handles.max_Distance,'Enable','on')
set(handles.imposed_Distance,'Enable','On')
set(handles.min_Distance,'String',num2str(min_dist_val))
set(handles.max_Distance,'String',num2str(max_dist_val))
set(handles.imposed_Distance,'Visible','On')
set(handles.min_text,'Visible','On')
set(handles.max_text,'Visible','On')
set(handles.distance_text,'Visible','On')
set(handles.min_Distance,'Visible','On')
set(handles.max_Distance,'Visible','On')

function selectStartEndCallback(src, event, handles, zone_x, zone_y, zone_dist, min_dist_val, max_dist_val, min_volu_val, max_volu_val) 
    hold(handles.Map, 'on');
    xlim([min(zone_x(:)) max(zone_x(:))])
    ylim([min(zone_y(:)) max(zone_y(:))])
    imposed=get(handles.imposed_Distance,'Value');
    satisfy = false;
    while ~satisfy
    if imposed==0
        [xx, yy] = ginput(2); 
        distances = sqrt((zone_x - xx(1)).^2 + (zone_y - yy(1)).^2);
        [~, idx(1)] = min(distances);
        distances = sqrt((zone_x - xx(2)).^2 + (zone_y - yy(2)).^2);
        [~, idx(2)] = min(distances);
        idx=sort(idx);          
        plot(handles.Map, zone_x(idx(1):idx(2)), zone_y(idx(1):idx(2)), 'ko','MarkerSize', 8, 'MarkerFaceColor', 'k');        
        set(handles.min_Distance,'String',num2str(zone_dist(idx(1))))
        set(handles.max_Distance,'String',num2str(zone_dist(idx(2))))
    else
        [xx, yy] = ginput(1);
        distances = sqrt((zone_x - xx).^2 + (zone_y - yy).^2);
        [~, idx] = min(distances);
        idx=sort(idx);         
        plot(handles.Map, zone_x(idx), zone_y(idx), 'ko', 'MarkerSize', 8, 'MarkerFaceColor', 'k');
        set(handles.min_Distance,'String',num2str(zone_dist(idx))) 
    end
    drawnow;
    answer = questdlg('Are you satisfied with the selection?', ...
                          'Confirm Selection', ...
                          'Yes', 'No', 'No');
    if strcmp(answer, 'Yes')
        satisfy = true;
    else
        delete(findobj(gca,'Marker','o'))
        set(handles.min_Distance,'String',num2str(min_dist_val))
        set(handles.max_Distance,'String',num2str(max_dist_val))
    end
    end
    hold(handles.Map, 'off');    
    set(handles.Infos,'String','Define values for other Input Parameters, then press Download')
    set(handles.min_Volume,'Enable','on')
    set(handles.max_Volume,'Enable','on')
    set(handles.min_Volume,'String',num2str(min_volu_val))
    set(handles.max_Volume,'String',num2str(max_volu_val))
    set(handles.min_Friction,'Enable','on')
    set(handles.max_Friction,'Enable','on')
    set(handles.min_Friction,'String','4')
    set(handles.max_Friction,'String','12')
    set(handles.volume_text,'Visible','On')
    set(handles.friction_text,'Visible','On')
    set(handles.imposed_Tide,'Visible','On')
    set(handles.min_Volume,'Visible','On')
    set(handles.max_Volume,'Visible','On')
    set(handles.min_Friction,'Visible','On')
    set(handles.max_Friction,'Visible','On')
    set(handles.imposed_Volume,'Visible','On')
    set(handles.imposed_Friction,'Visible','On')
    set(handles.year,'Visible','On')
    set(handles.month,'Visible','On')
    set(handles.day,'Visible','On')
    set(handles.hour,'Visible','On')
    set(handles.minute_text,'Visible','On')
    set(handles.minus_1_text,'Visible','On')
    set(handles.minus_1_text,'Visible','On')
    set(handles.Download,'Visible','On')
%--------------------------------------------------------------------------
%% Surrogate model run
%--------------------------------------------------------------------------
function Download_Callback(hObject, eventdata, handles)
%
min_dist_val=evalin('base','min_dist_val');
min_volu_val=evalin('base','min_volu_val');
max_dist_val=evalin('base','max_dist_val');
max_volu_val=evalin('base','max_volu_val');
coeff_hat=evalin('base','coeff_hat');
coast_lon=evalin('base','coast_lon');
coast_lat=evalin('base','coast_lat');
coast_x=evalin('base','coast_x');
coast_y=evalin('base','coast_y');
zone_x=evalin('base','zone_x');
zone_y=evalin('base','zone_y');
zone_name=evalin('base','zone_name');
area_name=evalin('base','area_name');
%
min_Distance=get(handles.min_Distance,'String');
min_Distance=str2double(min_Distance);
max_Distance=get(handles.max_Distance,'String');
max_Distance=str2double(max_Distance);        
min_Volume=get(handles.min_Volume,'String');
min_Volume=str2double(min_Volume);
max_Volume=get(handles.max_Volume,'String');
max_Volume=str2double(max_Volume);
min_Friction=get(handles.min_Friction,'String');
min_Friction=str2double(min_Friction);
max_Friction=get(handles.max_Friction,'String');
max_Friction=str2double(max_Friction);    
if or(min_Distance < min_dist_val , max_Distance > max_dist_val)
    set(handles.Infos,'String',['The choosen value of distance is not within the interval [' num2str(min_dist_val) ', ' num2str(max_dist_val) '] m'])
elseif or(min_Volume < min_volu_val , max_Volume > max_volu_val)
    set(handles.Infos,'String',['The choosen value of volume is not within the interval [' num2str(min_volu_val) ', ' num2str(max_volu_val) '] Mm3'])
elseif or(min_Friction < 4 , max_Friction > 12)
    set(handles.Infos,'String','The choosen value of speed is not within the interval [4 12] ')
else
    h=waitbar(0,'Downloading Surrogate Model Data');     
    maxdeg  = 6;
    nmaxdeg = 5;
    nmodes = 3;
    a = nan(nmodes,1);
    b = nan(nmodes,1);
    s = nan(nmodes,1);
    % Read the data input 
    % Distance
    a(1) = str2double(get(handles.min_Distance,'String'));
    b(1) = str2double(get(handles.max_Distance,'String'));
    s(1) = get(handles.imposed_Distance,'Value');
    % Volume
    a(2) = str2double(get(handles.min_Volume,'String'));
    b(2) = str2double(get(handles.max_Volume,'String'));
    s(2) = get(handles.imposed_Volume,'Value');
    % Friction
    a(3) = str2double(get(handles.min_Friction,'String'));
    b(3) = str2double(get(handles.max_Friction,'String'));
    s(3) = get(handles.imposed_Friction,'Value');
    % number of random samples
    nw = 1000;
    % generation of the random numbers on [0 1] interval
    rng(0,'twister')
    seed_01 = rand(nmodes,nw);
    % rescaling to fit the appropriate intervals
    Zw = repmat(a,1,nw) + repmat((b-a),1,nw).*seed_01;
    for i=1:nmodes
        if s(i)== 1
            Zw(i,:)=a(i).*ones(1,nw);
        end
    end
    waitbar(0.5,h,'Generating data for all Coastal Locations')
    ar = nan(nmodes,1);
    br = nan(nmodes,1);
    ar(1) = min_dist_val;
    br(1) = max_dist_val;
    ar(2) = min_volu_val;
    br(2) = max_volu_val;
    ar(3) = 4;
    br(3) = 12;
    % Legendre Polynomials up to maxdeg
    Le    = cell(maxdeg,1);
    Le{1} = 1;       % L_1 = 1
    Le{2} = [1 0];   % L_2 = x
    for n = 3:maxdeg    
        Le{n} = ((2*n-3)/(n-1))*[Le{n-1} 0] - ((n-2)/(n-1))*[0 0 Le{n-2}];
    end
    % Computation of the PCE
    nx=size(coeff_hat(1,1).zeta,1);
    zeta_temp = zeros(nx,nw);
    velo_temp = zeros(nx,nw);
    time_temp = zeros(nx,nw);
    for alpha_norm1 = max(0,nmaxdeg-nmodes+1):nmaxdeg
        % Smolyak coefficient
        C_alpha       = (-1).^(nmaxdeg-alpha_norm1).*nchoosek(nmodes-1,nmaxdeg-alpha_norm1);
        % Retrieving the hat coefficients
        alpha         = coeff_hat(alpha_norm1+1).alpha;
        zeta_hat      = coeff_hat(alpha_norm1+1).zeta;
        velo_hat      = coeff_hat(alpha_norm1+1).velo;
        time_hat      = coeff_hat(alpha_norm1+1).time;
        nww           = size(alpha,1);
        for l = 1:nww
            multH = 1;
            for n = 1:nmodes;
                multH = multH.*polyval(Le{alpha(l,n)+1},(2*Zw(n,:)-ar(n)-br(n))./(br(n)-ar(n)));
            end
            for i = 1:nx
                zeta_temp(i,:)= squeeze(zeta_temp(i,:))+ C_alpha.*squeeze(zeta_hat(i,l))'*multH;
                velo_temp(i,:)= squeeze(velo_temp(i,:))+ C_alpha.*squeeze(velo_hat(i,l))'*multH;
                time_temp(i,:)= squeeze(time_temp(i,:))+ C_alpha.*squeeze(time_hat(i,l))'*multH;                
            end
        end
        max_ele=exp(zeta_temp);
        max_vel=exp(velo_temp);
        max_toa=exp(time_temp);
    end
    % adding the tides
    tide=get(handles.imposed_Tide,'Value');
    if tide == 1
        waitbar(0.75,h,'Adding Maximum Tidal Elevation')    
        lon=coast_lon; 
        lat=coast_lat;
        nnn=length(lon);
        d0=datenum(1992,1,1);
        ds=datenum(str2num(get(handles.year,'String')),str2num(get(handles.month,'String')),str2num(get(handles.day,'String')))+str2num(get(handles.hour,'String'))/24; %#ok<*ST2NM>
        [amp,pha,~,conList] = tmd_extract_HC('Model_tpxo7.2',lat,lon,'z',[]);
        for jj = 1:nnn        
            d1 = ds + (min(floor(max_toa(jj,:))):max(ceil(max_toa(jj,:))))./24;
            time = d1-d0;
            cph=-1i*pha(:,jj)*pi/180;
            hc=amp(:,jj).*exp(cph);
            hc=squeeze(hc);
            TS=harp1(time,hc,conList);
            dh = InferMinor(hc,conList,d1);
            e_tide(jj,1:length(d1))= TS + dh; %#ok<AGROW>
        end
        max_tide_ele = max(e_tide,[],2);
        max_tide_ele(isnan(max_tide_ele))=0;
        [u,phu,~,~] = tmd_extract_HC('Model_tpxo7.2',lat,lon,'u',[]);
        for jj = 1:nnn
            d1 = ds + (min(floor(max_toa(jj,:))):max(ceil(max_toa(jj,:))))./24;
            time = d1-d0;
            cph=-1i*phu(:,jj)*pi/180;
            hc=u(:,jj).*exp(cph);
            hc=squeeze(hc);
            TS=harp1(time,hc,conList);
            dh = InferMinor(hc,conList,d1);
            u_tide(jj,1:length(d1))= TS + dh; %#ok<AGROW>
        end    
        [v,phv,~,~] = tmd_extract_HC('Model_tpxo7.2',lat,lon,'u',[]);
        for jj = 1:nnn        
            d1 = ds + (min(floor(max_toa(jj,:))):max(ceil(max_toa(jj,:))))./24;
            time = d1-d0;
            cph=-1i*phv(:,jj)*pi/180;
            hc=v(:,jj).*exp(cph);
            hc=squeeze(hc);
            TS=harp1(time,hc,conList);
            dh = InferMinor(hc,conList,d1);
            v_tide(jj,1:length(d1))= TS + dh; %#ok<AGROW>
        end   
        max_tide_vel = max(sqrt(u_tide.^2+v_tide.^2),[],2)./100;
        max_tide_vel(isnan(max_tide_vel))=0;
        max_ele = max_ele + repmat(max_tide_ele,1,nw);
        max_vel = max_vel + repmat(max_tide_vel,1,nw);   
    else
    	max_tide_ele = zeros(nx,1);
    	max_tide_vel = zeros(nx,1);       
    end 
    %
    PTHA.input_parameters.zone = zone_name;
    PTHA.input_parameters.area = area_name;
    if (s(1) == 0)
    	PTHA.input_parameters.distance.value  = [a(1) b(1)];
    else
    	PTHA.input_parameters.distance.value  = a(1);
    end
    PTHA.input_parameters.distance.units = 'm';
    PTHA.input_parameters.distance.long_name = 'Distance from the southern point of the landslide zone.';
    if (s(2) == 0)
    	PTHA.input_parameters.volume.value  = [a(2) b(2)];
    else
    	PTHA.input_parameters.volume.value  = a(2);
    end
    PTHA.input_parameters.volume.units = 'Mm3';
    PTHA.input_parameters.volume.long_name = 'Volume of the landslide';
    if (s(3) == 0)
    	PTHA.input_parameters.friction.value  = [a(3) b(3)];
    else
    	PTHA.input_parameters.friction.value  = a(3);
    end
    PTHA.input_parameters.friction.units = 'degree';
    PTHA.input_parameters.friction.long_name = 'Angle of friction of the landslide.';
    %
    PTHA.max_ele.value          = max_ele;
    PTHA.max_vel.value          = max_vel;
    PTHA.max_tide_ele.value     = max_tide_ele;
    PTHA.max_tide_vel.value     = max_tide_vel;
    PTHA.max_ele.units = 'm';
    PTHA.max_ele.long_name = 'Maximum Tsunami Elevation including tide';
    PTHA.max_tide_ele.units = 'm';
    PTHA.max_tide_ele.long_name = 'Maximum Tidal Elevation extracted with TMD TPXO 7.2 model';
    PTHA.max_vel.units = 'm/s';
    PTHA.max_vel.long_name = 'Maximum Tsunami Currents including tide';
    PTHA.max_tide_vel.units = 'm/s';
    PTHA.max_tide_vel.long_name = 'Maximum Tidal Currents extracted with TMD TPXO 7.2 model';
    save ../results/PTHA_results.mat PTHA
    %    
    assignin('base','max_ele',max_ele);
    assignin('base','max_vel',max_vel);
    assignin('base','time_max',max_toa);    
    waitbar(1)
    close (h)
    set(handles.Select_OK,'Value',1)
    set(handles.Infos,'String','Click on the Map to Select a Location (White Dot)')
    set(handles.location_text,'Visible','On')
    set(handles.surrogate_text,'Visible','On')
    set(handles.coordinates_text,'Visible','On')
    set(handles.east_text,'Visible','On')
    set(handles.north_text,'Visible','On')
    set(handles.Surrogate,'Visible','On')
    set(handles.East,'Visible','On')
    set(handles.North,'Visible','On')
    axes(handles.Map)    
    hold(handles.Map, 'on');
    xlim([min([zone_x(:);coast_x(:)])-50 max([zone_x(:);coast_x(:)+2000])])
    ylim([min([zone_y(:);coast_y(:)])-50 max([zone_y(:);coast_y(:)+2000])])
    handles.dots = gobjects(length(coast_x), 1);
    set(gcf, 'Pointer', 'crosshair');  % selection mode
    for i = 1:length(coast_x)
       handles.dots(i) = plot(handles.Map, coast_x(i),coast_y(i),'o','MarkerSize',10,'MarkerEdgeColor','k','MarkerFaceColor','w','LineWidth',1, 'ButtonDownFcn', {@pointClickCallback, handles, i});
    end
    hold(handles.Map, 'off');    
end
%--------------------------------------------------------------------------
function pointClickCallback(hObject, eventdata, handles, pointIndex)
    for i = 1:length(handles.dots)
        if isvalid(handles.dots(i)) && isprop(handles.dots(i), 'MarkerFaceColor')
            set(handles.dots(i), 'MarkerFaceColor', 'w');
        end
    end
    ancestor(hObject, 'axes');
    assignin('base','point_select',pointIndex); 
    set(hObject, 'MarkerFaceColor', 'red');
    x_select = get(hObject, 'Xdata');
    y_select = get(hObject, 'Ydata');
    setappdata(gcf, 'selection_mode', false);
    set(handles.Surrogate, 'String', num2str(pointIndex));
    set(handles.East, 'String', num2str(x_select));
    set(handles.North, 'String', num2str(y_select));
    axes(handles.Plot_1)
    cla(handles.Plot_1,'reset')
    set(handles.Plot_1,'Visible','Off')
    axes(handles.Plot_2)
    cla(handles.Plot_2,'reset')
    set(handles.Plot_2,'Visible','Off')
    axes(handles.Plot_3)
    cla(handles.Plot_3,'reset')
    set(handles.Plot_3,'Visible','Off')
    set(handles.frame_stat,'Visible','Off')
    set(handles.P_01,'Visible','Off')
    set(handles.P_05,'Visible','Off')
    set(handles.P_10,'Visible','Off')
    set(handles.P_50,'Visible','Off')
    set(handles.P_100,'Visible','Off')
    set(handles.P_150,'Visible','Off')
    set(handles.P_200,'Visible','Off')
    set(handles.PP_01,'Visible','Off')
    set(handles.PP_05,'Visible','Off')
    set(handles.PP_10,'Visible','Off')
    set(handles.PP_50,'Visible','Off')
    set(handles.PP_100,'Visible','Off')
    set(handles.PP_150,'Visible','Off')
    set(handles.PP_200,'Visible','Off')
    set(handles.D_01,'Visible','Off')
    set(handles.D_05,'Visible','Off')
    set(handles.D_10,'Visible','Off')
    set(handles.D_50,'Visible','Off')
    set(handles.D_100,'Visible','Off')
    set(handles.D_150,'Visible','Off')
    set(handles.D_200,'Visible','Off')    
    select_ok=get(handles.Select_OK,'String');
    select_ok=str2double(select_ok);
    download_ok=get(handles.Download_OK,'String');
    download_ok=str2double(download_ok);
    if select_ok == 0
        set(handles.Infos,'String','First Select Landslide Zone and Coastal Area')
    elseif download_ok == 0
        set(handles.Infos,'String','First Edit and Download Input Parameters')
    else
        time_max=evalin('base','time_max');
        point_select=evalin('base','point_select');
        ele_max=evalin('base','max_ele');
        vel_max=evalin('base','max_vel');
        %
        set(handles.Plot_1,'Visible','On')
        axes(handles.Plot_1)
        hold on;
        histogram(ele_max(point_select,:),'Normalization','probability');        
        [f,x] = ecdf(ele_max(point_select,:));
        ylabel('Probability')
        yyaxis right
        plot(x,f,'r');
        ylabel('CDF')
        box on;
        set(gca,'FontSize',14)
        title('Maximum Elevation [m]')
        hold off;
        %
        set(handles.Plot_2,'Visible','On')
        axes(handles.Plot_2)
        hold on;
        histogram(vel_max(point_select,:),'Normalization','probability');        
        [f,x] = ecdf(vel_max(point_select,:));
        ylabel('Probability')
        yyaxis right
        plot(x,f,'r');
        ylabel('CDF')
        box on;
        set(gca,'FontSize',14)
        title('Maximum Speed [m/s]')
        hold off;        
        %
        set(handles.Plot_3,'Visible','On')
        axes(handles.Plot_3)
        hold on;
        [f,x] = ksdensity(time_max(point_select,:)./60);        
        area(x,f,'FaceAlpha',0.5,'FaceColor',[1 0.843137264251709 0]);
        box on;
        set(gca,'FontSize',14)
        ylabel('Probabilty Density')
        title('Time of Arrival [min]')
        hold off;
        %                             
        val = ele_max(point_select,:);
        tot = length(val);    
        set(handles.frame_stat,'Visible','On')
        set(handles.P_01,'Visible','On')
        set(handles.P_05,'Visible','On')
        set(handles.P_10,'Visible','On')
        set(handles.P_50,'Visible','On')
        set(handles.P_100,'Visible','On')
        set(handles.P_150,'Visible','On')
        set(handles.P_200,'Visible','On')
        set(handles.PP_01,'Visible','On')
        set(handles.PP_05,'Visible','On')
        set(handles.PP_10,'Visible','On')
        set(handles.PP_50,'Visible','On')
        set(handles.PP_100,'Visible','On')
        set(handles.PP_150,'Visible','On')
        set(handles.PP_200,'Visible','On')
        set(handles.D_01,'Visible','On')
        set(handles.D_05,'Visible','On')
        set(handles.D_10,'Visible','On')
        set(handles.D_50,'Visible','On')
        set(handles.D_100,'Visible','On')
        set(handles.D_150,'Visible','On')
        set(handles.D_200,'Visible','On')
        %                
        ind_01   = length(find(val >= 0.5))*100/tot;
        ind_05   = length(find(val >= 1.0))*100/tot;
        ind_10   = length(find(val >= 3.0))*100/tot;
        ind_50   = length(find(val >= 5.0))*100/tot;
        ind_100  = length(find(val >= 10.0))*100/tot;
        ind_500  = length(find(val >= 15.0))*100/tot;
        ind_1000 = length(find(val >= 20.0))*100/tot;              
        set(handles.D_01,'String',num2str(ceil(ind_01)));
        set(handles.D_05,'String',num2str(ceil(ind_05)));
        set(handles.D_10,'String',num2str(ceil(ind_10)));
        set(handles.D_50,'String',num2str(ceil(ind_50)));
        set(handles.D_100,'String',num2str(ceil(ind_100)));
        set(handles.D_150,'String',num2str(ceil(ind_500)));
        set(handles.D_200,'String',num2str(ceil(ind_1000)));
    end
    set(handles.Infos,'String','Select another Location or Press Reset for other selections')
    
%--------------------------------------------------------------------------
%% Non visible - to insure everything went OK
function Select_OK_Callback(hObject, eventdata, handles) 
%--------------------------------------------------------------------------
%% Non visible - to insure everything went OK
function Download_OK_Callback(hObject, eventdata, handles)     
%--------------------------------------------------------------------------
%% Zone choice
%--------------------------------------------------------------------------
function Zone_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Select Coastal Area then press Select')
function Zone_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
%--------------------------------------------------------------------------
%% Area choice
%--------------------------------------------------------------------------
function Area_Callback(hObject, eventdata, handles)
set(handles.Infos,'String','Press Select')
function Area_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
%--------------------------------------------------------------------------
%% Reset
%--------------------------------------------------------------------------
function Reset_Callback(hObject, eventdata, handles)
%
set(gcf,'Pointer','arrow')
%
set(handles.Infos,'String','Select Landslide Zone and Coastal Area')
%
axes(handles.Map)
cla(handles.Map,'reset')
axes(handles.Map)
fig_eq = imread('map_mayotte_landslide_zones.png');
fig_eq_res = imresize(fig_eq,1.2);
imshow(fig_eq_res)
%
axes(handles.Plot_1)
cla(handles.Plot_1,'reset')
set(handles.Plot_1,'Visible','Off')
axes(handles.Plot_2)
cla(handles.Plot_2,'reset')
set(handles.Plot_2,'Visible','Off')
%
set(handles.Plot_3,'Visible','on')
axes(handles.Plot_3)
fig_eq = imread('splash.png');
fig_eq_res = imresize(fig_eq,1.8);
imshow(fig_eq_res)
%
set(handles.location_text,'Visible','Off')
set(handles.surrogate_text,'Visible','Off')
set(handles.Surrogate,'String','')
set(handles.Surrogate,'Visible','Off')
set(handles.coordinates_text,'Visible','Off')
set(handles.east_text,'Visible','Off')
set(handles.East,'String','')
set(handles.East,'Visible','Off')
set(handles.north_text,'Visible','Off')
set(handles.North,'String','')
set(handles.North,'Visible','Off')
%
set(handles.Zone,'Value',1)
set(handles.Area,'Value',1)
set(handles.min_text,'Visible','off')
set(handles.max_text,'Visible','off')
set(handles.distance_text,'Visible','off')
set(handles.imposed_Distance,'Value',0)
set(handles.imposed_Distance,'Visible','off')
set(handles.min_Distance,'String','')
set(handles.min_Distance,'Visible','off')
set(handles.max_Distance,'String','')
set(handles.max_Distance,'Visible','off')
set(handles.volume_text,'Visible','off')
set(handles.imposed_Volume,'Value',0)
set(handles.imposed_Volume,'Visible','off')
set(handles.min_Volume,'String','')
set(handles.min_Volume,'Visible','off')
set(handles.max_Volume,'String','')
set(handles.max_Volume,'Visible','off')
set(handles.friction_text,'Visible','off')
set(handles.imposed_Friction,'Value',0)
set(handles.imposed_Friction,'Visible','off')
set(handles.min_Friction,'String','')
set(handles.min_Friction,'Visible','off')
set(handles.max_Friction,'String','')
set(handles.max_Friction,'Visible','off')
%
set(handles.imposed_Tide,'Value',0)
set(handles.imposed_Tide,'Visible','off')
set(handles.year,'String','2025')
set(handles.month,'String','1')
set(handles.day,'String','1')
set(handles.hour,'String','0')
set(handles.year,'Visible','off')
set(handles.month,'Visible','off')
set(handles.day,'Visible','off')
set(handles.hour,'Visible','off')
set(handles.minus_1_text,'Visible','off')
set(handles.minus_2_text,'Visible','off')
set(handles.minute_text,'Visible','off')
set(handles.Download,'Visible','off')
%
set(handles.frame_stat,'Visible','Off')
set(handles.P_01,'Visible','Off')
set(handles.P_05,'Visible','Off')
set(handles.P_10,'Visible','Off')
set(handles.P_50,'Visible','Off')
set(handles.P_100,'Visible','Off')
set(handles.P_150,'Visible','Off')
set(handles.P_200,'Visible','Off')
set(handles.PP_01,'Visible','Off')
set(handles.PP_05,'Visible','Off')
set(handles.PP_10,'Visible','Off')
set(handles.PP_50,'Visible','Off')
set(handles.PP_100,'Visible','Off')
set(handles.PP_150,'Visible','Off')
set(handles.PP_200,'Visible','Off')
set(handles.D_01,'Visible','Off')
set(handles.D_05,'Visible','Off')
set(handles.D_10,'Visible','Off')
set(handles.D_50,'Visible','Off')
set(handles.D_100,'Visible','Off')
set(handles.D_150,'Visible','Off')
set(handles.D_200,'Visible','Off')
%
set(gcf, 'Pointer', 'arrow');
evalin('base', 'clear');
%--------------------------------------------------------------------------
%% statistics
%--------------------------------------------------------------------------
function D_01_Callback(hObject, eventdata, handles)
function D_01_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function D_05_Callback(hObject, eventdata, handles)
function D_05_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function D_10_Callback(hObject, eventdata, handles)
function D_10_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function D_50_Callback(hObject, eventdata, handles)
function D_50_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function D_100_Callback(hObject, eventdata, handles)
function D_100_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function D_150_Callback(hObject, eventdata, handles)
function D_150_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
function D_200_Callback(hObject, eventdata, handles)
function D_200_CreateFcn(hObject, eventdata, handles)
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end
