function cols=s_colmap(nme,m,ncol)
% s_colmap  Useful colormaps

% NOTE: changed all calls from m_colmap to s_colmap- would need to change
% back if using this script for m_map package


%   s_colmap(NAME) returns an M-by-3 matrix containing the NAME colormap
%   where NAME is one of:
%     'jet' : a perceptually uniform variation of the JET colormap. It
%             contains the multiple colours which make JET useful, while
%             avoiding the weird highlighting especially around yellow and
%             cyan. The colors begin with dark blue, range through shades
%             of blue, green, orange and red, and ends with dark red.
%     'blue' : a perceptually useful blue shading (good for bathymetry)
%     'green' : a perceptually useful green shading
%     'diverging' : a blue/red diverging colormap
%     'odv'  : an isoluminant map
%     'cyclic': a cyclic colormap
%     'land' : a topographic height (green-brown-white) shading
%     'water' : blue shading for water (goes with 'land').
%     'gland' : a topographic height  shading with more green
%     'bland' : a topographic height shading with browns only.
%     'BOD' : a blue/orange diverging colormap
%     'mBOD' : a modified blue/orange diverging colormap without white,
%              useful as a colorblind-friendly jet alternative
%
%   and M is the same length as the current figure's colormap. If no
%   figure exists, the length of the default colormap is used. The
%   length can be explicitly specified with s_colmap(NAME,M).
%
%   s_colmap('demo') demonstrates the colormaps.
%
%   s_colmap(NAME,'step') returns a 256-color map in which colours are
%   perceptually bunched into 16 separate colours. This is useful
%   if you want to see "edges" in what would be an otherwise
%   smooth gradation (i.e. approaching the look of contouring).
%
%   s_colmap(NAME,'demo') Gives a demo of this behavior
%
%   s_colmap(NAME,'step',M) bunches into M colours.
%
%   The jet and odv colourmaps were designed using the CET Perceptually
%   Uniform Colour Maps toolbox at peterkovesi.com/projects/colourmaps/
%
%   The blue, green, and diverging colormaps are from CBREWER, a matlab
%   implementation of colours described at http://colorbrewer2.org.
%
%   The land and gland colormaps are derived from the ETOPO1 water/land
%   colormap at http://soliton.vm.bytemark.co.uk/pub/cpt-city/ngdc/index.html
%
%   See also PARULA, HSV, HOT, PINK, FLAG, COLORMAP, RGBPLOT.

% R Pawlowicz Nov/2017

% names
if nargin==0
    s_colmap('get');
    return;
end

switch lower(nme(1:3))
    case 'jet'
        nme='jet';
    case 'blu'
        nme='blue';
    case 'gre'
        nme='green';
    case 'div'
        nme='diverging';
    case 'odv'
        nme='odv';
    case 'cyc'
        nme='cyclic';
    case 'lan'
        nme='land';
    case 'wat'
        nme='water';
    case 'gla'
        nme='gland';
    case 'bla'
        nme='bland';
    case 'chl'
        nme='chlorophyll';
    case 'bod'
        nme='BOD';
    case 'mbo'
        nme='mBOD';
    case 'get'
        fprintf('Colormaps include:\n');
        fprintf('   ''jet'' {,M|,''step''{,M}}\n');
        fprintf('   ''blue'' {,M|,''step''{,M}}\n');
        fprintf('   ''green'' {,M|,''step''{,M}}\n');
        fprintf('   ''diverging'' {,M|,''step''{,M}}\n');
        fprintf('   ''cyclic'' {,M|,''step''{,M}}\n');
        fprintf('   ''land'' {,M|,''step''{,M}}\n');
        fprintf('   ''gland'' {,M|,''step''{,M}}\n');
        fprintf('   ''bland'' {,M|,''step''{,M}}\n');
        fprintf('   ''water'' {,M|,''step''{,M}}\n');
        fprintf('   ''chlorophyll'' {,M|,''step''{,M}}\n');
        fprintf('   ''BOD'' {,M|,''step''{,M}}\n');
        fprintf('   ''mBOD'' {,M|,''step''{,M}}\n');
    case 'dem'
        colmap_demo;
        return
    otherwise
        error(['map: ' mfilename ':invalidMapname'],...
            ' First argument must be an valid colormap name ');
end


if nargin < 2
    f = gcf;
    m = size(get(f,'Colormap'),1);
    ncol=m;
else
    if isnumeric(m)
        ncol=m;
    else
        switch m(1:3)
            case 'dem'   % demo
                clf;
                subplot(2,1,1);
                imagesc(peaks(10*10));
                title('Peaks function displayed with IMAGESC');
                subplot(2,1,2);
                imagesc(1:256);
                cols=s_colmap(nme,'step');
                colormap(cols);
                title(['The colormap used above: m\_colmap(''' nme ''',256,16)']);
                if nargout==0
                    clear cols
                end
                return;
                
            case 'ste'    % step
                m=256;
                if nargin<3
                    ncol=16;
                end
            otherwise
                error(['map: ' mfilename ':invalidOption'],...
                    ' Second argument must be an valid option');
        end
    end
end

% Whatever length colormap we have

if ncol<m/2
    xx=[0:(m-1)]'/(m-1)*pi*ncol;
    yy=cumsum(cos(xx).^4 + 1/30);
    yy=yy-yy(1);
    yy=yy/max(yy)*255;
else
    yy=[0:(m-1)]'/(m-1)*255;
end

if nargout>0
    [r,basecols]=mycols(nme);
    cols=interp1(r,basecols,yy);
end



end



function [r,Data]=mycols(nme)
%
% Returns raw colourmap - 0<=r<=255 and
% [0 0 0]<=Data(row,:)<=[1 1 1]
%

r=[0:255]; % Default for most of these.
switch nme
    case 'jet'
        Data=[...
            0.0000000000000000e+00   0.0000000000000000e+00   8.2097987032366326e-01
            0.0000000000000000e+00   0.0000000000000000e+00   8.2810185708388151e-01
            0.0000000000000000e+00   1.6244801768825715e-02   8.3523804428475601e-01
            0.0000000000000000e+00   3.3036471648334995e-02   8.4238547984011569e-01
            0.0000000000000000e+00   4.9815313296183217e-02   8.4954008831091266e-01
            0.0000000000000000e+00   6.4029974334975218e-02   8.5670244416752706e-01
            0.0000000000000000e+00   7.6535959975665882e-02   8.6388438002720569e-01
            0.0000000000000000e+00   8.8013865699010582e-02   8.7105469052628581e-01
            0.0000000000000000e+00   9.8600980176800038e-02   8.7821143148088110e-01
            0.0000000000000000e+00   1.0855337679557225e-01   8.8532422388838095e-01
            0.0000000000000000e+00   1.1812539206874099e-01   8.9236038456129052e-01
            0.0000000000000000e+00   1.2726308783402634e-01   8.9926170553603024e-01
            0.0000000000000000e+00   1.3619917896297423e-01   9.0595230153199524e-01
            0.0000000000000000e+00   1.4506118969622983e-01   9.1232172813169876e-01
            0.0000000000000000e+00   1.5377890552725290e-01   9.1823409334055117e-01
            0.0000000000000000e+00   1.6261365170233669e-01   9.2351427566801991e-01
            0.0000000000000000e+00   1.7154074948381501e-01   9.2794646225220667e-01
            0.0000000000000000e+00   1.8070256379609848e-01   9.3131899478139557e-01
            0.0000000000000000e+00   1.9017887289112076e-01   9.3338241072484596e-01
            0.0000000000000000e+00   1.9997783192436316e-01   9.3392154621153756e-01
            0.0000000000000000e+00   2.1019119971968461e-01   9.3274045531274641e-01
            0.0000000000000000e+00   2.2078830806041727e-01   9.2970178242713375e-01
            0.0000000000000000e+00   2.3170788397910427e-01   9.2472675872455312e-01
            0.0000000000000000e+00   2.4291058720581157e-01   9.1783217933659011e-01
            0.0000000000000000e+00   2.5439689340691901e-01   9.0907966847559551e-01
            0.0000000000000000e+00   2.6601511242910086e-01   8.9863138701299472e-01
            0.0000000000000000e+00   2.7770789341482360e-01   8.8668327712843464e-01
            0.0000000000000000e+00   2.8940651531291417e-01   8.7345282146153225e-01
            0.0000000000000000e+00   3.0102041216268027e-01   8.5918692717013723e-01
            0.0000000000000000e+00   3.1252327536427388e-01   8.4411199405573478e-01
            0.0000000000000000e+00   3.2386802405602533e-01   8.2843647500968565e-01
            0.0000000000000000e+00   3.3502534309709658e-01   8.1233252376445897e-01
            0.0000000000000000e+00   3.4591820537992940e-01   7.9594701247201749e-01
            0.0000000000000000e+00   3.5661427072219648e-01   7.7939361987056155e-01
            0.0000000000000000e+00   3.6703710422430275e-01   7.6275539017719729e-01
            0.0000000000000000e+00   3.7721017332291257e-01   7.4610487594974018e-01
            0.0000000000000000e+00   3.8709592071744631e-01   7.2949277318103267e-01
            0.0000000000000000e+00   3.9670474982054660e-01   7.1295673554612449e-01
            0.0000000000000000e+00   4.0602438327051393e-01   6.9655675508496329e-01
            0.0000000000000000e+00   4.1500561550145126e-01   6.8030864681305458e-01
            0.0000000000000000e+00   4.2366289307317656e-01   6.6426240828904592e-01
            0.0000000000000000e+00   4.3200712706984007e-01   6.4842386975272537e-01
            3.0566524560783279e-03   4.3999127354957085e-01   6.3282182663250508e-01
            2.1259594640010666e-02   4.4764881469817519e-01   6.1745638962714788e-01
            4.3803836689370011e-02   4.5501957945750093e-01   6.0233557663170700e-01
            6.5847269982244869e-02   4.6208402302425822e-01   5.8744496819586389e-01
            8.5714142185715644e-02   4.6890583204144093e-01   5.7277998204506675e-01
            1.0409596887009070e-01   4.7550636110461003e-01   5.5828074688552098e-01
            1.2102697913139572e-01   4.8192127879442975e-01   5.4395369747328215e-01
            1.3667902142111943e-01   4.8820071402585424e-01   5.2974025041805628e-01
            1.5103696859797716e-01   4.9437177951608591e-01   5.1559526287877988e-01
            1.6416735480159775e-01   5.0046436145743556e-01   5.0151677895326252e-01
            1.7599785165409093e-01   5.0651659371728996e-01   4.8743987594925903e-01
            1.8673494649110914e-01   5.1254773876555138e-01   4.7333098265162471e-01
            1.9635984308719795e-01   5.1857739467224218e-01   4.5920475411903505e-01
            2.0490590950992249e-01   5.2459707667461242e-01   4.4499829451551892e-01
            2.1249481521562941e-01   5.3063533990020706e-01   4.3070141183233479e-01
            2.1917765060054359e-01   5.3669591908749759e-01   4.1629180215797296e-01
            2.2499535088875952e-01   5.4276707337012853e-01   4.0174506555606909e-01
            2.3004736186449742e-01   5.4886421494692972e-01   3.8704858786501528e-01
            2.3439364402268012e-01   5.5497461212998789e-01   3.7217181537750266e-01
            2.3802214916377087e-01   5.6111021864810351e-01   3.5711075761778077e-01
            2.4102958080802567e-01   5.6727120617849991e-01   3.4180313298247084e-01
            2.4347078965167188e-01   5.7343697984137976e-01   3.2624718187968538e-01
            2.4541755424164602e-01   5.7961519583221810e-01   3.1046646770536129e-01
            2.4695087697744367e-01   5.8578978746082422e-01   2.9436976897198003e-01
            2.4816602364067705e-01   5.9195396468447525e-01   2.7803594615233906e-01
            2.4914804420757949e-01   5.9808477862815501e-01   2.6142686912908059e-01
            2.5009122010144075e-01   6.0419223727339377e-01   2.4457986428005363e-01
            2.5125107794115892e-01   6.1022050248824800e-01   2.2760242777985751e-01
            2.5277963649326868e-01   6.1617179084321716e-01   2.1050609513494345e-01
            2.5499887511783337e-01   6.2200150973129531e-01   1.9345024424320145e-01
            2.5811871983643009e-01   6.2770454035851242e-01   1.7653075332522819e-01
            2.6246816164275255e-01   6.3323695072608899e-01   1.5996210212986198e-01
            2.6818346039674890e-01   6.3858125645704489e-01   1.4398097898551054e-01
            2.7543338752629509e-01   6.4372114150893245e-01   1.2870986574286711e-01
            2.8418213538041015e-01   6.4865628006181242e-01   1.1444933871183166e-01
            2.9441479687067434e-01   6.5336988668170226e-01   1.0152426951505689e-01
            3.0599728766863188e-01   6.5788824014397052e-01   9.0139191413144706e-02
            3.1866282501793303e-01   6.6220359310488208e-01   8.0483895706916511e-02
            3.3224320620395048e-01   6.6633263204236381e-01   7.2801598535599379e-02
            3.4646179112792114e-01   6.7032401486586635e-01   6.7067497770581408e-02
            3.6113631412522007e-01   6.7416768477092770e-01   6.3249077315340047e-02
            3.7608731018537811e-01   6.7791233106441862e-01   6.0980464884529913e-02
            3.9114649219215752e-01   6.8155555404442747e-01   6.0237646826049381e-02
            4.0620969841971383e-01   6.8514312914868336e-01   6.0434850173806676e-02
            4.2119208206707631e-01   6.8867051135431134e-01   6.1317869943769421e-02
            4.3607730241284837e-01   6.9215167952165702e-01   6.2899876768280863e-02
            4.5080272445331049e-01   6.9559278562020677e-01   6.4630034822273294e-02
            4.6536143503241884e-01   6.9900108657586113e-01   6.6611409445883923e-02
            4.7977927019121297e-01   7.0239032638575571e-01   6.8765586458024650e-02
            4.9402278124389287e-01   7.0575587274235263e-01   7.0874113597661562e-02
            5.0812209701947120e-01   7.0910213329145999e-01   7.3065156643943877e-02
            5.2205818272473403e-01   7.1241552273829678e-01   7.5181298203090122e-02
            5.3588169773612182e-01   7.1572588892233568e-01   7.7316510344089567e-02
            5.4958796386661235e-01   7.1900198771359247e-01   7.9498911575721087e-02
            5.6318325188735974e-01   7.2225882977126032e-01   8.1739922275486848e-02
            5.7667734958705708e-01   7.2550816943346963e-01   8.3886170589407089e-02
            5.9006715802606269e-01   7.2872236431771908e-01   8.5969093205613856e-02
            6.0339198534874716e-01   7.3192594564488933e-01   8.8191004728472841e-02
            6.1663360653820753e-01   7.3510592591557200e-01   9.0348464852867502e-02
            6.2979877564545061e-01   7.3826041744638315e-01   9.2484236233912193e-02
            6.4290502541913341e-01   7.4139762708091694e-01   9.4695676765631190e-02
            6.5595456455707568e-01   7.4451903129559338e-01   9.6766064389575565e-02
            6.6894100563615921e-01   7.4762140253119092e-01   9.8966110102230720e-02
            6.8187256965709520e-01   7.5069894456381092e-01   1.0111568171360964e-01
            6.9476451769134873e-01   7.5375241811299143e-01   1.0331287444459691e-01
            7.0761384378249492e-01   7.5678894789262396e-01   1.0543392059916874e-01
            7.2042165068418740e-01   7.5980213993578027e-01   1.0760350744563843e-01
            7.3319470176764145e-01   7.6279840902831653e-01   1.0981215158679503e-01
            7.4592828571528091e-01   7.6577879845079466e-01   1.1194689692155048e-01
            7.5863468570665482e-01   7.6872670099528584e-01   1.1407090775631480e-01
            7.7131848568430994e-01   7.7166810405759800e-01   1.1627716014608953e-01
            7.8398678935532840e-01   7.7456683356005351e-01   1.1844264905380968e-01
            7.9662697582417896e-01   7.7744081794020037e-01   1.2054078396447886e-01
            8.0926185607475876e-01   7.8025300319044533e-01   1.2266783407386730e-01
            8.2187414124312763e-01   7.8300130137588675e-01   1.2480676238195154e-01
            8.3446602223939059e-01   7.8565722796567439e-01   1.2687749576058471e-01
            8.4702437066066527e-01   7.8816898161099214e-01   1.2890567635476541e-01
            8.5954289047780741e-01   7.9049204962457431e-01   1.3088474793982860e-01
            8.7197831670600356e-01   7.9255502154840052e-01   1.3269172868341109e-01
            8.8427738654617694e-01   7.9427245270485536e-01   1.3436638425443168e-01
            8.9637053729861804e-01   7.9553909023023628e-01   1.3576481705392310e-01
            9.0816444779814065e-01   7.9626169503294353e-01   1.3697659943588600e-01
            9.1953673575996886e-01   7.9633432352513089e-01   1.3786713706219822e-01
            9.3037259899096236e-01   7.9566413293269556e-01   1.3835047336237846e-01
            9.4053209111364600e-01   7.9418180027718388e-01   1.3842916138252423e-01
            9.4989207994255664e-01   7.9182887326844242e-01   1.3808153617074001e-01
            9.5836294828986246e-01   7.8861031080564803e-01   1.3725685332614224e-01
            9.6586462349906821e-01   7.8454305319884909e-01   1.3594823160952096e-01
            9.7239039139483674e-01   7.7967621023720923e-01   1.3431101877433596e-01
            9.7794177385612990e-01   7.7410205347271077e-01   1.3221480879967804e-01
            9.8259333615419153e-01   7.6791778909022512e-01   1.2980073335970169e-01
            9.8642494090379840e-01   7.6122992421425550e-01   1.2702534915073754e-01
            9.8955166954763185e-01   7.5413730388263700e-01   1.2406071795849168e-01
            9.9208992758455905e-01   7.4672824419867512e-01   1.2089567778081263e-01
            9.9415891855407701e-01   7.3909488972694637e-01   1.1760517777184400e-01
            9.9586234078238622e-01   7.3127941939636465e-01   1.1416775566548833e-01
            9.9729483003018982e-01   7.2335091637232807e-01   1.1070869560896780e-01
            9.9852727658269325e-01   7.1534184353124297e-01   1.0714334894572426e-01
            9.9960957379964188e-01   7.0726043228880653e-01   1.0352445558761095e-01
            1.0000000000000000e+00   6.9912890339600753e-01   9.9829372782125531e-02
            1.0000000000000000e+00   6.9095883736804042e-01   9.6116559516242178e-02
            1.0000000000000000e+00   6.8275705650119767e-01   9.2417489373060394e-02
            1.0000000000000000e+00   6.7452426257956222e-01   8.8677593797862289e-02
            1.0000000000000000e+00   6.6625666540404749e-01   8.4887313721726754e-02
            1.0000000000000000e+00   6.5796365276303015e-01   8.1068177807262484e-02
            1.0000000000000000e+00   6.4962701740088724e-01   7.7151213817244532e-02
            1.0000000000000000e+00   6.4124528876170073e-01   7.3351998210193614e-02
            1.0000000000000000e+00   6.3283767086881249e-01   6.9416379806024719e-02
            1.0000000000000000e+00   6.2438475544545868e-01   6.5331601005328144e-02
            1.0000000000000000e+00   6.1588280808086227e-01   6.1211652569017221e-02
            1.0000000000000000e+00   6.0734668246046875e-01   5.7123569446539284e-02
            1.0000000000000000e+00   5.9874800807184747e-01   5.2865554268786509e-02
            1.0000000000000000e+00   5.9010246493802909e-01   4.8647547440398876e-02
            1.0000000000000000e+00   5.8140951034192445e-01   4.4198570670319334e-02
            1.0000000000000000e+00   5.7266585316082430e-01   3.9705280914958743e-02
            1.0000000000000000e+00   5.6385731706157016e-01   3.5191257367171103e-02
            1.0000000000000000e+00   5.5497816055012050e-01   3.0647747322206934e-02
            1.0000000000000000e+00   5.4604761486237297e-01   2.6461611736228074e-02
            1.0000000000000000e+00   5.3704122502628560e-01   2.2436402898803214e-02
            1.0000000000000000e+00   5.2795094033373646e-01   1.8569728931739027e-02
            1.0000000000000000e+00   5.1880030560715795e-01   1.4859141284721791e-02
            1.0000000000000000e+00   5.0955197815230335e-01   1.1196012834395055e-02
            1.0000000000000000e+00   5.0021693769032671e-01   7.7056436600708390e-03
            1.0000000000000000e+00   4.9080757314831269e-01   4.4527213011970471e-03
            1.0000000000000000e+00   4.8127050430505153e-01   1.3486232596699801e-03
            1.0000000000000000e+00   4.7165053711191246e-01   0.0000000000000000e+00
            1.0000000000000000e+00   4.6190083189547609e-01   0.0000000000000000e+00
            1.0000000000000000e+00   4.5204021136716938e-01   0.0000000000000000e+00
            1.0000000000000000e+00   4.4204343506591337e-01   0.0000000000000000e+00
            1.0000000000000000e+00   4.3190627002582327e-01   0.0000000000000000e+00
            1.0000000000000000e+00   4.2159591143411812e-01   0.0000000000000000e+00
            1.0000000000000000e+00   4.1114619470166724e-01   0.0000000000000000e+00
            1.0000000000000000e+00   4.0051916733978549e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.8967477462351169e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.7863908422433407e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.6737229102939645e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.5588805687874975e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.4412420334830296e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.3213896185541525e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.1988349723243259e-01   0.0000000000000000e+00
            1.0000000000000000e+00   3.0737131121967348e-01   0.0000000000000000e+00
            1.0000000000000000e+00   2.9466234283445780e-01   0.0000000000000000e+00
            1.0000000000000000e+00   2.8180431610272688e-01   0.0000000000000000e+00
            1.0000000000000000e+00   2.6885346259089427e-01   0.0000000000000000e+00
            9.9971380921896635e-01   2.5588279168968908e-01   0.0000000000000000e+00
            9.9745107484299911e-01   2.4306303891550010e-01   0.0000000000000000e+00
            9.9471175718486304e-01   2.3053240966922203e-01   0.0000000000000000e+00
            9.9144050807482531e-01   2.1845350104652131e-01   0.0000000000000000e+00
            9.8759742405304762e-01   2.0697738741569063e-01   0.0000000000000000e+00
            9.8316166027396978e-01   1.9624582890323888e-01   0.0000000000000000e+00
            9.7814117536301937e-01   1.8638373147524231e-01   0.0000000000000000e+00
            9.7256691186559741e-01   1.7750982585337141e-01   0.0000000000000000e+00
            9.6646763333593055e-01   1.6960615229744733e-01   0.0000000000000000e+00
            9.5991667718773344e-01   1.6265283853733181e-01   0.0000000000000000e+00
            9.5297568794473220e-01   1.5659747318781603e-01   0.0000000000000000e+00
            9.4571233760129314e-01   1.5128687901353635e-01   0.0000000000000000e+00
            9.3820118122169338e-01   1.4664168011978534e-01   0.0000000000000000e+00
            9.3050268815754777e-01   1.4253427062145158e-01   0.0000000000000000e+00
            9.2266803031005806e-01   1.3878427919273950e-01   0.0000000000000000e+00
            9.1474108307638347e-01   1.3532555938839283e-01   0.0000000000000000e+00
            9.0675852417667746e-01   1.3206323417870042e-01   0.0000000000000000e+00
            8.9873962287707232e-01   1.2889180676634279e-01   0.0000000000000000e+00
            8.9070644004606214e-01   1.2586340008396713e-01   0.0000000000000000e+00
            8.8266708081748724e-01   1.2278481340036466e-01   0.0000000000000000e+00
            8.7463547859087032e-01   1.1980508899179866e-01   0.0000000000000000e+00
            8.6661131238848954e-01   1.1682153274204797e-01   0.0000000000000000e+00
            8.5859901159723606e-01   1.1382680416595528e-01   0.0000000000000000e+00
            8.5060093722026298e-01   1.1083435058870025e-01   0.0000000000000000e+00
            8.4262210493452183e-01   1.0777800601488426e-01   0.0000000000000000e+00
            8.3465143106553319e-01   1.0471633325123153e-01   0.0000000000000000e+00
            8.2669639277606366e-01   1.0170369534475480e-01   0.0000000000000000e+00
            8.1874989394241815e-01   9.8659152046269100e-02   0.0000000000000000e+00
            8.1082521361314908e-01   9.5589480024985407e-02   0.0000000000000000e+00
            8.0291243524621259e-01   9.2477562506154257e-02   0.0000000000000000e+00
            7.9501601582959724e-01   8.9390544379782902e-02   0.0000000000000000e+00
            7.8713591423470530e-01   8.6218414092508924e-02   0.0000000000000000e+00
            7.7926327536719753e-01   8.3082652674223073e-02   0.0000000000000000e+00
            7.7141345413247508e-01   7.9904163323746544e-02   0.0000000000000000e+00
            7.6356901068481819e-01   7.6683348793519776e-02   0.0000000000000000e+00
            7.5574754344824191e-01   7.3559378952035598e-02   0.0000000000000000e+00
            7.4794071009249607e-01   7.0232910613491642e-02   0.0000000000000000e+00
            7.4014602737794621e-01   6.6954364586541498e-02   0.0000000000000000e+00
            7.3236760838059556e-01   6.3633307967642341e-02   0.0000000000000000e+00
            7.2460119293990866e-01   6.0280688573269070e-02   0.0000000000000000e+00
            7.1685625358364513e-01   5.6849608855938531e-02   0.0000000000000000e+00
            7.0912481497916180e-01   5.3299226112309528e-02   0.0000000000000000e+00
            7.0140312192465981e-01   4.9829293199622995e-02   0.0000000000000000e+00
            6.9370510178801092e-01   4.6158764888123434e-02   0.0000000000000000e+00
            6.8602792806999691e-01   4.2524088717921474e-02   0.0000000000000000e+00
            6.7836388922453772e-01   3.8797586133169637e-02   0.0000000000000000e+00
            6.7072178335076416e-01   3.5066982469895418e-02   0.0000000000000000e+00
            6.6309318632075742e-01   3.1291508120498114e-02   0.0000000000000000e+00
            6.5547739533880844e-01   2.7799830769093408e-02   0.0000000000000000e+00
            6.4788052533085139e-01   2.4415535891618653e-02   0.0000000000000000e+00
            6.4031120522707030e-01   2.1137242643728692e-02   0.0000000000000000e+00
            6.3275181226008126e-01   1.7963580322279168e-02   0.0000000000000000e+00
            6.2521102502886350e-01   1.4892646934414970e-02   0.0000000000000000e+00
            6.1768851440583761e-01   1.1947271916774600e-02   0.0000000000000000e+00
            6.1018680816921134e-01   8.8617158086171516e-03   0.0000000000000000e+00
            6.0270479360661422e-01   6.0916955702822869e-03   0.0000000000000000e+00
            5.9523986538913543e-01   3.4203333573827339e-03   0.0000000000000000e+00
            5.8779123408662970e-01   8.4517490685311037e-04   0.0000000000000000e+00
            5.8037263574738407e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.7296951136703345e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.6557284013198694e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.5820630194758614e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.5086745375910373e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.4354221465215014e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.3623817219036729e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.2896046694295906e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.2168511872923218e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.1445545856746699e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.0723090697621742e-01   0.0000000000000000e+00   0.0000000000000000e+00
            5.0002769723112239e-01   0.0000000000000000e+00   0.0000000000000000e+00];
    case 'blue'
        Data=[...
            3.1372549019607843e-02   1.8823529411764706e-01   4.1960784313725491e-01
            3.1372549019607843e-02   1.9215686274509805e-01   4.2745098039215684e-01
            3.1372549019607843e-02   1.9607843137254902e-01   4.3529411764705883e-01
            3.1372549019607843e-02   2.0000000000000001e-01   4.4313725490196076e-01
            3.1372549019607843e-02   2.0392156862745098e-01   4.4705882352941179e-01
            3.1372549019607843e-02   2.0784313725490197e-01   4.5490196078431372e-01
            3.1372549019607843e-02   2.1176470588235294e-01   4.6274509803921571e-01
            3.1372549019607843e-02   2.1568627450980393e-01   4.7058823529411764e-01
            3.1372549019607843e-02   2.1960784313725490e-01   4.7843137254901963e-01
            3.1372549019607843e-02   2.2352941176470589e-01   4.8235294117647060e-01
            3.1372549019607843e-02   2.2745098039215686e-01   4.9019607843137253e-01
            3.1372549019607843e-02   2.3137254901960785e-01   4.9803921568627452e-01
            3.1372549019607843e-02   2.3529411764705882e-01   5.0588235294117645e-01
            3.1372549019607843e-02   2.4313725490196078e-01   5.0980392156862742e-01
            3.1372549019607843e-02   2.4705882352941178e-01   5.1764705882352946e-01
            3.1372549019607843e-02   2.5098039215686274e-01   5.2156862745098043e-01
            3.1372549019607843e-02   2.5490196078431371e-01   5.2941176470588236e-01
            3.1372549019607843e-02   2.5882352941176473e-01   5.3725490196078429e-01
            3.1372549019607843e-02   2.6274509803921570e-01   5.4117647058823526e-01
            3.1372549019607843e-02   2.6666666666666666e-01   5.4901960784313730e-01
            3.1372549019607843e-02   2.7058823529411763e-01   5.5294117647058827e-01
            3.1372549019607843e-02   2.7450980392156865e-01   5.5686274509803924e-01
            3.1372549019607843e-02   2.7843137254901962e-01   5.6470588235294117e-01
            3.1372549019607843e-02   2.8235294117647058e-01   5.6862745098039214e-01
            3.1372549019607843e-02   2.8627450980392155e-01   5.7647058823529407e-01
            3.1372549019607843e-02   2.9019607843137257e-01   5.8039215686274515e-01
            3.1372549019607843e-02   2.9411764705882354e-01   5.8431372549019611e-01
            3.1372549019607843e-02   2.9803921568627451e-01   5.8823529411764708e-01
            3.1372549019607843e-02   3.0196078431372547e-01   5.9607843137254901e-01
            3.1372549019607843e-02   3.0588235294117649e-01   5.9999999999999998e-01
            3.1372549019607843e-02   3.0980392156862746e-01   6.0392156862745094e-01
            3.1372549019607843e-02   3.1372549019607843e-01   6.0784313725490191e-01
            3.1372549019607843e-02   3.1764705882352939e-01   6.1176470588235299e-01
            3.1372549019607843e-02   3.2156862745098042e-01   6.1568627450980395e-01
            3.1372549019607843e-02   3.2549019607843138e-01   6.1960784313725492e-01
            3.1372549019607843e-02   3.2941176470588235e-01   6.2352941176470589e-01
            3.5294117647058823e-02   3.3333333333333331e-01   6.2745098039215685e-01
            3.5294117647058823e-02   3.3725490196078434e-01   6.3137254901960782e-01
            3.5294117647058823e-02   3.4117647058823530e-01   6.3529411764705879e-01
            3.9215686274509803e-02   3.4509803921568627e-01   6.3921568627450975e-01
            4.3137254901960784e-02   3.4901960784313724e-01   6.4313725490196083e-01
            4.3137254901960784e-02   3.5294117647058826e-01   6.4313725490196083e-01
            4.7058823529411764e-02   3.5686274509803922e-01   6.4705882352941180e-01
            5.0980392156862744e-02   3.6078431372549019e-01   6.5098039215686276e-01
            5.0980392156862744e-02   3.6470588235294116e-01   6.5490196078431373e-01
            5.4901960784313725e-02   3.6862745098039218e-01   6.5882352941176470e-01
            5.8823529411764705e-02   3.7254901960784315e-01   6.6274509803921566e-01
            6.2745098039215685e-02   3.7647058823529411e-01   6.6274509803921566e-01
            6.6666666666666666e-02   3.8039215686274508e-01   6.6666666666666663e-01
            7.0588235294117646e-02   3.8431372549019610e-01   6.7058823529411760e-01
            7.4509803921568626e-02   3.8823529411764707e-01   6.7450980392156867e-01
            7.8431372549019607e-02   3.9215686274509803e-01   6.7450980392156867e-01
            8.2352941176470587e-02   3.9607843137254900e-01   6.7843137254901964e-01
            8.6274509803921567e-02   4.0000000000000002e-01   6.8235294117647061e-01
            9.0196078431372548e-02   4.0392156862745099e-01   6.8235294117647061e-01
            9.4117647058823528e-02   4.0784313725490196e-01   6.8627450980392157e-01
            9.8039215686274508e-02   4.1176470588235292e-01   6.9019607843137254e-01
            1.0196078431372549e-01   4.1568627450980394e-01   6.9019607843137254e-01
            1.0588235294117647e-01   4.1960784313725491e-01   6.9411764705882351e-01
            1.0980392156862745e-01   4.2352941176470588e-01   6.9803921568627447e-01
            1.1372549019607843e-01   4.2745098039215684e-01   6.9803921568627447e-01
            1.1764705882352941e-01   4.3137254901960786e-01   7.0196078431372544e-01
            1.2156862745098039e-01   4.3529411764705883e-01   7.0588235294117652e-01
            1.2549019607843137e-01   4.3921568627450980e-01   7.0588235294117652e-01
            1.2941176470588237e-01   4.4313725490196076e-01   7.0980392156862748e-01
            1.3333333333333333e-01   4.4705882352941179e-01   7.1372549019607845e-01
            1.3725490196078433e-01   4.5098039215686275e-01   7.1372549019607845e-01
            1.4117647058823529e-01   4.5490196078431372e-01   7.1764705882352942e-01
            1.4509803921568629e-01   4.5882352941176469e-01   7.1764705882352942e-01
            1.4901960784313725e-01   4.6274509803921571e-01   7.2156862745098038e-01
            1.5294117647058825e-01   4.6666666666666667e-01   7.2549019607843135e-01
            1.5294117647058825e-01   4.7058823529411764e-01   7.2549019607843135e-01
            1.5686274509803921e-01   4.7450980392156861e-01   7.2941176470588232e-01
            1.6078431372549021e-01   4.7843137254901963e-01   7.2941176470588232e-01
            1.6470588235294117e-01   4.8235294117647060e-01   7.3333333333333328e-01
            1.6862745098039217e-01   4.8627450980392156e-01   7.3333333333333328e-01
            1.7254901960784313e-01   4.9019607843137253e-01   7.3725490196078436e-01
            1.7647058823529413e-01   4.9803921568627452e-01   7.3725490196078436e-01
            1.8039215686274510e-01   5.0196078431372548e-01   7.4117647058823533e-01
            1.8431372549019609e-01   5.0588235294117645e-01   7.4117647058823533e-01
            1.8823529411764706e-01   5.0980392156862742e-01   7.4509803921568629e-01
            1.9215686274509805e-01   5.1372549019607838e-01   7.4509803921568629e-01
            2.0000000000000001e-01   5.1764705882352946e-01   7.4901960784313726e-01
            2.0392156862745098e-01   5.2156862745098043e-01   7.4901960784313726e-01
            2.0784313725490197e-01   5.2549019607843139e-01   7.5294117647058822e-01
            2.1176470588235294e-01   5.2941176470588236e-01   7.5294117647058822e-01
            2.1568627450980393e-01   5.3333333333333333e-01   7.5686274509803919e-01
            2.1960784313725490e-01   5.3725490196078429e-01   7.5686274509803919e-01
            2.2352941176470589e-01   5.4117647058823526e-01   7.6078431372549016e-01
            2.2745098039215686e-01   5.4509803921568623e-01   7.6078431372549016e-01
            2.3137254901960785e-01   5.4901960784313730e-01   7.6470588235294112e-01
            2.3529411764705882e-01   5.5294117647058827e-01   7.6470588235294112e-01
            2.3921568627450981e-01   5.5686274509803924e-01   7.6862745098039220e-01
            2.4705882352941178e-01   5.6078431372549020e-01   7.6862745098039220e-01
            2.5098039215686274e-01   5.6470588235294117e-01   7.7254901960784317e-01
            2.5490196078431371e-01   5.6862745098039214e-01   7.7254901960784317e-01
            2.5882352941176473e-01   5.7254901960784310e-01   7.7647058823529413e-01
            2.6274509803921570e-01   5.7647058823529407e-01   7.8039215686274510e-01
            2.6666666666666666e-01   5.8039215686274515e-01   7.8039215686274510e-01
            2.7450980392156865e-01   5.8431372549019611e-01   7.8431372549019607e-01
            2.7843137254901962e-01   5.8823529411764708e-01   7.8431372549019607e-01
            2.8235294117647058e-01   5.9215686274509804e-01   7.8823529411764703e-01
            2.8627450980392155e-01   5.9607843137254901e-01   7.8823529411764703e-01
            2.9019607843137257e-01   5.9999999999999998e-01   7.9215686274509800e-01
            2.9803921568627451e-01   6.0392156862745094e-01   7.9215686274509800e-01
            3.0196078431372547e-01   6.0392156862745094e-01   7.9607843137254897e-01
            3.0588235294117649e-01   6.0784313725490191e-01   7.9607843137254897e-01
            3.0980392156862746e-01   6.1176470588235299e-01   8.0000000000000004e-01
            3.1764705882352939e-01   6.1568627450980395e-01   8.0392156862745101e-01
            3.2156862745098042e-01   6.1960784313725492e-01   8.0392156862745101e-01
            3.2549019607843138e-01   6.2352941176470589e-01   8.0784313725490198e-01
            3.3333333333333331e-01   6.2745098039215685e-01   8.0784313725490198e-01
            3.3725490196078434e-01   6.3137254901960782e-01   8.1176470588235294e-01
            3.4117647058823530e-01   6.3529411764705879e-01   8.1176470588235294e-01
            3.4901960784313724e-01   6.3921568627450975e-01   8.1568627450980391e-01
            3.5294117647058826e-01   6.3921568627450975e-01   8.1568627450980391e-01
            3.5686274509803922e-01   6.4313725490196083e-01   8.1960784313725488e-01
            3.6470588235294116e-01   6.4705882352941180e-01   8.1960784313725488e-01
            3.6862745098039218e-01   6.5098039215686276e-01   8.2352941176470584e-01
            3.7647058823529411e-01   6.5490196078431373e-01   8.2352941176470584e-01
            3.8039215686274508e-01   6.5882352941176470e-01   8.2745098039215681e-01
            3.8431372549019610e-01   6.6274509803921566e-01   8.2745098039215681e-01
            3.9215686274509803e-01   6.6666666666666663e-01   8.3137254901960789e-01
            3.9607843137254900e-01   6.6666666666666663e-01   8.3137254901960789e-01
            4.0392156862745099e-01   6.7058823529411760e-01   8.3529411764705885e-01
            4.0784313725490196e-01   6.7450980392156867e-01   8.3529411764705885e-01
            4.1568627450980394e-01   6.7843137254901964e-01   8.3921568627450982e-01
            4.1960784313725491e-01   6.8235294117647061e-01   8.3921568627450982e-01
            4.2352941176470588e-01   6.8627450980392157e-01   8.3921568627450982e-01
            4.3137254901960786e-01   6.9019607843137254e-01   8.4313725490196079e-01
            4.3529411764705883e-01   6.9411764705882351e-01   8.4313725490196079e-01
            4.4313725490196076e-01   6.9803921568627447e-01   8.4705882352941175e-01
            4.5098039215686275e-01   7.0196078431372544e-01   8.4705882352941175e-01
            4.5490196078431372e-01   7.0588235294117652e-01   8.4705882352941175e-01
            4.6274509803921571e-01   7.0588235294117652e-01   8.5098039215686272e-01
            4.6666666666666667e-01   7.0980392156862748e-01   8.5098039215686272e-01
            4.7450980392156861e-01   7.1372549019607845e-01   8.5098039215686272e-01
            4.8235294117647060e-01   7.1764705882352942e-01   8.5490196078431369e-01
            4.8627450980392156e-01   7.2156862745098038e-01   8.5490196078431369e-01
            4.9411764705882355e-01   7.2549019607843135e-01   8.5490196078431369e-01
            5.0196078431372548e-01   7.2941176470588232e-01   8.5882352941176465e-01
            5.0588235294117645e-01   7.3333333333333328e-01   8.5882352941176465e-01
            5.1372549019607838e-01   7.3725490196078436e-01   8.5882352941176465e-01
            5.2156862745098043e-01   7.4117647058823533e-01   8.6274509803921573e-01
            5.2549019607843139e-01   7.4509803921568629e-01   8.6274509803921573e-01
            5.3333333333333333e-01   7.4901960784313726e-01   8.6274509803921573e-01
            5.4117647058823526e-01   7.5294117647058822e-01   8.6666666666666670e-01
            5.4509803921568623e-01   7.5686274509803919e-01   8.6666666666666670e-01
            5.5294117647058827e-01   7.5686274509803919e-01   8.6666666666666670e-01
            5.6078431372549020e-01   7.6078431372549016e-01   8.7058823529411766e-01
            5.6470588235294117e-01   7.6470588235294112e-01   8.7058823529411766e-01
            5.7254901960784310e-01   7.6862745098039220e-01   8.7058823529411766e-01
            5.7647058823529407e-01   7.7254901960784317e-01   8.7450980392156863e-01
            5.8431372549019611e-01   7.7647058823529413e-01   8.7450980392156863e-01
            5.9215686274509804e-01   7.7647058823529413e-01   8.7450980392156863e-01
            5.9607843137254901e-01   7.8039215686274510e-01   8.7843137254901960e-01
            6.0392156862745094e-01   7.8431372549019607e-01   8.7843137254901960e-01
            6.0784313725490191e-01   7.8823529411764703e-01   8.7843137254901960e-01
            6.1568627450980395e-01   7.8823529411764703e-01   8.8235294117647056e-01
            6.1960784313725492e-01   7.9215686274509800e-01   8.8235294117647056e-01
            6.2352941176470589e-01   7.9607843137254897e-01   8.8235294117647056e-01
            6.3137254901960782e-01   7.9607843137254897e-01   8.8627450980392153e-01
            6.3529411764705879e-01   8.0000000000000004e-01   8.8627450980392153e-01
            6.4313725490196083e-01   8.0392156862745101e-01   8.9019607843137250e-01
            6.4705882352941180e-01   8.0392156862745101e-01   8.9019607843137250e-01
            6.5098039215686276e-01   8.0784313725490198e-01   8.9411764705882357e-01
            6.5882352941176470e-01   8.0784313725490198e-01   8.9411764705882357e-01
            6.6274509803921566e-01   8.1176470588235294e-01   8.9411764705882357e-01
            6.7058823529411760e-01   8.1176470588235294e-01   8.9803921568627454e-01
            6.7450980392156867e-01   8.1568627450980391e-01   8.9803921568627454e-01
            6.7843137254901964e-01   8.1568627450980391e-01   9.0196078431372551e-01
            6.8627450980392157e-01   8.1960784313725488e-01   9.0196078431372551e-01
            6.9019607843137254e-01   8.2352941176470584e-01   9.0588235294117647e-01
            6.9411764705882351e-01   8.2352941176470584e-01   9.0588235294117647e-01
            7.0196078431372544e-01   8.2745098039215681e-01   9.0980392156862744e-01
            7.0588235294117652e-01   8.2745098039215681e-01   9.0980392156862744e-01
            7.0980392156862748e-01   8.3137254901960789e-01   9.1372549019607840e-01
            7.1372549019607845e-01   8.3137254901960789e-01   9.1372549019607840e-01
            7.2156862745098038e-01   8.3529411764705885e-01   9.1764705882352937e-01
            7.2549019607843135e-01   8.3529411764705885e-01   9.1764705882352937e-01
            7.2941176470588232e-01   8.3921568627450982e-01   9.2156862745098034e-01
            7.3333333333333328e-01   8.3921568627450982e-01   9.2156862745098034e-01
            7.3725490196078436e-01   8.4313725490196079e-01   9.2549019607843142e-01
            7.4509803921568629e-01   8.4313725490196079e-01   9.2549019607843142e-01
            7.4901960784313726e-01   8.4705882352941175e-01   9.2549019607843142e-01
            7.5294117647058822e-01   8.4705882352941175e-01   9.2941176470588238e-01
            7.5686274509803919e-01   8.4705882352941175e-01   9.2941176470588238e-01
            7.6078431372549016e-01   8.5098039215686272e-01   9.3333333333333335e-01
            7.6470588235294112e-01   8.5098039215686272e-01   9.3333333333333335e-01
            7.6862745098039220e-01   8.5490196078431369e-01   9.3333333333333335e-01
            7.7254901960784317e-01   8.5490196078431369e-01   9.3725490196078431e-01
            7.7647058823529413e-01   8.5882352941176465e-01   9.3725490196078431e-01
            7.8039215686274510e-01   8.6274509803921573e-01   9.3725490196078431e-01
            7.8431372549019607e-01   8.6274509803921573e-01   9.4117647058823528e-01
            7.8823529411764703e-01   8.6666666666666670e-01   9.4117647058823528e-01
            7.9215686274509800e-01   8.6666666666666670e-01   9.4117647058823528e-01
            7.9215686274509800e-01   8.7058823529411766e-01   9.4117647058823528e-01
            7.9607843137254897e-01   8.7058823529411766e-01   9.4509803921568625e-01
            8.0000000000000004e-01   8.7450980392156863e-01   9.4509803921568625e-01
            8.0392156862745101e-01   8.7450980392156863e-01   9.4509803921568625e-01
            8.0784313725490198e-01   8.7843137254901960e-01   9.4901960784313721e-01
            8.0784313725490198e-01   8.7843137254901960e-01   9.4901960784313721e-01
            8.1176470588235294e-01   8.8235294117647056e-01   9.4901960784313721e-01
            8.1568627450980391e-01   8.8235294117647056e-01   9.4901960784313721e-01
            8.1960784313725488e-01   8.8627450980392153e-01   9.5294117647058818e-01
            8.1960784313725488e-01   8.8627450980392153e-01   9.5294117647058818e-01
            8.2352941176470584e-01   8.9019607843137250e-01   9.5294117647058818e-01
            8.2745098039215681e-01   8.9019607843137250e-01   9.5294117647058818e-01
            8.2745098039215681e-01   8.9411764705882357e-01   9.5686274509803926e-01
            8.3137254901960789e-01   8.9411764705882357e-01   9.5686274509803926e-01
            8.3529411764705885e-01   8.9803921568627454e-01   9.5686274509803926e-01
            8.3529411764705885e-01   8.9803921568627454e-01   9.5686274509803926e-01
            8.3921568627450982e-01   9.0196078431372551e-01   9.5686274509803926e-01
            8.4313725490196079e-01   9.0196078431372551e-01   9.6078431372549022e-01
            8.4705882352941175e-01   9.0588235294117647e-01   9.6078431372549022e-01
            8.4705882352941175e-01   9.0588235294117647e-01   9.6078431372549022e-01
            8.5098039215686272e-01   9.0980392156862744e-01   9.6078431372549022e-01
            8.5490196078431369e-01   9.0980392156862744e-01   9.6470588235294119e-01
            8.5490196078431369e-01   9.1372549019607840e-01   9.6470588235294119e-01
            8.5882352941176465e-01   9.1372549019607840e-01   9.6470588235294119e-01
            8.6274509803921573e-01   9.1764705882352937e-01   9.6470588235294119e-01
            8.6274509803921573e-01   9.1764705882352937e-01   9.6862745098039216e-01
            8.6666666666666670e-01   9.2156862745098034e-01   9.6862745098039216e-01
            8.7058823529411766e-01   9.2156862745098034e-01   9.6862745098039216e-01
            8.7450980392156863e-01   9.2549019607843142e-01   9.6862745098039216e-01
            8.7843137254901960e-01   9.2549019607843142e-01   9.7254901960784312e-01
            8.7843137254901960e-01   9.2941176470588238e-01   9.7254901960784312e-01
            8.8235294117647056e-01   9.2941176470588238e-01   9.7254901960784312e-01
            8.8627450980392153e-01   9.3333333333333335e-01   9.7254901960784312e-01
            8.9019607843137250e-01   9.3333333333333335e-01   9.7647058823529409e-01
            8.9019607843137250e-01   9.3725490196078431e-01   9.7647058823529409e-01
            8.9411764705882357e-01   9.3725490196078431e-01   9.7647058823529409e-01
            8.9803921568627454e-01   9.4117647058823528e-01   9.7647058823529409e-01
            9.0196078431372551e-01   9.4117647058823528e-01   9.8039215686274506e-01
            9.0196078431372551e-01   9.4509803921568625e-01   9.8039215686274506e-01
            9.0588235294117647e-01   9.4509803921568625e-01   9.8039215686274506e-01
            9.0980392156862744e-01   9.4901960784313721e-01   9.8039215686274506e-01
            9.1372549019607840e-01   9.4901960784313721e-01   9.8431372549019602e-01
            9.1764705882352937e-01   9.5294117647058818e-01   9.8431372549019602e-01
            9.1764705882352937e-01   9.5294117647058818e-01   9.8431372549019602e-01
            9.2156862745098034e-01   9.5686274509803926e-01   9.8431372549019602e-01
            9.2549019607843142e-01   9.5686274509803926e-01   9.8823529411764710e-01
            9.2941176470588238e-01   9.6078431372549022e-01   9.8823529411764710e-01
            9.3333333333333335e-01   9.6078431372549022e-01   9.8823529411764710e-01
            9.3333333333333335e-01   9.6470588235294119e-01   9.8823529411764710e-01
            9.3725490196078431e-01   9.6470588235294119e-01   9.9215686274509807e-01
            9.4117647058823528e-01   9.6862745098039216e-01   9.9215686274509807e-01
            9.4509803921568625e-01   9.6862745098039216e-01   9.9215686274509807e-01
            9.4509803921568625e-01   9.7254901960784312e-01   9.9215686274509807e-01
            9.4901960784313721e-01   9.7254901960784312e-01   9.9607843137254903e-01
            9.5294117647058818e-01   9.7647058823529409e-01   9.9607843137254903e-01
            9.5686274509803926e-01   9.7647058823529409e-01   9.9607843137254903e-01
            9.6078431372549022e-01   9.8039215686274506e-01   9.9607843137254903e-01
            9.6078431372549022e-01   9.8039215686274506e-01   1.0000000000000000e+00
            9.6470588235294119e-01   9.8431372549019602e-01   1.0000000000000000e+00
            9.6862745098039216e-01   9.8431372549019602e-01   1.0000000000000000e+00];
    case 'green'
        Data=[...
            0.0000000000000000e+00   2.6666666666666666e-01   1.0588235294117647e-01
            0.0000000000000000e+00   2.7058823529411763e-01   1.0588235294117647e-01
            0.0000000000000000e+00   2.7843137254901962e-01   1.0980392156862745e-01
            0.0000000000000000e+00   2.8235294117647058e-01   1.0980392156862745e-01
            0.0000000000000000e+00   2.9019607843137257e-01   1.1372549019607843e-01
            0.0000000000000000e+00   2.9411764705882354e-01   1.1372549019607843e-01
            0.0000000000000000e+00   3.0196078431372547e-01   1.1764705882352941e-01
            0.0000000000000000e+00   3.0588235294117649e-01   1.1764705882352941e-01
            0.0000000000000000e+00   3.0980392156862746e-01   1.2156862745098039e-01
            0.0000000000000000e+00   3.1764705882352939e-01   1.2156862745098039e-01
            0.0000000000000000e+00   3.2156862745098042e-01   1.2549019607843137e-01
            0.0000000000000000e+00   3.2549019607843138e-01   1.2549019607843137e-01
            0.0000000000000000e+00   3.3333333333333331e-01   1.2941176470588237e-01
            0.0000000000000000e+00   3.3725490196078434e-01   1.2941176470588237e-01
            0.0000000000000000e+00   3.4117647058823530e-01   1.3333333333333333e-01
            0.0000000000000000e+00   3.4901960784313724e-01   1.3333333333333333e-01
            0.0000000000000000e+00   3.5294117647058826e-01   1.3725490196078433e-01
            0.0000000000000000e+00   3.5686274509803922e-01   1.3725490196078433e-01
            0.0000000000000000e+00   3.6470588235294116e-01   1.4117647058823529e-01
            0.0000000000000000e+00   3.6862745098039218e-01   1.4117647058823529e-01
            0.0000000000000000e+00   3.7254901960784315e-01   1.4509803921568629e-01
            0.0000000000000000e+00   3.7647058823529411e-01   1.4509803921568629e-01
            0.0000000000000000e+00   3.8039215686274508e-01   1.4901960784313725e-01
            0.0000000000000000e+00   3.8823529411764707e-01   1.5294117647058825e-01
            0.0000000000000000e+00   3.9215686274509803e-01   1.5294117647058825e-01
            0.0000000000000000e+00   3.9607843137254900e-01   1.5686274509803921e-01
            0.0000000000000000e+00   4.0000000000000002e-01   1.5686274509803921e-01
            0.0000000000000000e+00   4.0392156862745099e-01   1.6078431372549021e-01
            0.0000000000000000e+00   4.1176470588235292e-01   1.6470588235294117e-01
            0.0000000000000000e+00   4.1568627450980394e-01   1.6470588235294117e-01
            0.0000000000000000e+00   4.1960784313725491e-01   1.6862745098039217e-01
            0.0000000000000000e+00   4.2352941176470588e-01   1.6862745098039217e-01
            0.0000000000000000e+00   4.2745098039215684e-01   1.7254901960784313e-01
            0.0000000000000000e+00   4.3137254901960786e-01   1.7647058823529413e-01
            0.0000000000000000e+00   4.3529411764705883e-01   1.7647058823529413e-01
            3.9215686274509803e-03   4.3921568627450980e-01   1.8039215686274510e-01
            3.9215686274509803e-03   4.4313725490196076e-01   1.8431372549019609e-01
            7.8431372549019607e-03   4.4705882352941179e-01   1.8431372549019609e-01
            7.8431372549019607e-03   4.5098039215686275e-01   1.8823529411764706e-01
            1.1764705882352941e-02   4.5490196078431372e-01   1.9215686274509805e-01
            1.5686274509803921e-02   4.5882352941176469e-01   1.9607843137254902e-01
            1.9607843137254902e-02   4.6274509803921571e-01   1.9607843137254902e-01
            2.3529411764705882e-02   4.6666666666666667e-01   2.0000000000000001e-01
            2.7450980392156862e-02   4.7058823529411764e-01   2.0392156862745098e-01
            3.1372549019607843e-02   4.7450980392156861e-01   2.0784313725490197e-01
            3.5294117647058823e-02   4.7843137254901963e-01   2.1176470588235294e-01
            4.3137254901960784e-02   4.8235294117647060e-01   2.1176470588235294e-01
            4.7058823529411764e-02   4.8627450980392156e-01   2.1568627450980393e-01
            5.0980392156862744e-02   4.8627450980392156e-01   2.1960784313725490e-01
            5.8823529411764705e-02   4.9019607843137253e-01   2.2352941176470589e-01
            6.2745098039215685e-02   4.9411764705882355e-01   2.2745098039215686e-01
            7.0588235294117646e-02   4.9803921568627452e-01   2.2745098039215686e-01
            7.4509803921568626e-02   5.0196078431372548e-01   2.3137254901960785e-01
            8.2352941176470587e-02   5.0588235294117645e-01   2.3529411764705882e-01
            8.6274509803921567e-02   5.0980392156862742e-01   2.3921568627450981e-01
            9.4117647058823528e-02   5.1372549019607838e-01   2.4313725490196078e-01
            9.8039215686274508e-02   5.1764705882352946e-01   2.4705882352941178e-01
            1.0196078431372549e-01   5.1764705882352946e-01   2.4705882352941178e-01
            1.0980392156862745e-01   5.2156862745098043e-01   2.5098039215686274e-01
            1.1372549019607843e-01   5.2549019607843139e-01   2.5490196078431371e-01
            1.1764705882352941e-01   5.2941176470588236e-01   2.5882352941176473e-01
            1.2549019607843137e-01   5.3333333333333333e-01   2.6274509803921570e-01
            1.2941176470588237e-01   5.3725490196078429e-01   2.6274509803921570e-01
            1.3333333333333333e-01   5.4117647058823526e-01   2.6666666666666666e-01
            1.3725490196078433e-01   5.4509803921568623e-01   2.7058823529411763e-01
            1.4117647058823529e-01   5.4901960784313730e-01   2.7450980392156865e-01
            1.4509803921568629e-01   5.5294117647058827e-01   2.7843137254901962e-01
            1.4901960784313725e-01   5.5686274509803924e-01   2.7843137254901962e-01
            1.5294117647058825e-01   5.6078431372549020e-01   2.8235294117647058e-01
            1.5686274509803921e-01   5.6470588235294117e-01   2.8627450980392155e-01
            1.6078431372549021e-01   5.6862745098039214e-01   2.9019607843137257e-01
            1.6470588235294117e-01   5.7254901960784310e-01   2.9019607843137257e-01
            1.6470588235294117e-01   5.7647058823529407e-01   2.9411764705882354e-01
            1.6862745098039217e-01   5.8039215686274515e-01   2.9803921568627451e-01
            1.7254901960784313e-01   5.8431372549019611e-01   3.0196078431372547e-01
            1.7647058823529413e-01   5.8823529411764708e-01   3.0196078431372547e-01
            1.8039215686274510e-01   5.9215686274509804e-01   3.0588235294117649e-01
            1.8431372549019609e-01   5.9607843137254901e-01   3.0980392156862746e-01
            1.8823529411764706e-01   5.9999999999999998e-01   3.0980392156862746e-01
            1.8823529411764706e-01   6.0392156862745094e-01   3.1372549019607843e-01
            1.9215686274509805e-01   6.0784313725490191e-01   3.1764705882352939e-01
            1.9607843137254902e-01   6.1176470588235299e-01   3.2156862745098042e-01
            2.0000000000000001e-01   6.1568627450980395e-01   3.2156862745098042e-01
            2.0392156862745098e-01   6.1960784313725492e-01   3.2549019607843138e-01
            2.0784313725490197e-01   6.2352941176470589e-01   3.2941176470588235e-01
            2.1176470588235294e-01   6.2745098039215685e-01   3.3333333333333331e-01
            2.1568627450980393e-01   6.3137254901960782e-01   3.3333333333333331e-01
            2.1568627450980393e-01   6.3529411764705879e-01   3.3725490196078434e-01
            2.1960784313725490e-01   6.3921568627450975e-01   3.4117647058823530e-01
            2.2352941176470589e-01   6.4313725490196083e-01   3.4509803921568627e-01
            2.2745098039215686e-01   6.4705882352941180e-01   3.4509803921568627e-01
            2.3137254901960785e-01   6.5098039215686276e-01   3.4901960784313724e-01
            2.3529411764705882e-01   6.5490196078431373e-01   3.5294117647058826e-01
            2.4313725490196078e-01   6.5882352941176470e-01   3.5686274509803922e-01
            2.4705882352941178e-01   6.6274509803921566e-01   3.5686274509803922e-01
            2.5098039215686274e-01   6.6666666666666663e-01   3.6078431372549019e-01
            2.5490196078431371e-01   6.7058823529411760e-01   3.6470588235294116e-01
            2.5882352941176473e-01   6.7450980392156867e-01   3.6862745098039218e-01
            2.6666666666666666e-01   6.7843137254901964e-01   3.7254901960784315e-01
            2.7058823529411763e-01   6.8235294117647061e-01   3.7254901960784315e-01
            2.7450980392156865e-01   6.8627450980392157e-01   3.7647058823529411e-01
            2.8235294117647058e-01   6.8627450980392157e-01   3.8039215686274508e-01
            2.8627450980392155e-01   6.9019607843137254e-01   3.8431372549019610e-01
            2.9411764705882354e-01   6.9411764705882351e-01   3.8431372549019610e-01
            2.9803921568627451e-01   6.9803921568627447e-01   3.8823529411764707e-01
            3.0588235294117649e-01   7.0196078431372544e-01   3.9215686274509803e-01
            3.1372549019607843e-01   7.0588235294117652e-01   3.9607843137254900e-01
            3.1764705882352939e-01   7.0588235294117652e-01   3.9607843137254900e-01
            3.2549019607843138e-01   7.0980392156862748e-01   4.0000000000000002e-01
            3.3333333333333331e-01   7.1372549019607845e-01   4.0392156862745099e-01
            3.4117647058823530e-01   7.1764705882352942e-01   4.0784313725490196e-01
            3.4509803921568627e-01   7.2156862745098038e-01   4.0784313725490196e-01
            3.5294117647058826e-01   7.2549019607843135e-01   4.1176470588235292e-01
            3.6078431372549019e-01   7.2549019607843135e-01   4.1568627450980394e-01
            3.6862745098039218e-01   7.2941176470588232e-01   4.1960784313725491e-01
            3.7647058823529411e-01   7.3333333333333328e-01   4.2352941176470588e-01
            3.8039215686274508e-01   7.3725490196078436e-01   4.2352941176470588e-01
            3.8823529411764707e-01   7.4117647058823533e-01   4.2745098039215684e-01
            3.9607843137254900e-01   7.4117647058823533e-01   4.3137254901960786e-01
            4.0392156862745099e-01   7.4509803921568629e-01   4.3529411764705883e-01
            4.0784313725490196e-01   7.4901960784313726e-01   4.3921568627450980e-01
            4.1568627450980394e-01   7.5294117647058822e-01   4.4313725490196076e-01
            4.2352941176470588e-01   7.5294117647058822e-01   4.4313725490196076e-01
            4.3137254901960786e-01   7.5686274509803919e-01   4.4705882352941179e-01
            4.3529411764705883e-01   7.6078431372549016e-01   4.5098039215686275e-01
            4.4313725490196076e-01   7.6470588235294112e-01   4.5490196078431372e-01
            4.4705882352941179e-01   7.6470588235294112e-01   4.5882352941176469e-01
            4.5490196078431372e-01   7.6862745098039220e-01   4.6274509803921571e-01
            4.6274509803921571e-01   7.7254901960784317e-01   4.6666666666666667e-01
            4.6666666666666667e-01   7.7254901960784317e-01   4.7058823529411764e-01
            4.7450980392156861e-01   7.7647058823529413e-01   4.7450980392156861e-01
            4.7843137254901963e-01   7.8039215686274510e-01   4.7843137254901963e-01
            4.8627450980392156e-01   7.8431372549019607e-01   4.8235294117647060e-01
            4.9019607843137253e-01   7.8431372549019607e-01   4.8627450980392156e-01
            4.9411764705882355e-01   7.8823529411764703e-01   4.9019607843137253e-01
            5.0196078431372548e-01   7.9215686274509800e-01   4.9411764705882355e-01
            5.0588235294117645e-01   7.9215686274509800e-01   4.9803921568627452e-01
            5.1372549019607838e-01   7.9607843137254897e-01   5.0588235294117645e-01
            5.1764705882352946e-01   8.0000000000000004e-01   5.0980392156862742e-01
            5.2549019607843139e-01   8.0000000000000004e-01   5.1372549019607838e-01
            5.2941176470588236e-01   8.0392156862745101e-01   5.1764705882352946e-01
            5.3725490196078429e-01   8.0784313725490198e-01   5.2156862745098043e-01
            5.4117647058823526e-01   8.0784313725490198e-01   5.2549019607843139e-01
            5.4509803921568623e-01   8.1176470588235294e-01   5.3333333333333333e-01
            5.5294117647058827e-01   8.1568627450980391e-01   5.3725490196078429e-01
            5.5686274509803924e-01   8.1568627450980391e-01   5.4117647058823526e-01
            5.6470588235294117e-01   8.1960784313725488e-01   5.4509803921568623e-01
            5.6862745098039214e-01   8.2352941176470584e-01   5.5294117647058827e-01
            5.7254901960784310e-01   8.2352941176470584e-01   5.5686274509803924e-01
            5.8039215686274515e-01   8.2745098039215681e-01   5.6078431372549020e-01
            5.8431372549019611e-01   8.3137254901960789e-01   5.6470588235294117e-01
            5.8823529411764708e-01   8.3137254901960789e-01   5.6862745098039214e-01
            5.9607843137254901e-01   8.3529411764705885e-01   5.7647058823529407e-01
            5.9999999999999998e-01   8.3529411764705885e-01   5.8039215686274515e-01
            6.0392156862745094e-01   8.3921568627450982e-01   5.8431372549019611e-01
            6.1176470588235299e-01   8.4313725490196079e-01   5.8823529411764708e-01
            6.1568627450980395e-01   8.4313725490196079e-01   5.9215686274509804e-01
            6.1960784313725492e-01   8.4705882352941175e-01   5.9999999999999998e-01
            6.2745098039215685e-01   8.4705882352941175e-01   6.0392156862745094e-01
            6.3137254901960782e-01   8.5098039215686272e-01   6.0784313725490191e-01
            6.3529411764705879e-01   8.5490196078431369e-01   6.1176470588235299e-01
            6.4313725490196083e-01   8.5490196078431369e-01   6.1568627450980395e-01
            6.4705882352941180e-01   8.5882352941176465e-01   6.1960784313725492e-01
            6.5098039215686276e-01   8.5882352941176465e-01   6.2745098039215685e-01
            6.5490196078431373e-01   8.6274509803921573e-01   6.3137254901960782e-01
            6.6274509803921566e-01   8.6274509803921573e-01   6.3529411764705879e-01
            6.6666666666666663e-01   8.6666666666666670e-01   6.3921568627450975e-01
            6.7058823529411760e-01   8.6666666666666670e-01   6.4313725490196083e-01
            6.7450980392156867e-01   8.7058823529411766e-01   6.5098039215686276e-01
            6.8235294117647061e-01   8.7058823529411766e-01   6.5490196078431373e-01
            6.8627450980392157e-01   8.7450980392156863e-01   6.5882352941176470e-01
            6.9019607843137254e-01   8.7843137254901960e-01   6.6274509803921566e-01
            6.9411764705882351e-01   8.7843137254901960e-01   6.6666666666666663e-01
            7.0196078431372544e-01   8.8235294117647056e-01   6.7058823529411760e-01
            7.0588235294117652e-01   8.8235294117647056e-01   6.7843137254901964e-01
            7.0980392156862748e-01   8.8627450980392153e-01   6.8235294117647061e-01
            7.1372549019607845e-01   8.8627450980392153e-01   6.8627450980392157e-01
            7.1764705882352942e-01   8.9019607843137250e-01   6.9019607843137254e-01
            7.2549019607843135e-01   8.9019607843137250e-01   6.9411764705882351e-01
            7.2941176470588232e-01   8.9411764705882357e-01   7.0196078431372544e-01
            7.3333333333333328e-01   8.9411764705882357e-01   7.0588235294117652e-01
            7.3725490196078436e-01   8.9411764705882357e-01   7.0980392156862748e-01
            7.4117647058823533e-01   8.9803921568627454e-01   7.1372549019607845e-01
            7.4509803921568629e-01   8.9803921568627454e-01   7.1764705882352942e-01
            7.4901960784313726e-01   9.0196078431372551e-01   7.2156862745098038e-01
            7.5686274509803919e-01   9.0196078431372551e-01   7.2549019607843135e-01
            7.6078431372549016e-01   9.0588235294117647e-01   7.3333333333333328e-01
            7.6470588235294112e-01   9.0588235294117647e-01   7.3725490196078436e-01
            7.6862745098039220e-01   9.0980392156862744e-01   7.4117647058823533e-01
            7.7254901960784317e-01   9.0980392156862744e-01   7.4509803921568629e-01
            7.7647058823529413e-01   9.1372549019607840e-01   7.4901960784313726e-01
            7.8039215686274510e-01   9.1372549019607840e-01   7.5294117647058822e-01
            7.8431372549019607e-01   9.1372549019607840e-01   7.5686274509803919e-01
            7.8823529411764703e-01   9.1764705882352937e-01   7.6078431372549016e-01
            7.9215686274509800e-01   9.1764705882352937e-01   7.6470588235294112e-01
            7.9607843137254897e-01   9.2156862745098034e-01   7.6862745098039220e-01
            8.0000000000000004e-01   9.2156862745098034e-01   7.7254901960784317e-01
            8.0392156862745101e-01   9.2549019607843142e-01   7.7647058823529413e-01
            8.0784313725490198e-01   9.2549019607843142e-01   7.8431372549019607e-01
            8.1176470588235294e-01   9.2549019607843142e-01   7.8823529411764703e-01
            8.1568627450980391e-01   9.2941176470588238e-01   7.9215686274509800e-01
            8.1960784313725488e-01   9.2941176470588238e-01   7.9607843137254897e-01
            8.2352941176470584e-01   9.3333333333333335e-01   8.0000000000000004e-01
            8.2745098039215681e-01   9.3333333333333335e-01   8.0392156862745101e-01
            8.3137254901960789e-01   9.3333333333333335e-01   8.0784313725490198e-01
            8.3529411764705885e-01   9.3725490196078431e-01   8.1176470588235294e-01
            8.3921568627450982e-01   9.3725490196078431e-01   8.1568627450980391e-01
            8.4313725490196079e-01   9.4117647058823528e-01   8.1960784313725488e-01
            8.4705882352941175e-01   9.4117647058823528e-01   8.2352941176470584e-01
            8.5098039215686272e-01   9.4117647058823528e-01   8.2745098039215681e-01
            8.5490196078431369e-01   9.4509803921568625e-01   8.3137254901960789e-01
            8.5882352941176465e-01   9.4509803921568625e-01   8.3529411764705885e-01
            8.6274509803921573e-01   9.4509803921568625e-01   8.3921568627450982e-01
            8.6666666666666670e-01   9.4901960784313721e-01   8.4313725490196079e-01
            8.7058823529411766e-01   9.4901960784313721e-01   8.4705882352941175e-01
            8.7450980392156863e-01   9.5294117647058818e-01   8.5098039215686272e-01
            8.7843137254901960e-01   9.5294117647058818e-01   8.5490196078431369e-01
            8.7843137254901960e-01   9.5294117647058818e-01   8.5882352941176465e-01
            8.8235294117647056e-01   9.5686274509803926e-01   8.6274509803921573e-01
            8.8627450980392153e-01   9.5686274509803926e-01   8.6666666666666670e-01
            8.9019607843137250e-01   9.5686274509803926e-01   8.7058823529411766e-01
            8.9411764705882357e-01   9.5686274509803926e-01   8.7058823529411766e-01
            8.9411764705882357e-01   9.6078431372549022e-01   8.7450980392156863e-01
            8.9803921568627454e-01   9.6078431372549022e-01   8.7843137254901960e-01
            9.0196078431372551e-01   9.6078431372549022e-01   8.8235294117647056e-01
            9.0196078431372551e-01   9.6470588235294119e-01   8.8627450980392153e-01
            9.0588235294117647e-01   9.6470588235294119e-01   8.8627450980392153e-01
            9.0980392156862744e-01   9.6470588235294119e-01   8.9019607843137250e-01
            9.0980392156862744e-01   9.6470588235294119e-01   8.9411764705882357e-01
            9.1372549019607840e-01   9.6862745098039216e-01   8.9803921568627454e-01
            9.1764705882352937e-01   9.6862745098039216e-01   8.9803921568627454e-01
            9.1764705882352937e-01   9.6862745098039216e-01   9.0196078431372551e-01
            9.2156862745098034e-01   9.6862745098039216e-01   9.0588235294117647e-01
            9.2549019607843142e-01   9.7254901960784312e-01   9.0980392156862744e-01
            9.2549019607843142e-01   9.7254901960784312e-01   9.0980392156862744e-01
            9.2941176470588238e-01   9.7254901960784312e-01   9.1372549019607840e-01
            9.3333333333333335e-01   9.7254901960784312e-01   9.1764705882352937e-01
            9.3333333333333335e-01   9.7647058823529409e-01   9.1764705882352937e-01
            9.3725490196078431e-01   9.7647058823529409e-01   9.2156862745098034e-01
            9.3725490196078431e-01   9.7647058823529409e-01   9.2549019607843142e-01
            9.4117647058823528e-01   9.7647058823529409e-01   9.2549019607843142e-01
            9.4117647058823528e-01   9.7647058823529409e-01   9.2941176470588238e-01
            9.4509803921568625e-01   9.8039215686274506e-01   9.3333333333333335e-01
            9.4901960784313721e-01   9.8039215686274506e-01   9.3333333333333335e-01
            9.4901960784313721e-01   9.8039215686274506e-01   9.3725490196078431e-01
            9.5294117647058818e-01   9.8039215686274506e-01   9.4117647058823528e-01
            9.5294117647058818e-01   9.8039215686274506e-01   9.4117647058823528e-01
            9.5686274509803926e-01   9.8431372549019602e-01   9.4509803921568625e-01
            9.5686274509803926e-01   9.8431372549019602e-01   9.4509803921568625e-01
            9.5686274509803926e-01   9.8431372549019602e-01   9.4901960784313721e-01
            9.6078431372549022e-01   9.8431372549019602e-01   9.4901960784313721e-01
            9.6078431372549022e-01   9.8431372549019602e-01   9.5294117647058818e-01
            9.6470588235294119e-01   9.8823529411764710e-01   9.5294117647058818e-01
            9.6470588235294119e-01   9.8823529411764710e-01   9.5686274509803926e-01
            9.6862745098039216e-01   9.8823529411764710e-01   9.6078431372549022e-01
            9.6862745098039216e-01   9.8823529411764710e-01   9.6078431372549022e-01 ];
    case 'diverging'
        Data=flipud([...
            4.0392156862745099e-01   0.0000000000000000e+00   1.2156862745098039e-01
            4.1960784313725491e-01   0.0000000000000000e+00   1.2156862745098039e-01
            4.3137254901960786e-01   0.0000000000000000e+00   1.2156862745098039e-01
            4.4705882352941179e-01   0.0000000000000000e+00   1.2156862745098039e-01
            4.5882352941176469e-01   3.9215686274509803e-03   1.2156862745098039e-01
            4.7450980392156861e-01   3.9215686274509803e-03   1.2549019607843137e-01
            4.8627450980392156e-01   7.8431372549019607e-03   1.2549019607843137e-01
            4.9803921568627452e-01   7.8431372549019607e-03   1.2549019607843137e-01
            5.1372549019607838e-01   1.1764705882352941e-02   1.2941176470588237e-01
            5.2549019607843139e-01   1.5686274509803921e-02   1.2941176470588237e-01
            5.3725490196078429e-01   1.9607843137254902e-02   1.2941176470588237e-01
            5.4901960784313730e-01   1.9607843137254902e-02   1.3333333333333333e-01
            5.6078431372549020e-01   2.3529411764705882e-02   1.3333333333333333e-01
            5.7254901960784310e-01   2.7450980392156862e-02   1.3725490196078433e-01
            5.8431372549019611e-01   3.5294117647058823e-02   1.3725490196078433e-01
            5.9607843137254901e-01   3.9215686274509803e-02   1.4117647058823529e-01
            6.0784313725490191e-01   4.3137254901960784e-02   1.4117647058823529e-01
            6.1568627450980395e-01   4.7058823529411764e-02   1.4509803921568629e-01
            6.2745098039215685e-01   5.0980392156862744e-02   1.4901960784313725e-01
            6.3529411764705879e-01   5.4901960784313725e-02   1.4901960784313725e-01
            6.4705882352941180e-01   6.2745098039215685e-02   1.5294117647058825e-01
            6.5490196078431373e-01   6.6666666666666666e-02   1.5686274509803921e-01
            6.6666666666666663e-01   7.0588235294117646e-02   1.5686274509803921e-01
            6.7450980392156867e-01   7.8431372549019607e-02   1.6078431372549021e-01
            6.8235294117647061e-01   8.2352941176470587e-02   1.6470588235294117e-01
            6.9019607843137254e-01   9.0196078431372548e-02   1.6470588235294117e-01
            6.9803921568627447e-01   9.4117647058823528e-02   1.6862745098039217e-01
            7.0588235294117652e-01   1.0196078431372549e-01   1.7254901960784313e-01
            7.1372549019607845e-01   1.0588235294117647e-01   1.7647058823529413e-01
            7.1764705882352942e-01   1.1372549019607843e-01   1.7647058823529413e-01
            7.2549019607843135e-01   1.2156862745098039e-01   1.8039215686274510e-01
            7.3333333333333328e-01   1.3333333333333333e-01   1.8431372549019609e-01
            7.3725490196078436e-01   1.4117647058823529e-01   1.9215686274509805e-01
            7.4509803921568629e-01   1.5294117647058825e-01   1.9607843137254902e-01
            7.5294117647058822e-01   1.6470588235294117e-01   2.0000000000000001e-01
            7.5686274509803919e-01   1.7647058823529413e-01   2.0392156862745098e-01
            7.6078431372549016e-01   1.8823529411764706e-01   2.1176470588235294e-01
            7.6862745098039220e-01   2.0000000000000001e-01   2.1568627450980393e-01
            7.7254901960784317e-01   2.1176470588235294e-01   2.1960784313725490e-01
            7.8039215686274510e-01   2.2352941176470589e-01   2.2745098039215686e-01
            7.8431372549019607e-01   2.3921568627450981e-01   2.3137254901960785e-01
            7.8823529411764703e-01   2.5098039215686274e-01   2.3921568627450981e-01
            7.9607843137254897e-01   2.6274509803921570e-01   2.4313725490196078e-01
            8.0000000000000004e-01   2.7843137254901962e-01   2.5098039215686274e-01
            8.0392156862745101e-01   2.9019607843137257e-01   2.5882352941176473e-01
            8.0784313725490198e-01   3.0588235294117649e-01   2.6274509803921570e-01
            8.1568627450980391e-01   3.1764705882352939e-01   2.7058823529411763e-01
            8.1960784313725488e-01   3.2941176470588235e-01   2.7450980392156865e-01
            8.2352941176470584e-01   3.4117647058823530e-01   2.8235294117647058e-01
            8.2745098039215681e-01   3.5294117647058826e-01   2.9019607843137257e-01
            8.3529411764705885e-01   3.6470588235294116e-01   2.9411764705882354e-01
            8.3921568627450982e-01   3.7647058823529411e-01   3.0196078431372547e-01
            8.4313725490196079e-01   3.8823529411764707e-01   3.0980392156862746e-01
            8.5098039215686272e-01   4.0000000000000002e-01   3.1372549019607843e-01
            8.5490196078431369e-01   4.0784313725490196e-01   3.2156862745098042e-01
            8.5882352941176465e-01   4.1960784313725491e-01   3.2941176470588235e-01
            8.6666666666666670e-01   4.3137254901960786e-01   3.3725490196078434e-01
            8.7058823529411766e-01   4.4313725490196076e-01   3.4117647058823530e-01
            8.7450980392156863e-01   4.5098039215686275e-01   3.4901960784313724e-01
            8.8235294117647056e-01   4.6274509803921571e-01   3.5686274509803922e-01
            8.8627450980392153e-01   4.7450980392156861e-01   3.6470588235294116e-01
            8.9019607843137250e-01   4.8627450980392156e-01   3.7254901960784315e-01
            8.9803921568627454e-01   4.9411764705882355e-01   3.8039215686274508e-01
            9.0196078431372551e-01   5.0588235294117645e-01   3.8823529411764707e-01
            9.0588235294117647e-01   5.1764705882352946e-01   3.9607843137254900e-01
            9.1372549019607840e-01   5.2549019607843139e-01   4.0392156862745099e-01
            9.1764705882352937e-01   5.3725490196078429e-01   4.1176470588235292e-01
            9.2156862745098034e-01   5.4901960784313730e-01   4.1960784313725491e-01
            9.2549019607843142e-01   5.5686274509803924e-01   4.3137254901960786e-01
            9.2941176470588238e-01   5.6862745098039214e-01   4.3921568627450980e-01
            9.3333333333333335e-01   5.8039215686274515e-01   4.4705882352941179e-01
            9.3725490196078431e-01   5.8823529411764708e-01   4.5490196078431372e-01
            9.4117647058823528e-01   5.9999999999999998e-01   4.6274509803921571e-01
            9.4509803921568625e-01   6.0784313725490191e-01   4.7450980392156861e-01
            9.4901960784313721e-01   6.1960784313725492e-01   4.8235294117647060e-01
            9.5294117647058818e-01   6.2745098039215685e-01   4.9019607843137253e-01
            9.5294117647058818e-01   6.3921568627450975e-01   5.0196078431372548e-01
            9.5686274509803926e-01   6.4705882352941180e-01   5.0980392156862742e-01
            9.6078431372549022e-01   6.5490196078431373e-01   5.1764705882352946e-01
            9.6078431372549022e-01   6.6666666666666663e-01   5.2941176470588236e-01
            9.6470588235294119e-01   6.7450980392156867e-01   5.3725490196078429e-01
            9.6470588235294119e-01   6.8627450980392157e-01   5.4901960784313730e-01
            9.6862745098039216e-01   6.9411764705882351e-01   5.6078431372549020e-01
            9.6862745098039216e-01   7.0588235294117652e-01   5.7254901960784310e-01
            9.7254901960784312e-01   7.1372549019607845e-01   5.8431372549019611e-01
            9.7254901960784312e-01   7.2156862745098038e-01   5.9607843137254901e-01
            9.7647058823529409e-01   7.3333333333333328e-01   6.0392156862745094e-01
            9.7647058823529409e-01   7.4117647058823533e-01   6.1568627450980395e-01
            9.8039215686274506e-01   7.4901960784313726e-01   6.2745098039215685e-01
            9.8039215686274506e-01   7.6078431372549016e-01   6.3921568627450975e-01
            9.8039215686274506e-01   7.6862745098039220e-01   6.5098039215686276e-01
            9.8431372549019602e-01   7.7647058823529413e-01   6.6274509803921566e-01
            9.8431372549019602e-01   7.8823529411764703e-01   6.7450980392156867e-01
            9.8431372549019602e-01   7.9607843137254897e-01   6.8627450980392157e-01
            9.8823529411764710e-01   8.0392156862745101e-01   6.9803921568627447e-01
            9.8823529411764710e-01   8.1176470588235294e-01   7.0980392156862748e-01
            9.8823529411764710e-01   8.1960784313725488e-01   7.2156862745098038e-01
            9.9215686274509807e-01   8.2745098039215681e-01   7.3333333333333328e-01
            9.9215686274509807e-01   8.3529411764705885e-01   7.4117647058823533e-01
            9.9215686274509807e-01   8.3921568627450982e-01   7.5294117647058822e-01
            9.9215686274509807e-01   8.4705882352941175e-01   7.6078431372549016e-01
            9.9215686274509807e-01   8.5490196078431369e-01   7.7254901960784317e-01
            9.9215686274509807e-01   8.5882352941176465e-01   7.8039215686274510e-01
            9.9215686274509807e-01   8.6274509803921573e-01   7.8823529411764703e-01
            9.9215686274509807e-01   8.7058823529411766e-01   8.0000000000000004e-01
            9.9215686274509807e-01   8.7450980392156863e-01   8.0784313725490198e-01
            9.9215686274509807e-01   8.8235294117647056e-01   8.1568627450980391e-01
            9.9215686274509807e-01   8.8627450980392153e-01   8.2745098039215681e-01
            9.9215686274509807e-01   8.9411764705882357e-01   8.3529411764705885e-01
            9.8823529411764710e-01   8.9803921568627454e-01   8.4705882352941175e-01
            9.8823529411764710e-01   9.0588235294117647e-01   8.5490196078431369e-01
            9.8823529411764710e-01   9.0980392156862744e-01   8.6666666666666670e-01
            9.8823529411764710e-01   9.1764705882352937e-01   8.7450980392156863e-01
            9.8823529411764710e-01   9.2156862745098034e-01   8.8627450980392153e-01
            9.8823529411764710e-01   9.2549019607843142e-01   8.9411764705882357e-01
            9.8431372549019602e-01   9.3333333333333335e-01   9.0196078431372551e-01
            9.8431372549019602e-01   9.3725490196078431e-01   9.0980392156862744e-01
            9.8431372549019602e-01   9.4117647058823528e-01   9.1764705882352937e-01
            9.8431372549019602e-01   9.4509803921568625e-01   9.2549019607843142e-01
            9.8039215686274506e-01   9.4901960784313721e-01   9.3333333333333335e-01
            9.8039215686274506e-01   9.5294117647058818e-01   9.4117647058823528e-01
            9.8039215686274506e-01   9.5686274509803926e-01   9.4509803921568625e-01
            9.7647058823529409e-01   9.6078431372549022e-01   9.5294117647058818e-01
            9.7647058823529409e-01   9.6078431372549022e-01   9.5686274509803926e-01
            9.7647058823529409e-01   9.6470588235294119e-01   9.6078431372549022e-01
            9.7254901960784312e-01   9.6470588235294119e-01   9.6470588235294119e-01
            9.7254901960784312e-01   9.6862745098039216e-01   9.6862745098039216e-01
            9.6862745098039216e-01   9.6862745098039216e-01   9.6862745098039216e-01
            9.6862745098039216e-01   9.6862745098039216e-01   9.6862745098039216e-01
            9.6862745098039216e-01   9.6862745098039216e-01   9.6862745098039216e-01
            9.6470588235294119e-01   9.6862745098039216e-01   9.6862745098039216e-01
            9.6078431372549022e-01   9.6862745098039216e-01   9.6862745098039216e-01
            9.5686274509803926e-01   9.6470588235294119e-01   9.6862745098039216e-01
            9.5294117647058818e-01   9.6470588235294119e-01   9.6862745098039216e-01
            9.4901960784313721e-01   9.6078431372549022e-01   9.6470588235294119e-01
            9.4509803921568625e-01   9.6078431372549022e-01   9.6470588235294119e-01
            9.4117647058823528e-01   9.5686274509803926e-01   9.6470588235294119e-01
            9.3333333333333335e-01   9.5686274509803926e-01   9.6470588235294119e-01
            9.2941176470588238e-01   9.5294117647058818e-01   9.6078431372549022e-01
            9.2156862745098034e-01   9.4901960784313721e-01   9.6078431372549022e-01
            9.1372549019607840e-01   9.4509803921568625e-01   9.6078431372549022e-01
            9.0980392156862744e-01   9.4117647058823528e-01   9.6078431372549022e-01
            9.0196078431372551e-01   9.4117647058823528e-01   9.5686274509803926e-01
            8.9411764705882357e-01   9.3725490196078431e-01   9.5686274509803926e-01
            8.8627450980392153e-01   9.3333333333333335e-01   9.5686274509803926e-01
            8.7843137254901960e-01   9.2941176470588238e-01   9.5294117647058818e-01
            8.7058823529411766e-01   9.2549019607843142e-01   9.5294117647058818e-01
            8.6666666666666670e-01   9.2156862745098034e-01   9.4901960784313721e-01
            8.5882352941176465e-01   9.1764705882352937e-01   9.4901960784313721e-01
            8.5098039215686272e-01   9.1372549019607840e-01   9.4901960784313721e-01
            8.4313725490196079e-01   9.0980392156862744e-01   9.4509803921568625e-01
            8.3529411764705885e-01   9.0588235294117647e-01   9.4509803921568625e-01
            8.2745098039215681e-01   9.0196078431372551e-01   9.4117647058823528e-01
            8.1960784313725488e-01   8.9803921568627454e-01   9.4117647058823528e-01
            8.1176470588235294e-01   8.9411764705882357e-01   9.4117647058823528e-01
            8.0392156862745101e-01   8.9019607843137250e-01   9.3725490196078431e-01
            7.9607843137254897e-01   8.8627450980392153e-01   9.3725490196078431e-01
            7.8823529411764703e-01   8.8235294117647056e-01   9.3333333333333335e-01
            7.8039215686274510e-01   8.7843137254901960e-01   9.3333333333333335e-01
            7.7254901960784317e-01   8.7450980392156863e-01   9.2941176470588238e-01
            7.6470588235294112e-01   8.7058823529411766e-01   9.2549019607843142e-01
            7.5294117647058822e-01   8.6666666666666670e-01   9.2549019607843142e-01
            7.4509803921568629e-01   8.6274509803921573e-01   9.2156862745098034e-01
            7.3725490196078436e-01   8.5882352941176465e-01   9.2156862745098034e-01
            7.2549019607843135e-01   8.5098039215686272e-01   9.1764705882352937e-01
            7.1764705882352942e-01   8.4705882352941175e-01   9.1372549019607840e-01
            7.0588235294117652e-01   8.4313725490196079e-01   9.0980392156862744e-01
            6.9803921568627447e-01   8.3921568627450982e-01   9.0980392156862744e-01
            6.8627450980392157e-01   8.3137254901960789e-01   9.0588235294117647e-01
            6.7843137254901964e-01   8.2745098039215681e-01   9.0196078431372551e-01
            6.6666666666666663e-01   8.2352941176470584e-01   8.9803921568627454e-01
            6.5882352941176470e-01   8.1960784313725488e-01   8.9803921568627454e-01
            6.4705882352941180e-01   8.1176470588235294e-01   8.9411764705882357e-01
            6.3529411764705879e-01   8.0784313725490198e-01   8.9019607843137250e-01
            6.2745098039215685e-01   8.0000000000000004e-01   8.8627450980392153e-01
            6.1568627450980395e-01   7.9607843137254897e-01   8.8235294117647056e-01
            6.0392156862745094e-01   7.8823529411764703e-01   8.8235294117647056e-01
            5.9215686274509804e-01   7.8431372549019607e-01   8.7843137254901960e-01
            5.8431372549019611e-01   7.8039215686274510e-01   8.7450980392156863e-01
            5.7254901960784310e-01   7.7254901960784317e-01   8.7058823529411766e-01
            5.6078431372549020e-01   7.6470588235294112e-01   8.6666666666666670e-01
            5.4901960784313730e-01   7.6078431372549016e-01   8.6274509803921573e-01
            5.3725490196078429e-01   7.5294117647058822e-01   8.5882352941176465e-01
            5.2549019607843139e-01   7.4509803921568629e-01   8.5490196078431369e-01
            5.0980392156862742e-01   7.3725490196078436e-01   8.5098039215686272e-01
            4.9803921568627452e-01   7.2941176470588232e-01   8.4705882352941175e-01
            4.8627450980392156e-01   7.2549019607843135e-01   8.4313725490196079e-01
            4.7058823529411764e-01   7.1764705882352942e-01   8.3921568627450982e-01
            4.5490196078431372e-01   7.0588235294117652e-01   8.3529411764705885e-01
            4.4313725490196076e-01   6.9803921568627447e-01   8.3137254901960789e-01
            4.2745098039215684e-01   6.9019607843137254e-01   8.2745098039215681e-01
            4.1568627450980394e-01   6.8235294117647061e-01   8.1960784313725488e-01
            4.0000000000000002e-01   6.7450980392156867e-01   8.1568627450980391e-01
            3.8431372549019610e-01   6.6666666666666663e-01   8.1176470588235294e-01
            3.7254901960784315e-01   6.5882352941176470e-01   8.0784313725490198e-01
            3.5686274509803922e-01   6.4705882352941180e-01   8.0392156862745101e-01
            3.4509803921568627e-01   6.3921568627450975e-01   8.0000000000000004e-01
            3.3333333333333331e-01   6.3137254901960782e-01   7.9607843137254897e-01
            3.2156862745098042e-01   6.2352941176470589e-01   7.8823529411764703e-01
            3.0980392156862746e-01   6.1568627450980395e-01   7.8431372549019607e-01
            2.9803921568627451e-01   6.0784313725490191e-01   7.8039215686274510e-01
            2.9019607843137257e-01   5.9999999999999998e-01   7.7647058823529413e-01
            2.7843137254901962e-01   5.9215686274509804e-01   7.7254901960784317e-01
            2.7058823529411763e-01   5.8431372549019611e-01   7.6862745098039220e-01
            2.6274509803921570e-01   5.7647058823529407e-01   7.6470588235294112e-01
            2.5490196078431371e-01   5.6862745098039214e-01   7.6078431372549016e-01
            2.4705882352941178e-01   5.6078431372549020e-01   7.5686274509803919e-01
            2.4313725490196078e-01   5.5686274509803924e-01   7.5294117647058822e-01
            2.3529411764705882e-01   5.4901960784313730e-01   7.4901960784313726e-01
            2.3137254901960785e-01   5.4117647058823526e-01   7.4901960784313726e-01
            2.2352941176470589e-01   5.3333333333333333e-01   7.4509803921568629e-01
            2.1960784313725490e-01   5.2941176470588236e-01   7.4117647058823533e-01
            2.1176470588235294e-01   5.2156862745098043e-01   7.3725490196078436e-01
            2.0784313725490197e-01   5.1372549019607838e-01   7.3725490196078436e-01
            2.0392156862745098e-01   5.0980392156862742e-01   7.3333333333333328e-01
            1.9607843137254902e-01   5.0196078431372548e-01   7.2941176470588232e-01
            1.9215686274509805e-01   4.9411764705882355e-01   7.2941176470588232e-01
            1.8823529411764706e-01   4.9019607843137253e-01   7.2549019607843135e-01
            1.8431372549019609e-01   4.8235294117647060e-01   7.2156862745098038e-01
            1.8039215686274510e-01   4.7450980392156861e-01   7.1764705882352942e-01
            1.7254901960784313e-01   4.7058823529411764e-01   7.1372549019607845e-01
            1.6862745098039217e-01   4.6274509803921571e-01   7.1372549019607845e-01
            1.6470588235294117e-01   4.5490196078431372e-01   7.0980392156862748e-01
            1.6078431372549021e-01   4.5098039215686275e-01   7.0588235294117652e-01
            1.5686274509803921e-01   4.4313725490196076e-01   7.0196078431372544e-01
            1.5294117647058825e-01   4.3529411764705883e-01   6.9803921568627447e-01
            1.4901960784313725e-01   4.2745098039215684e-01   6.9411764705882351e-01
            1.4509803921568629e-01   4.2352941176470588e-01   6.9019607843137254e-01
            1.3725490196078433e-01   4.1568627450980394e-01   6.8627450980392157e-01
            1.3333333333333333e-01   4.0784313725490196e-01   6.7843137254901964e-01
            1.2941176470588237e-01   4.0000000000000002e-01   6.7450980392156867e-01
            1.2549019607843137e-01   3.9215686274509803e-01   6.7058823529411760e-01
            1.2156862745098039e-01   3.8431372549019610e-01   6.6274509803921566e-01
            1.1372549019607843e-01   3.7647058823529411e-01   6.5490196078431373e-01
            1.0980392156862745e-01   3.6862745098039218e-01   6.4705882352941180e-01
            1.0588235294117647e-01   3.6078431372549019e-01   6.3921568627450975e-01
            1.0196078431372549e-01   3.5294117647058826e-01   6.3137254901960782e-01
            9.8039215686274508e-02   3.4509803921568627e-01   6.1960784313725492e-01
            9.4117647058823528e-02   3.3725490196078434e-01   6.1176470588235299e-01
            8.6274509803921567e-02   3.2941176470588235e-01   5.9999999999999998e-01
            8.2352941176470587e-02   3.2156862745098042e-01   5.8823529411764708e-01
            7.8431372549019607e-02   3.1372549019607843e-01   5.7647058823529407e-01
            7.4509803921568626e-02   3.0196078431372547e-01   5.6470588235294117e-01
            7.0588235294117646e-02   2.9411764705882354e-01   5.5294117647058827e-01
            6.6666666666666666e-02   2.8627450980392155e-01   5.4117647058823526e-01
            6.2745098039215685e-02   2.7843137254901962e-01   5.2941176470588236e-01
            5.8823529411764705e-02   2.7058823529411763e-01   5.1372549019607838e-01
            5.0980392156862744e-02   2.5882352941176473e-01   5.0196078431372548e-01
            4.7058823529411764e-02   2.5098039215686274e-01   4.8627450980392156e-01
            4.3137254901960784e-02   2.4313725490196078e-01   4.7058823529411764e-01
            3.9215686274509803e-02   2.3529411764705882e-01   4.5882352941176469e-01
            3.5294117647058823e-02   2.2352941176470589e-01   4.4313725490196076e-01
            3.1372549019607843e-02   2.1568627450980393e-01   4.2745098039215684e-01
            2.7450980392156862e-02   2.0784313725490197e-01   4.1176470588235292e-01
            2.3529411764705882e-02   1.9607843137254902e-01   3.9607843137254900e-01
            1.9607843137254902e-02   1.8823529411764706e-01   3.8039215686274508e-01]);
    case 'land'
        % Land colours from ETOPO1
        Data=[
            0.56410 0.20000 0.40000 0.00000
            0.56920 0.20000 0.80000 0.40000
            0.57440 0.73333 0.89412 0.57255
            0.58970 1.00000 0.86275 0.72549
            0.61540 0.95294 0.79216 0.53725
            0.64100 0.90196 0.72157 0.34510
            0.66670 0.85098 0.65098 0.15294
            0.69230 0.65882 0.60392 0.12157
            0.71790 0.64314 0.56471 0.09804
            0.74360 0.63529 0.52549 0.07451
            0.76920 0.62353 0.48235 0.05098
            0.79490 0.61176 0.44314 0.02745
            0.82050 0.60000 0.40000 0.00000
            0.84620 0.63529 0.34902 0.34902
            0.87180 0.69804 0.46275 0.46275
            0.89740 0.71765 0.57647 0.57647
            0.92310 0.76078 0.69020 0.69020
            0.94870 0.80000 0.80000 0.80000
            0.97440 0.89804 0.89804 0.89804
            1.00000 1.00000 1.00000 1.00000];
        r=(Data(:,1)-Data(1,1))/(Data(end,1)-Data(1,1))*255;
        Data=Data(:,2:4);
    case 'water'
        % Water colours from ETOPO1
        Data=[0.00000 0.03922 0.00000 0.47451
            0.02560 0.10196 0.00000 0.53725
            0.05130 0.14902 0.00000 0.59608
            0.07690 0.10588 0.01176 0.65098
            0.10260 0.06275 0.02353 0.70588
            0.12820 0.01961 0.03529 0.75686
            0.15380 0.00000 0.05490 0.79608
            0.17950 0.00000 0.08627 0.82353
            0.20510 0.00000 0.11765 0.84706
            0.23080 0.00000 0.15294 0.87451
            0.25640 0.04706 0.26667 0.90588
            0.28210 0.10196 0.40000 0.94118
            0.30770 0.07451 0.45882 0.95686
            0.33330 0.05490 0.52157 0.97647
            0.35900 0.08235 0.61961 0.98824
            0.38460 0.11765 0.69804 1.00000
            0.41030 0.16863 0.72941 1.00000
            0.43590 0.21569 0.75686 1.00000
            0.46150 0.25490 0.78431 1.00000
            0.48720 0.30980 0.82353 1.00000
            0.51280 0.36863 0.87451 1.00000
            0.53850 0.54118 0.89020 1.00000
            0.56410 0.73725 0.90196 1.00000];
        r=(Data(:,1)-Data(1,1))/(Data(end,1)-Data(1,1))*255;
        Data=Data(:,2:4);
    case 'gland'
        % Land colours from ETOPO1
        Data=[
            0.56410 0.20000 0.40000 0.00000
            0.56920 0.20000 0.80000 0.40000
            0.57440 0.73333 0.89412 0.57255
            0.58970 1.00000 0.86275 0.72549
            0.61540 0.95294 0.79216 0.53725
            0.64100 0.90196 0.72157 0.34510
            0.66670 0.85098 0.65098 0.15294
            0.69230 0.65882 0.60392 0.12157
            0.71790 0.64314 0.56471 0.09804
            0.74360 0.63529 0.52549 0.07451
            0.76920 0.62353 0.48235 0.05098
            0.79490 0.61176 0.44314 0.02745
            0.82050 0.60000 0.40000 0.00000
            0.84620 0.63529 0.34902 0.34902
            0.87180 0.69804 0.46275 0.46275
            0.89740 0.71765 0.57647 0.57647
            0.92310 0.76078 0.69020 0.69020
            0.94870 0.80000 0.80000 0.80000
            0.97440 0.89804 0.89804 0.89804
            1.00000 1.00000 1.00000 1.00000];
        % Same colours as in 'land' but I stretch the
        % green a little
        r=[0:length(Data)-1]/(length(Data)-1)*255;
        Data=Data(:,2:4);
    case 'bland'
        % Land colours from ETOPO1
        Data=[
            %  0.56410 0.20000 0.40000 0.00000
            %  0.56920 0.20000 0.80000 0.40000
            %  0.57440 0.73333 0.89412 0.57255
            0.58970 1.00000 0.86275 0.72549
            0.61540 0.95294 0.79216 0.53725
            0.64100 0.90196 0.72157 0.34510
            0.66670 0.85098 0.65098 0.15294
            0.69230 0.65882 0.60392 0.12157
            0.71790 0.64314 0.56471 0.09804
            0.74360 0.63529 0.52549 0.07451
            0.76920 0.62353 0.48235 0.05098
            0.79490 0.61176 0.44314 0.02745
            0.82050 0.60000 0.40000 0.00000
            0.84620 0.63529 0.34902 0.34902
            0.87180 0.69804 0.46275 0.46275
            0.89740 0.71765 0.57647 0.57647
            0.92310 0.76078 0.69020 0.69020
            0.94870 0.80000 0.80000 0.80000
            0.97440 0.89804 0.89804 0.89804
            1.00000 1.00000 1.00000 1.00000];
        % Same colours as in 'land' but I remove the
        % Green
        r=[0:length(Data)-1]/(length(Data)-1)*255;
        Data=Data(:,2:4);
    case 'odv'
        % Created from CET toolbox.
        Data=[ 9.0101331e-01   0.0000000e+00   9.3251202e-01
            8.9241369e-01   5.4084707e-02   9.3493161e-01
            8.8374697e-01   9.2009868e-02   9.3733662e-01
            8.7501086e-01   1.1940643e-01   9.3972123e-01
            8.6621022e-01   1.4173077e-01   9.4208765e-01
            8.5733587e-01   1.6085164e-01   9.4443514e-01
            8.4839234e-01   1.7779733e-01   9.4676122e-01
            8.3937643e-01   1.9304161e-01   9.4906833e-01
            8.3028040e-01   2.0703619e-01   9.5135665e-01
            8.2110339e-01   2.1996946e-01   9.5362260e-01
            8.1184800e-01   2.3203951e-01   9.5586460e-01
            8.0250437e-01   2.4334747e-01   9.5808151e-01
            7.9308093e-01   2.5406699e-01   9.6027147e-01
            7.8356163e-01   2.6421451e-01   9.6243395e-01
            7.7394557e-01   2.7387323e-01   9.6457095e-01
            7.6424053e-01   2.8309721e-01   9.6668309e-01
            7.5443764e-01   2.9192664e-01   9.6876443e-01
            7.4452229e-01   3.0039475e-01   9.7081305e-01
            7.3450848e-01   3.0856348e-01   9.7283330e-01
            7.2437830e-01   3.1642474e-01   9.7481118e-01
            7.1413293e-01   3.2400124e-01   9.7675955e-01
            7.0377315e-01   3.3133238e-01   9.7866788e-01
            6.9328911e-01   3.3844029e-01   9.8054235e-01
            6.8267129e-01   3.4532943e-01   9.8237525e-01
            6.7192363e-01   3.5200255e-01   9.8416286e-01
            6.6103315e-01   3.5849315e-01   9.8590542e-01
            6.5000382e-01   3.6480311e-01   9.8760208e-01
            6.3881994e-01   3.7093737e-01   9.8924507e-01
            6.2748126e-01   3.7692778e-01   9.9083128e-01
            6.1597716e-01   3.8274509e-01   9.9236641e-01
            6.0430713e-01   3.8842528e-01   9.9384206e-01
            5.9245905e-01   3.9395999e-01   9.9524582e-01
            5.8041851e-01   3.9936053e-01   9.9658429e-01
            5.6818292e-01   4.0464332e-01   9.9785650e-01
            5.5574776e-01   4.0981228e-01   9.9904764e-01
            5.4311794e-01   4.1484604e-01   1.0000000e+00
            5.3025652e-01   4.1977665e-01   1.0000000e+00
            5.1717467e-01   4.2459106e-01   1.0000000e+00
            5.0384831e-01   4.2929918e-01   1.0000000e+00
            4.9028115e-01   4.3389560e-01   1.0000000e+00
            4.7646435e-01   4.3840653e-01   1.0000000e+00
            4.6239168e-01   4.4281244e-01   1.0000000e+00
            4.4807735e-01   4.4710328e-01   1.0000000e+00
            4.3348133e-01   4.5131439e-01   1.0000000e+00
            4.1882644e-01   4.5541169e-01   1.0000000e+00
            4.0463168e-01   4.5935111e-01   1.0000000e+00
            3.9127493e-01   4.6308440e-01   1.0000000e+00
            3.7887635e-01   4.6663821e-01   1.0000000e+00
            3.6754491e-01   4.6997659e-01   9.9710824e-01
            3.5727105e-01   4.7310156e-01   9.9361634e-01
            3.4806754e-01   4.7605687e-01   9.8960956e-01
            3.3991294e-01   4.7882200e-01   9.8516720e-01
            3.3278841e-01   4.8142044e-01   9.8033564e-01
            3.2657245e-01   4.8386769e-01   9.7517562e-01
            3.2131863e-01   4.8617771e-01   9.6972157e-01
            3.1692286e-01   4.8836273e-01   9.6400671e-01
            3.1327820e-01   4.9042788e-01   9.5808298e-01
            3.1036518e-01   4.9237894e-01   9.5195268e-01
            3.0812518e-01   4.9424286e-01   9.4564700e-01
            3.0652684e-01   4.9600072e-01   9.3918421e-01
            3.0545194e-01   4.9770307e-01   9.3258337e-01
            3.0485773e-01   4.9930273e-01   9.2586855e-01
            3.0471500e-01   5.0085383e-01   9.1903664e-01
            3.0497614e-01   5.0233287e-01   9.1211284e-01
            3.0559984e-01   5.0375514e-01   9.0509893e-01
            3.0652667e-01   5.0512067e-01   8.9799743e-01
            3.0771336e-01   5.0643775e-01   8.9082591e-01
            3.0918805e-01   5.0771753e-01   8.8359036e-01
            3.1084258e-01   5.0894669e-01   8.7628822e-01
            3.1265990e-01   5.1012895e-01   8.6893964e-01
            3.1464971e-01   5.1127898e-01   8.6153178e-01
            3.1677717e-01   5.1239777e-01   8.5407999e-01
            3.1899621e-01   5.1348864e-01   8.4658877e-01
            3.2129859e-01   5.1454679e-01   8.3906246e-01
            3.2370068e-01   5.1556074e-01   8.3149691e-01
            3.2615709e-01   5.1656269e-01   8.2388686e-01
            3.2867278e-01   5.1754427e-01   8.1625836e-01
            3.3117878e-01   5.1849579e-01   8.0859723e-01
            3.3372872e-01   5.1942240e-01   8.0091115e-01
            3.3627834e-01   5.2032367e-01   7.9320222e-01
            3.3885910e-01   5.2121527e-01   7.8546403e-01
            3.4141138e-01   5.2207962e-01   7.7770923e-01
            3.4392756e-01   5.2294489e-01   7.6993203e-01
            3.4645715e-01   5.2377972e-01   7.6213677e-01
            3.4896939e-01   5.2459818e-01   7.5432975e-01
            3.5143015e-01   5.2540630e-01   7.4649947e-01
            3.5384400e-01   5.2620510e-01   7.3866259e-01
            3.5626705e-01   5.2698442e-01   7.3080018e-01
            3.5861057e-01   5.2774386e-01   7.2292779e-01
            3.6090556e-01   5.2850832e-01   7.1504566e-01
            3.6318916e-01   5.2925382e-01   7.0715015e-01
            3.6541282e-01   5.2998346e-01   6.9923396e-01
            3.6755212e-01   5.3070830e-01   6.9133423e-01
            3.6961736e-01   5.3143077e-01   6.8342411e-01
            3.7153984e-01   5.3216045e-01   6.7551926e-01
            3.7333964e-01   5.3289069e-01   6.6762844e-01
            3.7498875e-01   5.3361944e-01   6.5974393e-01
            3.7651860e-01   5.3436635e-01   6.5187458e-01
            3.7790211e-01   5.3511255e-01   6.4400546e-01
            3.7916289e-01   5.3586250e-01   6.3616246e-01
            3.8030058e-01   5.3663010e-01   6.2832375e-01
            3.8132032e-01   5.3739134e-01   6.2049557e-01
            3.8220248e-01   5.3815863e-01   6.1266534e-01
            3.8298164e-01   5.3892674e-01   6.0486256e-01
            3.8363602e-01   5.3969563e-01   5.9706617e-01
            3.8417043e-01   5.4048220e-01   5.8926622e-01
            3.8459090e-01   5.4126271e-01   5.8148982e-01
            3.8489570e-01   5.4205784e-01   5.7372243e-01
            3.8508983e-01   5.4286004e-01   5.6595990e-01
            3.8517497e-01   5.4365697e-01   5.5820324e-01
            3.8514825e-01   5.4446264e-01   5.5046654e-01
            3.8500591e-01   5.4527490e-01   5.4273583e-01
            3.8475369e-01   5.4609572e-01   5.3500496e-01
            3.8439432e-01   5.4691158e-01   5.2728974e-01
            3.8392969e-01   5.4773483e-01   5.1957424e-01
            3.8335370e-01   5.4857211e-01   5.1186858e-01
            3.8264961e-01   5.4940346e-01   5.0418316e-01
            3.8184174e-01   5.5024295e-01   4.9647988e-01
            3.8093456e-01   5.5108859e-01   4.8880378e-01
            3.7989709e-01   5.5194307e-01   4.8112245e-01
            3.7875488e-01   5.5279313e-01   4.7344838e-01
            3.7750149e-01   5.5365179e-01   4.6578241e-01
            3.7613365e-01   5.5452404e-01   4.5812639e-01
            3.7463906e-01   5.5538969e-01   4.5047471e-01
            3.7301856e-01   5.5626729e-01   4.4282654e-01
            3.7128174e-01   5.5716409e-01   4.3518225e-01
            3.6943191e-01   5.5804236e-01   4.2753741e-01
            3.6744351e-01   5.5893816e-01   4.1989061e-01
            3.6533681e-01   5.5984781e-01   4.1225487e-01
            3.6309063e-01   5.6075086e-01   4.0460539e-01
            3.6070294e-01   5.6167024e-01   3.9696864e-01
            3.5819517e-01   5.6259588e-01   3.8934301e-01
            3.5553881e-01   5.6352351e-01   3.8170805e-01
            3.5271499e-01   5.6445380e-01   3.7408098e-01
            3.4974311e-01   5.6539030e-01   3.6644835e-01
            3.4660653e-01   5.6634992e-01   3.5879830e-01
            3.4330675e-01   5.6731182e-01   3.5117765e-01
            3.3982229e-01   5.6827515e-01   3.4353893e-01
            3.3613030e-01   5.6925183e-01   3.3590528e-01
            3.3226927e-01   5.7023982e-01   3.2827190e-01
            3.2817437e-01   5.7123976e-01   3.2064919e-01
            3.2386167e-01   5.7225439e-01   3.1300205e-01
            3.1930992e-01   5.7327706e-01   3.0538599e-01
            3.1448277e-01   5.7430284e-01   2.9775044e-01
            3.0940512e-01   5.7535700e-01   2.9013069e-01
            3.0397213e-01   5.7642766e-01   2.8255045e-01
            2.9821634e-01   5.7749511e-01   2.7494949e-01
            2.9213635e-01   5.7858904e-01   2.6736545e-01
            2.8585508e-01   5.7968241e-01   2.5973929e-01
            2.7979429e-01   5.8071733e-01   2.5182584e-01
            2.7409306e-01   5.8169092e-01   2.4353615e-01
            2.6895355e-01   5.8258780e-01   2.3488952e-01
            2.6452586e-01   5.8339123e-01   2.2571999e-01
            2.6115049e-01   5.8407359e-01   2.1600397e-01
            2.5921386e-01   5.8459763e-01   2.0562062e-01
            2.5917881e-01   5.8491193e-01   1.9457571e-01
            2.6172691e-01   5.8493530e-01   1.8281914e-01
            2.6736292e-01   5.8459009e-01   1.7079381e-01
            2.7605556e-01   5.8384546e-01   1.5913201e-01
            2.8684229e-01   5.8275563e-01   1.4851063e-01
            2.9881924e-01   5.8141442e-01   1.3929985e-01
            3.1118650e-01   5.7990916e-01   1.3150590e-01
            3.2354119e-01   5.7827384e-01   1.2482607e-01
            3.3571076e-01   5.7657878e-01   1.1919225e-01
            3.4760451e-01   5.7479693e-01   1.1436156e-01
            3.5917972e-01   5.7298881e-01   1.1035678e-01
            3.7046210e-01   5.7112054e-01   1.0690929e-01
            3.8146525e-01   5.6922129e-01   1.0397365e-01
            3.9217543e-01   5.6729630e-01   1.0155055e-01
            4.0262874e-01   5.6532047e-01   9.9563720e-02
            4.1285471e-01   5.6333643e-01   9.8014331e-02
            4.2283397e-01   5.6130090e-01   9.6695447e-02
            4.3264027e-01   5.5924194e-01   9.5762755e-02
            4.4224318e-01   5.5716236e-01   9.4951662e-02
            4.5166986e-01   5.5503657e-01   9.4080622e-02
            4.6093550e-01   5.5288878e-01   9.3190838e-02
            4.7006969e-01   5.5071469e-01   9.2394545e-02
            4.7905191e-01   5.4851288e-01   9.1613212e-02
            4.8791547e-01   5.4627320e-01   9.0767770e-02
            4.9663686e-01   5.4399494e-01   8.9948701e-02
            5.0525765e-01   5.4168472e-01   8.9142595e-02
            5.1377978e-01   5.3935013e-01   8.8344290e-02
            5.2217703e-01   5.3698984e-01   8.7579467e-02
            5.3049564e-01   5.3457717e-01   8.6781980e-02
            5.3871960e-01   5.3213942e-01   8.5936085e-02
            5.4684834e-01   5.2966356e-01   8.5197765e-02
            5.5489406e-01   5.2715147e-01   8.4499111e-02
            5.6286931e-01   5.2459702e-01   8.3736580e-02
            5.7075371e-01   5.2200153e-01   8.2965076e-02
            5.7857766e-01   5.1938381e-01   8.2294444e-02
            5.8633451e-01   5.1671589e-01   8.1575183e-02
            5.9402116e-01   5.1401457e-01   8.0830769e-02
            6.0164481e-01   5.1125679e-01   8.0119227e-02
            6.0921718e-01   5.0847758e-01   7.9421412e-02
            6.1672372e-01   5.0562705e-01   7.8739648e-02
            6.2417702e-01   5.0275308e-01   7.8075667e-02
            6.3157495e-01   4.9982760e-01   7.7428325e-02
            6.3892277e-01   4.9686181e-01   7.6797687e-02
            6.4622234e-01   4.9384916e-01   7.6184872e-02
            6.5347472e-01   4.9079302e-01   7.5589796e-02
            6.6068297e-01   4.8767582e-01   7.5007376e-02
            6.6785657e-01   4.8449686e-01   7.4444887e-02
            6.7497536e-01   4.8128060e-01   7.3917039e-02
            6.8205540e-01   4.7801106e-01   7.3382464e-02
            6.8910997e-01   4.7469164e-01   7.2799197e-02
            6.9611556e-01   4.7130644e-01   7.2248466e-02
            7.0308682e-01   4.6785815e-01   7.1781405e-02
            7.1002283e-01   4.6436799e-01   7.1317804e-02
            7.1692989e-01   4.6078972e-01   7.0812391e-02
            7.2379432e-01   4.5716541e-01   7.0334205e-02
            7.3063604e-01   4.5347355e-01   6.9928066e-02
            7.3744490e-01   4.4971842e-01   6.9550523e-02
            7.4422586e-01   4.4588089e-01   6.9158288e-02
            7.5098360e-01   4.4199555e-01   6.8753605e-02
            7.5770386e-01   4.3801889e-01   6.8363033e-02
            7.6440159e-01   4.3395308e-01   6.8000452e-02
            7.7107741e-01   4.2983143e-01   6.7662807e-02
            7.7772504e-01   4.2561806e-01   6.7345763e-02
            7.8435064e-01   4.2131316e-01   6.7047765e-02
            7.9094561e-01   4.1693930e-01   6.6769501e-02
            7.9752256e-01   4.1246414e-01   6.6511686e-02
            8.0408218e-01   4.0788143e-01   6.6274748e-02
            8.1061283e-01   4.0321333e-01   6.6058803e-02
            8.1712938e-01   3.9843601e-01   6.5863871e-02
            8.2362256e-01   3.9354671e-01   6.5689946e-02
            8.3010058e-01   3.8855623e-01   6.5537030e-02
            8.3655552e-01   3.8344453e-01   6.5405124e-02
            8.4299328e-01   3.7819900e-01   6.5294161e-02
            8.4941547e-01   3.7282716e-01   6.5204178e-02
            8.5581779e-01   3.6732198e-01   6.5135120e-02
            8.6220650e-01   3.6168376e-01   6.5087154e-02
            8.6857747e-01   3.5588751e-01   6.5060182e-02
            8.7492597e-01   3.4992373e-01   6.5054456e-02
            8.8126652e-01   3.4378373e-01   6.5069774e-02
            8.8758708e-01   3.3748220e-01   6.5106460e-02
            8.9389887e-01   3.3096597e-01   6.5164196e-02
            9.0019487e-01   3.2426128e-01   6.5243333e-02
            9.0647723e-01   3.1733910e-01   6.5343657e-02
            9.1274449e-01   3.1018523e-01   6.5465418e-02
            9.1899599e-01   3.0274206e-01   6.5608630e-02
            9.2524045e-01   2.9504892e-01   6.5773425e-02
            9.3147122e-01   2.8704671e-01   6.5959943e-02
            9.3769166e-01   2.7873707e-01   6.6168130e-02
            9.4389831e-01   2.7005145e-01   6.6398279e-02
            9.5009275e-01   2.6094683e-01   6.6650135e-02
            9.5627950e-01   2.5141834e-01   6.6923773e-02
            9.6245047e-01   2.4136103e-01   6.7218475e-02
            9.6861585e-01   2.3072982e-01   6.7533548e-02
            9.7476660e-01   2.1945144e-01   6.7869417e-02
            9.8091309e-01   2.0738932e-01   6.8229718e-02
            9.8705021e-01   1.9442270e-01   6.8619747e-02
            9.9317979e-01   1.8027042e-01   6.9032328e-02
            9.9929581e-01   1.6476261e-01   6.9441308e-02
            1.0000000e+00   1.4745190e-01   6.9829429e-02
            1.0000000e+00   1.2757129e-01   7.0234818e-02
            1.0000000e+00   1.0393299e-01   7.0711225e-02];
    case 'cyclic'
        % Created from CET toolbox.
        Data=[  0.975609 0.520118 0.970963
            0.980325 0.517215 0.963189
            0.984195 0.513568 0.954482
            0.987253 0.509216 0.944879
            0.989542 0.504200 0.934424
            0.991117 0.498555 0.923199
            0.992044 0.492333 0.911264
            0.992386 0.485608 0.898711
            0.992208 0.478440 0.885626
            0.991576 0.470885 0.872092
            0.990553 0.462981 0.858186
            0.989194 0.454827 0.843992
            0.987544 0.446415 0.829579
            0.985648 0.437830 0.814990
            0.983537 0.429071 0.800291
            0.981245 0.420188 0.785511
            0.978786 0.411201 0.770684
            0.976182 0.402116 0.755835
            0.973444 0.392971 0.740978
            0.970571 0.383765 0.726138
            0.967575 0.374503 0.711311
            0.964440 0.365189 0.696498
            0.961184 0.355864 0.681701
            0.957791 0.346503 0.666936
            0.954249 0.337156 0.652191
            0.950551 0.327802 0.637451
            0.946681 0.318488 0.622713
            0.942637 0.309226 0.607977
            0.938393 0.299997 0.593222
            0.933944 0.290868 0.578444
            0.929287 0.281847 0.563657
            0.924409 0.272939 0.548831
            0.919305 0.264182 0.533960
            0.913979 0.255566 0.519077
            0.908420 0.247117 0.504149
            0.902640 0.238833 0.489176
            0.896646 0.230734 0.474192
            0.890441 0.222811 0.459182
            0.884048 0.215011 0.444155
            0.877467 0.207399 0.429109
            0.870720 0.199872 0.414066
            0.863829 0.192502 0.399025
            0.856787 0.185227 0.384010
            0.849624 0.178009 0.369000
            0.842350 0.170816 0.354007
            0.834978 0.163729 0.339075
            0.827528 0.156645 0.324159
            0.820013 0.149618 0.309330
            0.812451 0.142614 0.294521
            0.804858 0.135667 0.279821
            0.797250 0.128854 0.265183
            0.789662 0.122170 0.250661
            0.782115 0.115783 0.236278
            0.774642 0.109705 0.222009
            0.767296 0.104001 0.207949
            0.760123 0.098945 0.194118
            0.753173 0.094721 0.180491
            0.746518 0.091421 0.167231
            0.740217 0.089258 0.154263
            0.734335 0.088445 0.141736
            0.728940 0.089069 0.129628
            0.724103 0.091195 0.117995
            0.719878 0.094767 0.106894
            0.716312 0.099636 0.096236
            0.713433 0.105801 0.086225
            0.711275 0.113024 0.076774
            0.709830 0.121038 0.068001
            0.709098 0.129894 0.059810
            0.709044 0.139202 0.052084
            0.709630 0.149017 0.045170
            0.710808 0.159138 0.038855
            0.712514 0.169499 0.033292
            0.714696 0.179947 0.028947
            0.717291 0.190504 0.025470
            0.720213 0.201067 0.022733
            0.723417 0.211657 0.020622
            0.726850 0.222137 0.019034
            0.730457 0.232565 0.017876
            0.734193 0.242879 0.017071
            0.738018 0.253114 0.016547
            0.741915 0.263227 0.016249
            0.745847 0.273221 0.016125
            0.749795 0.283111 0.016137
            0.753751 0.292884 0.016250
            0.757692 0.302570 0.016440
            0.761605 0.312172 0.016684
            0.765495 0.321712 0.016989
            0.769322 0.331165 0.017296
            0.773097 0.340580 0.017600
            0.776812 0.349937 0.017896
            0.780442 0.359250 0.018177
            0.784000 0.368555 0.018435
            0.787459 0.377827 0.018665
            0.790825 0.387090 0.018861
            0.794094 0.396362 0.019014
            0.797244 0.405638 0.019122
            0.800294 0.414903 0.019179
            0.803223 0.424179 0.019183
            0.806046 0.433460 0.019132
            0.808758 0.442759 0.019026
            0.811360 0.452053 0.018866
            0.813856 0.461340 0.018654
            0.816265 0.470631 0.018394
            0.818571 0.479917 0.018090
            0.820801 0.489183 0.017747
            0.822954 0.498459 0.017369
            0.825032 0.507709 0.016963
            0.827042 0.516942 0.016575
            0.828978 0.526150 0.016225
            0.830837 0.535317 0.015949
            0.832616 0.544456 0.015792
            0.834313 0.553536 0.015820
            0.835903 0.562570 0.016118
            0.837376 0.571505 0.016798
            0.838696 0.580351 0.018004
            0.839847 0.589056 0.019918
            0.840771 0.597613 0.022763
            0.841431 0.605951 0.026810
            0.841773 0.614040 0.032376
            0.841735 0.621817 0.040011
            0.841252 0.629224 0.049120
            0.840261 0.636189 0.059305
            0.838684 0.642636 0.070471
            0.836472 0.648502 0.082640
            0.833559 0.653724 0.095596
            0.829894 0.658244 0.109262
            0.825436 0.662013 0.123432
            0.820169 0.664981 0.138185
            0.814079 0.667153 0.153209
            0.807174 0.668516 0.168557
            0.799467 0.669078 0.184047
            0.790984 0.668874 0.199610
            0.781782 0.667950 0.215201
            0.771898 0.666350 0.230722
            0.761389 0.664160 0.246105
            0.750316 0.661435 0.261333
            0.738731 0.658248 0.276357
            0.726679 0.654667 0.291131
            0.714212 0.650774 0.305673
            0.701369 0.646617 0.319948
            0.688182 0.642255 0.333971
            0.674664 0.637733 0.347744
            0.660836 0.633096 0.361267
            0.646712 0.628377 0.374577
            0.632276 0.623598 0.387655
            0.617525 0.618782 0.400558
            0.602450 0.613930 0.413255
            0.587033 0.609090 0.425781
            0.571260 0.604231 0.438168
            0.555110 0.599363 0.450404
            0.538576 0.594493 0.462504
            0.521599 0.589621 0.474508
            0.504201 0.584753 0.486396
            0.486316 0.579863 0.498214
            0.467958 0.574931 0.509934
            0.449120 0.569964 0.521598
            0.429782 0.564934 0.533218
            0.409982 0.559832 0.544799
            0.389720 0.554613 0.556346
            0.369082 0.549264 0.567890
            0.348105 0.543747 0.579434
            0.326877 0.538047 0.590979
            0.305561 0.532112 0.602556
            0.284236 0.525924 0.614165
            0.263159 0.519446 0.625830
            0.242495 0.512650 0.637551
            0.222572 0.505507 0.649339
            0.203679 0.498022 0.661190
            0.186189 0.490122 0.673126
            0.170514 0.481834 0.685136
            0.157117 0.473118 0.697232
            0.146378 0.464018 0.709410
            0.138585 0.454497 0.721643
            0.133804 0.444580 0.733949
            0.131904 0.434263 0.746297
            0.132531 0.423594 0.758686
            0.135191 0.412602 0.771094
            0.139304 0.401284 0.783498
            0.144495 0.389694 0.795878
            0.150242 0.377897 0.808211
            0.156303 0.365938 0.820451
            0.162459 0.353867 0.832570
            0.168638 0.341794 0.844516
            0.174839 0.329789 0.856233
            0.181049 0.317937 0.867668
            0.187461 0.306394 0.878752
            0.194078 0.295237 0.889426
            0.201013 0.284653 0.899619
            0.208478 0.274783 0.909260
            0.216496 0.265785 0.918299
            0.225063 0.257783 0.926677
            0.234299 0.250981 0.934357
            0.244088 0.245443 0.941312
            0.254476 0.241339 0.947531
            0.265294 0.238660 0.953012
            0.276520 0.237496 0.957776
            0.287984 0.237734 0.961852
            0.299643 0.239327 0.965296
            0.311348 0.242176 0.968161
            0.323045 0.246128 0.970496
            0.334642 0.251060 0.972381
            0.346059 0.256764 0.973864
            0.357329 0.263129 0.975020
            0.368354 0.269998 0.975904
            0.379155 0.277244 0.976570
            0.389731 0.284764 0.977064
            0.400095 0.292503 0.977423
            0.410262 0.300363 0.977682
            0.420229 0.308302 0.977868
            0.430058 0.316272 0.978004
            0.439768 0.324207 0.978110
            0.449401 0.332129 0.978200
            0.458996 0.339971 0.978287
            0.468590 0.347722 0.978377
            0.478245 0.355355 0.978492
            0.488019 0.362824 0.978640
            0.497946 0.370147 0.978828
            0.508073 0.377305 0.979061
            0.518451 0.384258 0.979347
            0.529120 0.391021 0.979691
            0.540096 0.397551 0.980097
            0.551427 0.403865 0.980569
            0.563108 0.409948 0.981113
            0.575134 0.415789 0.981731
            0.587510 0.421405 0.982418
            0.600220 0.426814 0.983172
            0.613223 0.432013 0.983998
            0.626503 0.437003 0.984886
            0.640016 0.441823 0.985827
            0.653712 0.446473 0.986823
            0.667569 0.450995 0.987859
            0.681527 0.455396 0.988937
            0.695577 0.459695 0.990045
            0.709665 0.463920 0.991178
            0.723765 0.468069 0.992333
            0.737851 0.472183 0.993503
            0.751904 0.476241 0.994667
            0.765890 0.480264 0.995824
            0.779793 0.484245 0.996963
            0.793606 0.488206 0.998069
            0.807288 0.492107 0.999125
            0.820827 0.495947 1.000000
            0.834201 0.499717 1.000000
            0.847361 0.503390 1.000000
            0.860287 0.506933 1.000000
            0.872915 0.510299 1.000000
            0.885202 0.513454 1.000000
            0.897090 0.516325 1.000000
            0.908507 0.518885 1.000000
            0.919391 0.521035 1.000000
            0.929679 0.522735 0.998520
            0.939301 0.523911 0.995999
            0.948193 0.524505 0.992736
            0.956308 0.524462 0.988644
            0.963596 0.523740 0.983677
            0.970035 0.522293 0.977792];
        
    case 'chlorophyll'
        Data=[     0.8967    0.9640    0.8668
            0.8886    0.9581    0.8578
            0.8805    0.9523    0.8487
            0.8724    0.9464    0.8397
            0.8643    0.9406    0.8307
            0.8562    0.9348    0.8217
            0.8481    0.9290    0.8127
            0.8400    0.9232    0.8037
            0.8319    0.9174    0.7948
            0.8239    0.9116    0.7858
            0.8158    0.9058    0.7769
            0.8077    0.9000    0.7680
            0.7997    0.8942    0.7591
            0.7917    0.8884    0.7502
            0.7836    0.8826    0.7413
            0.7756    0.8768    0.7325
            0.7676    0.8711    0.7236
            0.7596    0.8653    0.7148
            0.7515    0.8595    0.7060
            0.7435    0.8538    0.6972
            0.7355    0.8480    0.6885
            0.7275    0.8423    0.6797
            0.7196    0.8365    0.6710
            0.7116    0.8308    0.6622
            0.7036    0.8251    0.6535
            0.6956    0.8193    0.6448
            0.6877    0.8136    0.6361
            0.6797    0.8079    0.6275
            0.6717    0.8022    0.6188
            0.6638    0.7965    0.6102
            0.6559    0.7908    0.6016
            0.6479    0.7851    0.5930
            0.6400    0.7794    0.5844
            0.6320    0.7737    0.5758
            0.6241    0.7680    0.5672
            0.6162    0.7623    0.5587
            0.6082    0.7566    0.5502
            0.6003    0.7510    0.5417
            0.5924    0.7453    0.5332
            0.5845    0.7396    0.5247
            0.5766    0.7340    0.5162
            0.5686    0.7283    0.5078
            0.5607    0.7227    0.4993
            0.5528    0.7170    0.4909
            0.5449    0.7114    0.4825
            0.5370    0.7058    0.4741
            0.5290    0.7001    0.4657
            0.5211    0.6945    0.4573
            0.5132    0.6889    0.4490
            0.5053    0.6833    0.4406
            0.4974    0.6777    0.4323
            0.4894    0.6721    0.4240
            0.4815    0.6665    0.4157
            0.4735    0.6609    0.4074
            0.4656    0.6553    0.3991
            0.4576    0.6497    0.3909
            0.4497    0.6441    0.3826
            0.4417    0.6385    0.3744
            0.4337    0.6330    0.3661
            0.4257    0.6274    0.3579
            0.4177    0.6218    0.3497
            0.4097    0.6163    0.3415
            0.4017    0.6108    0.3333
            0.3937    0.6053    0.3251
            0.3856    0.5999    0.3169
            0.3776    0.5945    0.3088
            0.3696    0.5891    0.3006
            0.3616    0.5838    0.2926
            0.3536    0.5786    0.2845
            0.3457    0.5735    0.2765
            0.3378    0.5685    0.2686
            0.3299    0.5637    0.2607
            0.3221    0.5590    0.2529
            0.3144    0.5545    0.2452
            0.3069    0.5502    0.2376
            0.2994    0.5462    0.2301
            0.2921    0.5425    0.2228
            0.2850    0.5391    0.2157
            0.2781    0.5361    0.2088
            0.2714    0.5335    0.2020
            0.2650    0.5314    0.1956
            0.2589    0.5297    0.1894
            0.2530    0.5284    0.1835
            0.2476    0.5277    0.1779
            0.2424    0.5275    0.1726
            0.2377    0.5279    0.1676
            0.2333    0.5288    0.1630
            0.2293    0.5302    0.1588
            0.2257    0.5321    0.1548
            0.2225    0.5346    0.1513
            0.2197    0.5375    0.1480
            0.2173    0.5409    0.1452
            0.2152    0.5448    0.1426
            0.2134    0.5490    0.1403
            0.2120    0.5536    0.1383
            0.2109    0.5586    0.1365
            0.2100    0.5638    0.1350
            0.2094    0.5693    0.1337
            0.2090    0.5751    0.1325
            0.2088    0.5811    0.1316
            0.2087    0.5872    0.1307
            0.2087    0.5935    0.1300
            0.2089    0.5999    0.1293
            0.2091    0.6064    0.1287
            0.2094    0.6129    0.1282
            0.2098    0.6196    0.1277
            0.2101    0.6263    0.1272
            0.2105    0.6331    0.1268
            0.2110    0.6399    0.1263
            0.2114    0.6468    0.1259
            0.2119    0.6536    0.1254
            0.2124    0.6605    0.1249
            0.2130    0.6674    0.1244
            0.2138    0.6743    0.1238
            0.2146    0.6812    0.1233
            0.2157    0.6882    0.1227
            0.2170    0.6951    0.1221
            0.2187    0.7020    0.1214
            0.2208    0.7088    0.1208
            0.2235    0.7157    0.1201
            0.2269    0.7225    0.1195
            0.2311    0.7293    0.1188
            0.2362    0.7359    0.1181
            0.2425    0.7426    0.1173
            0.2501    0.7491    0.1167
            0.2592    0.7555    0.1161
            0.2697    0.7618    0.1154
            0.2818    0.7679    0.1148
            0.2956    0.7739    0.1143
            0.3110    0.7796    0.1139
            0.3281    0.7852    0.1135
            0.3468    0.7904    0.1133
            0.3669    0.7954    0.1131
            0.3885    0.8000    0.1130
            0.4113    0.8043    0.1131
            0.4354    0.8082    0.1133
            0.4604    0.8117    0.1136
            0.4863    0.8147    0.1140
            0.5129    0.8173    0.1145
            0.5399    0.8193    0.1152
            0.5674    0.8208    0.1160
            0.5950    0.8217    0.1168
            0.6226    0.8221    0.1176
            0.6500    0.8218    0.1185
            0.6771    0.8209    0.1193
            0.7036    0.8194    0.1200
            0.7295    0.8172    0.1206
            0.7545    0.8144    0.1212
            0.7785    0.8110    0.1215
            0.8014    0.8069    0.1217
            0.8231    0.8023    0.1217
            0.8435    0.7970    0.1214
            0.8625    0.7912    0.1209
            0.8801    0.7849    0.1201
            0.8963    0.7780    0.1190
            0.9111    0.7707    0.1176
            0.9245    0.7630    0.1159
            0.9365    0.7549    0.1139
            0.9472    0.7465    0.1116
            0.9566    0.7377    0.1091
            0.9649    0.7287    0.1063
            0.9722    0.7195    0.1033
            0.9784    0.7100    0.0999
            0.9838    0.7004    0.0964
            0.9884    0.6906    0.0927
            0.9923    0.6806    0.0889
            0.9956    0.6706    0.0848
            0.9983    0.6604    0.0805
            1.0000    0.6502    0.0761
            1.0000    0.6398    0.0717
            1.0000    0.6294    0.0670
            1.0000    0.6189    0.0622
            1.0000    0.6083    0.0572
            1.0000    0.5976    0.0520
            1.0000    0.5869    0.0468
            1.0000    0.5760    0.0414
            1.0000    0.5651    0.0358
            1.0000    0.5541    0.0302
            1.0000    0.5429    0.0251
            1.0000    0.5317    0.0202
            1.0000    0.5204    0.0156
            1.0000    0.5090    0.0110
            1.0000    0.4974    0.0068
            1.0000    0.4857    0.0029
            1.0000    0.4739         0
            1.0000    0.4620         0
            1.0000    0.4499         0
            1.0000    0.4377         0
            1.0000    0.4254         0
            1.0000    0.4130         0
            1.0000    0.4004         0
            1.0000    0.3877         0
            1.0000    0.3750         0
            1.0000    0.3622         0
            1.0000    0.3493         0
            0.9982    0.3364         0
            0.9957    0.3235         0
            0.9928    0.3107         0
            0.9896    0.2979         0
            0.9861    0.2853         0
            0.9821    0.2728         0
            0.9777    0.2606         0
            0.9729    0.2486         0
            0.9677    0.2369         0
            0.9620    0.2256         0
            0.9560    0.2146         0
            0.9495    0.2042         0
            0.9427    0.1942         0
            0.9355    0.1846         0
            0.9279    0.1756         0
            0.9200    0.1671         0
            0.9118    0.1591         0
            0.9034    0.1515         0
            0.8947    0.1445         0
            0.8858    0.1380         0
            0.8768    0.1318         0
            0.8676    0.1260         0
            0.8582    0.1205         0
            0.8488    0.1154         0
            0.8393    0.1105         0
            0.8297    0.1058         0
            0.8200    0.1013         0
            0.8104    0.0969         0
            0.8007    0.0926         0
            0.7910    0.0884         0
            0.7813    0.0843         0
            0.7715    0.0802         0
            0.7618    0.0761         0
            0.7521    0.0720         0
            0.7425    0.0679         0
            0.7328    0.0637         0
            0.7232    0.0596         0
            0.7135    0.0552         0
            0.7039    0.0509         0
            0.6944    0.0464         0
            0.6848    0.0419         0
            0.6753    0.0372         0
            0.6658    0.0325         0
            0.6564    0.0281         0
            0.6469    0.0239         0
            0.6375    0.0199         0
            0.6282    0.0160         0
            0.6188    0.0123         0
            0.6095    0.0085         0
            0.6002    0.0051         0
            0.5909    0.0018         0
            0.5817         0         0
            0.5725         0         0
            0.5633         0         0
            0.5542         0         0
            0.5451         0         0
            0.5360         0         0
            0.5270         0         0
            0.5180         0         0
            0.5090         0         0
            0.5000         0         0];
        
        
        
    case 'BOD' % Created from Colorscvivis 
        Data=[0.086274509803921595, 0.0039215686274509803, 0.29803921568627501
            
        0.089823299319352859, 0.0064564620263742123, 0.31781134536328598
        
        0.093372088834784109, 0.0089913554252974443, 0.33758347504029695
        
        0.096920878350215373, 0.011526248824220677, 0.35735560471730793
        
        0.10046966786564662, 0.014061142223143908, 0.3771277343943189
        
        0.10401845738107789, 0.016596035622067143, 0.39689986407132988
        
        0.10756724689650915, 0.019130929020990374, 0.41667199374834085
        
        0.1111160364119404, 0.021665822419913605, 0.43644412342535183
        
        0.11340168241333767, 0.024661032165175043, 0.45340494373553814
        
        0.11218083286206953, 0.02893408342513443, 0.46256154886161188
        
        0.1109599833108014, 0.033207134685093821, 0.47171815398768568
        
        0.10973913375953327, 0.037480185945053208, 0.48087475911375943
        
        0.10851828420826513, 0.041753237205012603, 0.49003136423983323
        
        0.10729743465699701, 0.04602628846497199, 0.49918796936590698
        
        0.10607658510572887, 0.050299339724931377, 0.50834457449198078
        
        0.093310212417851993, 0.048761838157394419, 0.51941769883941225
        
        0.078354748165907859, 0.046122625387451474, 0.53085420525365912
        
        0.063399283913963725, 0.043483412617508529, 0.5422907116679061
        
        0.048443819662019591, 0.040844199847565592, 0.55372721808215308
        
        0.038521432139711945, 0.044422664692938367, 0.56425533930144023
        
        0.036708532985001138, 0.058019292781872872, 0.57331983507499429
        
        0.034895633830290332, 0.071615920870807384, 0.58238433084854824
        
        0.033082734675579525, 0.085212548959741896, 0.5914488266221023
        
        0.03153129829389241, 0.098939051599624878, 0.60052932764630806
        
        0.0343354355690541, 0.11482906748563648, 0.60987645189684703
        
        0.03713957284421579, 0.13071908337164809, 0.61922357614738599
        
        0.039943710119377487, 0.14660909925765969, 0.62857070039792495
        
        0.042747847394539178, 0.1624991151436713, 0.63791782464846392
        
        0.044817150705219091, 0.17590495710566681, 0.64481540715914276
        
        0.046767936262570996, 0.18891013888239441, 0.65131791513907888
        
        0.0487187218199229, 0.20191532065912202, 0.65782042311901501
        
        0.050669507377274804, 0.21492050243584959, 0.66432293109895124
        
        0.052620292934626708, 0.2279256842125772, 0.67082543907888736
        
        0.054571078491978599, 0.24093086598930472, 0.67732794705882349
        
        0.054901999999999999, 0.25087316671449372, 0.68168837160560747
        
        0.054901999999999999, 0.26018975245027703, 0.68561119061364306
        
        0.054901999999999999, 0.2695063381860604, 0.68953400962167855
        
        0.054901999999999999, 0.27882292392184366, 0.69345682862971414
        
        0.054901999999999999, 0.28813950965762702, 0.69737964763774973
        
        0.054901999999999999, 0.29745609539341039, 0.70130246664578533
        
        0.054901999999999999, 0.3067726811291937, 0.70522528565382092
        
        0.054901999999999999, 0.31608926686497696, 0.7091481046618564
        
        0.054569855699292166, 0.32428980131803758, 0.71246108664876739
        
        0.054171027288259704, 0.33226626783825564, 0.71565163257668207
        
        0.053772198877227242, 0.34024273435847369, 0.71884217850459686
        
        0.053373370466194781, 0.34821920087869168, 0.72203272443251154
        
        0.052974542055162312, 0.35619566739890979, 0.72522327036042633
        
        0.05257571364412985, 0.36417213391912784, 0.72841381628834101
        
        0.052176885233097388, 0.37214860043934583, 0.7316043622162558
        
        0.051778056822064933, 0.38012506695956383, 0.73479490814417048
        
        0.051379228411032464, 0.38810153347978193, 0.73798545407208527
        
        0.050980400000000002, 0.39607799999999999, 0.74117599999999995
        
        0.049134956862745098, 0.40715078431372548, 0.74548203921568623
        
        0.047289513725490208, 0.41822356862745091, 0.74978807843137252
        
        0.045444070588235297, 0.42929635294117646, 0.7540941176470588
        
        0.043598627450980393, 0.44036913725490195, 0.75840015686274509
        
        0.041753184313725489, 0.45144192156862745, 0.76270619607843138
        
        0.039907741176470599, 0.46251470588235288, 0.76701223529411755
        
        0.038446758823529409, 0.47358739215686274, 0.77054935294117644
        
        0.037216452941176467, 0.48466001960784316, 0.77362511764705877
        
        0.035986147058823525, 0.49573264705882353, 0.7767008823529411
        
        0.034755841176470598, 0.50680527450980384, 0.77977664705882355
        
        0.033525535294117642, 0.51787790196078431, 0.78285241176470588
        
        0.0322952294117647, 0.52895052941176468, 0.78592817647058821
        
        0.031372499999999998, 0.54020188756581855, 0.78911909257084201
        
        0.031372499999999998, 0.55198943782909249, 0.79265546285421007
        
        0.031372499999999998, 0.56377698809236632, 0.79619183313757802
        
        0.031372499999999998, 0.57556453835564048, 0.79972820342094608
        
        0.031372499999999998, 0.58735208861891441, 0.80326457370431403
        
        0.031372499999999998, 0.59913963888218846, 0.8068009439876821
        
        0.031372499999999998, 0.6109271891454624, 0.81033731427105005
        
        0.030686504498252627, 0.62391801599284191, 0.81348001062057884
        
        0.029536055465076611, 0.63772352173731295, 0.81635616987425608
        
        0.028385606431900596, 0.65152902748178387, 0.8192323291279332
        
        0.027235157398724598, 0.66533453322625469, 0.82210848838161033
        
        0.026084708365548565, 0.67914003897072583, 0.82498464763528756
        
        0.024934259332372546, 0.69294554471519687, 0.82786080688896468
        
        0.023783810299196531, 0.7067510504596678, 0.83073696614264192
        
        0.026665552705750366, 0.72010848542668249, 0.83361301115083775
        
        0.030692146324308684, 0.73333870064723861, 0.83648902372156642
        
        0.034718739942867005, 0.74656891586779472, 0.83936503629229509
        
        0.038745333561425323, 0.75979913108835084, 0.84224104886302376
        
        0.042771927179983585, 0.77302934630890674, 0.84511706143375243
        
        0.046798520798541958, 0.78625956152946308, 0.8479930740044811
        
        0.050825114417100276, 0.7994897767500192, 0.85086908657520977
        
        0.0543243452656655, 0.80936307403051688, 0.85432397937398219
        
        0.057802422583076996, 0.81910171890044237, 0.85780209216786041
        
        0.061280499900488491, 0.82884036377036796, 0.86128020496173863
        
        0.064758577217899987, 0.83857900864029344, 0.86475831775561685
        
        0.068236654535311483, 0.84831765351021904, 0.86823643054949495
        
        0.082455993738379557, 0.85780841031456811, 0.87010360857440328
        
        0.11909686111324025, 0.8667817205372278, 0.86860808865098116
        
        0.15573772848810041, 0.87575503075988737, 0.86711256872755915
        
        0.19237859586296055, 0.88472834098254693, 0.86561704880413703
        
        0.22901946323782069, 0.89370165120520639, 0.86412152888071492
        
        0.26553302715502125, 0.90264099604988157, 0.86294900748506587
        
        0.30057390134620043, 0.91118741784189505, 0.86551304299145471
        
        0.33561477553737962, 0.91973383963390853, 0.86807707849784355
        
        0.3706556497285583, 0.92828026142592179, 0.87064111400423227
        
        0.40569652391973798, 0.93682668321793539, 0.87320514951062123
        
        0.43386584145924617, 0.94280816937012524, 0.87178976396010066
        
        0.4549143991909399, 0.94613170019435522, 0.86625064008406993
        
        0.47596295692263368, 0.9494552310185852, 0.86071151620803921
        
        0.49701151465432741, 0.95277876184281518, 0.85517239233200837
        
        0.51806007238602114, 0.95610229266704516, 0.84963326845597764
        
        0.53910863011771493, 0.95942582349127514, 0.84409414457994691
        
        0.56015718784940838, 0.96274935431550512, 0.8385550207039163
        
        0.58006685842923267, 0.96607285166410828, 0.83597742583205414
        
        0.59834625403857589, 0.96939630109358566, 0.83763915054679283
        
        0.61662564964791899, 0.97271975052306292, 0.83930087526153152
        
        0.63490504525726221, 0.9760431999525403, 0.84096259997627021
        
        0.65318444086660543, 0.97936664938201767, 0.8426243246910089
        
        0.67352015427740419, 0.98039200000000004, 0.84640290459329559
        
        0.6947733420448079, 0.98039200000000004, 0.85112596902508764
        
        0.71602652981221127, 0.98039200000000004, 0.85584903345687968
        
        0.73727971757961519, 0.98039200000000004, 0.86057209788867173
        
        0.75853290534701889, 0.98039200000000004, 0.86529516232046388
        
        0.77828852745769239, 0.98039200000000004, 0.87091158041900441
        
        0.79743111525366339, 0.98039200000000004, 0.8768936963087709
        
        0.81657370304963439, 0.98039200000000004, 0.8828758121985375
        
        0.83539909466089479, 0.98138553201332723, 0.89124249862091298
        
        0.85380704940791496, 0.98368657309566698, 0.90274733012091402
        
        0.87221500415493491, 0.98598761417800673, 0.91425216162091483
        
        0.89040416437988268, 0.98823529151651845, 0.92559424208733632
        
        0.89937725400949331, 0.98823517935056127, 0.93008091776242496
        
        0.90835034363910405, 0.98823506718460419, 0.93456759343751372
        
        0.92924904176201384, 0.99035195568044154, 0.94360569063276134
        
        0.96795994946605657, 0.99563081779737062, 0.9594418370893697
        
        0.99909044689499049, 0.99909044689499049, 0.96739488534612239
        
        0.99381232065091951, 0.99381232065091951, 0.93748550329638725
        
        0.98853419440684864, 0.98853419440684864, 0.9075761212466521
        
        0.98911712032057086, 0.98470798930595138, 0.88295487166515496
        
        0.99005188208204709, 0.98096894226004638, 0.85865106586677264
        
        0.99098664384352342, 0.97722989521414116, 0.83434726006838889
        
        0.99192140560499964, 0.97349084816823606, 0.81004345427000579
        
        0.99215689286729924, 0.97082777678071608, 0.78412687161735894
        
        0.99215693313170061, 0.96852698835772078, 0.75766726405952378
        
        0.99215697339610198, 0.96622619993472558, 0.73120765650168851
        
        0.99180702644888941, 0.9640061421312186, 0.70742618994752837
        
        0.99077547767914675, 0.96194330760773794, 0.68886041622019711
        
        0.9897439289094041, 0.95988047308425717, 0.67029464249286574
        
        0.98871238013966145, 0.9578176385607764, 0.65172886876553449
        
        0.98748761435661658, 0.95312597648974562, 0.63043125347580697
        
        0.98609640471553195, 0.94616975090198929, 0.60678033481270222
        
        0.98470519507444743, 0.93921352531423286, 0.58312941614959735
        
        0.98331398543336279, 0.93225729972647653, 0.55947849748649259
        
        0.98192277579227827, 0.9253010741387202, 0.53582757882338772
        
        0.98053156615119363, 0.91834484855096377, 0.51217666016028296
        
        0.97855711513172161, 0.91091929676473216, 0.4939020283846548
        
        0.97651762901634886, 0.9034414121443467, 0.47622688653939166
        
        0.97447814290097612, 0.89596352752396158, 0.45855174469412902
        
        0.97243865678560348, 0.88848564290357634, 0.44087660284886637
        
        0.97039917067023074, 0.88100775828319111, 0.42320146100360367
        
        0.96813694417699969, 0.87323578487721842, 0.40568667439824913
        
        0.96439805791011757, 0.86351414664740667, 0.38923496461148988
        
        0.96065917164323544, 0.85379250841759502, 0.37278325482473063
        
        0.95692028537635332, 0.84407087018778337, 0.35633154503797143
        
        0.95318139910947119, 0.83434923195797162, 0.33987983525121218
        
        0.94944251284258907, 0.82462759372815997, 0.32342812546445299
        
        0.94487457376269657, 0.81358023072852803, 0.31244897771191388
        
        0.94020101163742709, 0.8023639676475951, 0.30216704569641628
        
        0.93552744951215761, 0.79114770456666217, 0.29188511368091874
        
        0.93085388738688812, 0.77993144148572935, 0.28160318166542114
        
        0.92721158845925389, 0.76986976537776175, 0.27359016194241303
        
        0.92402941176470599, 0.76032323529411794, 0.26658947058823546
        
        0.92084723507015787, 0.75077670521047357, 0.25958877923405749
        
        0.91766505837560985, 0.74123017512682954, 0.25258808787987969
        
        0.91448288168106173, 0.73168364504318539, 0.24558739652570191
        
        0.91130070498651372, 0.72213711495954136, 0.23858670517152414
        
        0.90791793265013709, 0.71198889413907662, 0.23152186530027419
        
        0.90435703065666395, 0.70130636976320604, 0.2244000613133279
        
        0.90079612866319081, 0.69062384538733534, 0.21727825732638159
        
        0.89723522666971767, 0.67994132101146476, 0.21015645333943531
        
        0.89367432467624452, 0.66925879663559418, 0.20303464935248902
        
        0.89013568187274905, 0.65862294299377211, 0.195937261121023
        
        0.88753464145658256, 0.64995295566182276, 0.18986831420524011
        
        0.88493360104041618, 0.64128296832987342, 0.18379936728945723
        
        0.8823325606242497, 0.63261298099792407, 0.17773042037367434
        
        0.87973152020808321, 0.62394299366597472, 0.17166147345789146
        
        0.87781586274509804, 0.61407168627450981, 0.16770433333333334
        
        0.87658558823529409, 0.60299905882352978, 0.16585900000000006
        
        0.87535531372549014, 0.59192643137254908, 0.16401366666666667
        
        0.87412503921568629, 0.5808538039215686, 0.16216833333333333
        
        0.87289476470588234, 0.56978117647058824, 0.16032299999999999
        
        0.8716644901960785, 0.55870854901960776, 0.15847766666666666
        
        0.87020352941176471, 0.54755901960784314, 0.15663231372549019
        
        0.86712776470588238, 0.53587117647058824, 0.15478682352941175
        
        0.86405200000000004, 0.52418333333333333, 0.15294133333333335
        
        0.8609762352941176, 0.51249549019607843, 0.15109584313725491
        
        0.85790047058823526, 0.50080764705882352, 0.14925035294117647
        
        0.85482470588235293, 0.48911980392156862, 0.14740486274509804
        
        0.8517489411764706, 0.47743196078431371, 0.1455593725490196
        
        0.84867329411764703, 0.46712823529411762, 0.14371388235294119
        
        0.84559768627450982, 0.45728588235294115, 0.14186839215686275
        
        0.8425220784313725, 0.44744352941176468, 0.14002290196078432
        
        0.83944647058823529, 0.43760117647058822, 0.13817741176470588
        
        0.83637086274509809, 0.42775882352941202, 0.1363319215686275
        
        0.83329525490196077, 0.41791647058823528, 0.13448643137254904
        
        0.83021958823529418, 0.4078434117647059, 0.13217958823529413
        
        0.82714382352941174, 0.39738584313725489, 0.12910382352941177
        
        0.8240680588235294, 0.38692827450980394, 0.12602805882352941
        
        0.82099229411764707, 0.37647070588235293, 0.12295229411764706
        
        0.81791652941176474, 0.36601313725490198, 0.11987652941176472
        
        0.81484076470588229, 0.35555556862745097, 0.11680076470588235
        
        0.81176499999999996, 0.34509800000000002, 0.11372500000000001
        
        0.80807401960784309, 0.3327950980392157, 0.11064932941176471
        
        0.80438303921568621, 0.32049219607843138, 0.10757365882352941
        
        0.80069205882352945, 0.30818929411764706, 0.10449798823529412
        
        0.79700107843137258, 0.29588639215686274, 0.10142231764705882
        
        0.79331009803921571, 0.28358349019607842, 0.098346647058823525
        
        0.78961911764705883, 0.27128058823529411, 0.095270976470588228
        
        0.78362137254901965, 0.25859317647058822, 0.092195266666666664
        
        0.77623956862745114, 0.24567505882352977, 0.089119533333333417
        
        0.7688577647058823, 0.23275694117647058, 0.086043800000000004
        
        0.76147596078431368, 0.21983882352941175, 0.082968066666666673
        
        0.75409415686274506, 0.20692070588235295, 0.079892333333333329
        
        0.74671235294117644, 0.19400258823529412, 0.076816599999999999
        
        0.73917678431372547, 0.18200717647058823, 0.074048439215686282
        
        0.73117992156862743, 0.17277988235294117, 0.072202996078431378
        
        0.7231830588235294, 0.16355258823529412, 0.070357552941176474
        
        0.71518619607843137, 0.15432529411764706, 0.06851210980392157
        
        0.70718933333333334, 0.145098, 0.066666666666666666
        
        0.6991924705882353, 0.13587070588235295, 0.064821223529411762
        
        0.69119560784313727, 0.12664341176470587, 0.062975780392156858
        
        0.68050745098039223, 0.11687795490196078, 0.060053833333333334
        
        0.66943482352941175, 0.10703561764705882, 0.056978099999999997
        
        0.65836219607843138, 0.097193280392156856, 0.053902366666666666
        
        0.64728956862745102, 0.087350943137254902, 0.050826633333333336
        
        0.63621694117647098, 0.077508605882353226, 0.047750900000000089
        
        0.62514431372549029, 0.067666268627450982, 0.044675166666666669
        
        0.6136913649751099, 0.059786774105627376, 0.045438214552115892
        
        0.6018580949253296, 0.053870122316882124, 0.050040043656347677
        
        0.5900248248755493, 0.047953470528136879, 0.054641872760579462
        
        0.57819155482576901, 0.042036818739391628, 0.059243701864811246
        
        0.56635828477598882, 0.036120166950646376, 0.063845530969043024
        
        0.55452501472620852, 0.030203515161901132, 0.068447360073274816
        
        0.54102197982964451, 0.026251307171840824, 0.072587694845194714
        
        0.5260663992311696, 0.024007989150282301, 0.076326570926600737
        
        0.5111108186326947, 0.021764671128723774, 0.08006544700800676
        
        0.49615523803421974, 0.019521353107165247, 0.083804323089412783
        
        0.48119965743574478, 0.017278035085606727, 0.087543199170818792
        
        0.46711278820207192, 0.015107094915631003, 0.090775313453865317
        
        0.45514807147167302, 0.013112964725519002, 0.092769472459099486
        
        0.44318335474127407, 0.011118834535407001, 0.094763631464333656
        
        0.43121863801087557, 0.009124704345295059, 0.096757790469567784
        
        0.41925392128047628, 0.0071305741551830008, 0.098751949474802009
        
        0.40728920455007739, 0.0051364439650710015, 0.10074610848003619
        
        0.39013769849135815, 0.0037389347229546776, 0.10049990825007929
        
        0.36489989794464422, 0.0032715678825853429, 0.096760949130584084
        
        0.33966209739793035, 0.0028042010422160086, 0.093021990011088865
        
        0.31442429685121648, 0.0023368342018466734, 0.089283030891593659
        
        0.28918649630450255, 0.001869467361477339, 0.08554407177209844
        
        0.26394869575778868, 0.0014021005211080043, 0.081805112652603235
        
        0.23871089521107478, 0.00093473368073866909, 0.078066153533108029
        
        0.2134730946643609, 0.00046736684036933476, 0.074327194413612824
        
        0.188235294117647, 0.0, 0.070588235294117604];
    
    case 'mBOD' % Created from ColorSciVis
        % BOD color without "whites"
        Data=[0.0862745098039216, 0.00392156862745098, 0.298039215686275
            0.0898232993193529, 0.00645646202637421, 0.317811345363286
            0.0933720888347841, 0.00899135542529744, 0.337583475040297
            0.0969208783502154, 0.0115262488242207, 0.357355604717308
            0.100469667865647, 0.0140611422231439, 0.377127734394319
            0.104018457381078, 0.0165960356220671, 0.396899864071330
            0.107567246896509, 0.0191309290209904, 0.416671993748341
            0.111116036411940, 0.0216658224199136, 0.436444123425352
            0.113401682413338, 0.0246610321651750, 0.453404943735538
            0.112180832862070, 0.0289340834251344, 0.462561548861612
            0.110959983310801, 0.0332071346850938, 0.471718153987686
            0.109739133759533, 0.0374801859450532, 0.480874759113759
            0.108518284208265, 0.0417532372050126, 0.490031364239833
            0.107297434656997, 0.0460262884649720, 0.499187969365907
            0.106076585105729, 0.0502993397249314, 0.508344574491981
            0.0933102124178520, 0.0487618381573944, 0.519417698839412
            0.0783547481659079, 0.0461226253874515, 0.530854205253659
            0.0633992839139637, 0.0434834126175085, 0.542290711667906
            0.0484438196620196, 0.0408441998475656, 0.553727218082153
            0.0385214321397119, 0.0444226646929384, 0.564255339301440
            0.0367085329850011, 0.0580192927818729, 0.573319835074994
            0.0348956338302903, 0.0716159208708074, 0.582384330848548
            0.0330827346755795, 0.0852125489597419, 0.591448826622102
            0.0315312982938924, 0.0989390515996249, 0.600529327646308
            0.0343354355690541, 0.114829067485636, 0.609876451896847
            0.0371395728442158, 0.130719083371648, 0.619223576147386
            0.0399437101193775, 0.146609099257660, 0.628570700397925
            0.0427478473945392, 0.162499115143671, 0.637917824648464
            0.0448171507052191, 0.175904957105667, 0.644815407159143
            0.0467679362625710, 0.188910138882394, 0.651317915139079
            0.0487187218199229, 0.201915320659122, 0.657820423119015
            0.0506695073772748, 0.214920502435850, 0.664322931098951
            0.0526202929346267, 0.227925684212577, 0.670825439078887
            0.0545710784919786, 0.240930865989305, 0.677327947058824
            0.0549020000000000, 0.250873166714494, 0.681688371605608
            0.0549020000000000, 0.260189752450277, 0.685611190613643
            0.0549020000000000, 0.269506338186060, 0.689534009621679
            0.0549020000000000, 0.278822923921844, 0.693456828629714
            0.0549020000000000, 0.288139509657627, 0.697379647637750
            0.0549020000000000, 0.297456095393410, 0.701302466645785
            0.0549020000000000, 0.306772681129194, 0.705225285653821
            0.0549020000000000, 0.316089266864977, 0.709148104661856
            0.0545698556992922, 0.324289801318038, 0.712461086648767
            0.0541710272882597, 0.332266267838256, 0.715651632576682
            0.0537721988772272, 0.340242734358474, 0.718842178504597
            0.0533733704661948, 0.348219200878692, 0.722032724432512
            0.0529745420551623, 0.356195667398910, 0.725223270360426
            0.0525757136441299, 0.364172133919128, 0.728413816288341
            0.0521768852330974, 0.372148600439346, 0.731604362216256
            0.0517780568220649, 0.380125066959564, 0.734794908144171
            0.0513792284110325, 0.388101533479782, 0.737985454072085
            0.0509804000000000, 0.396078000000000, 0.741176000000000
            0.0491349568627451, 0.407150784313726, 0.745482039215686
            0.0472895137254902, 0.418223568627451, 0.749788078431373
            0.0454440705882353, 0.429296352941176, 0.754094117647059
            0.0435986274509804, 0.440369137254902, 0.758400156862745
            0.0417531843137255, 0.451441921568627, 0.762706196078431
            0.0399077411764706, 0.462514705882353, 0.767012235294118
            0.0384467588235294, 0.473587392156863, 0.770549352941176
            0.0372164529411765, 0.484660019607843, 0.773625117647059
            0.0359861470588235, 0.495732647058824, 0.776700882352941
            0.0347558411764706, 0.506805274509804, 0.779776647058824
            0.0335255352941176, 0.517877901960784, 0.782852411764706
            0.0322952294117647, 0.528950529411765, 0.785928176470588
            0.0313725000000000, 0.540201887565819, 0.789119092570842
            0.0313725000000000, 0.551989437829093, 0.792655462854210
            0.0313725000000000, 0.563776988092366, 0.796191833137578
            0.0313725000000000, 0.575564538355641, 0.799728203420946
            0.0313725000000000, 0.587352088618914, 0.803264573704314
            0.0313725000000000, 0.599139638882189, 0.806800943987682
            0.0313725000000000, 0.610927189145462, 0.810337314271050
            0.0306865044982526, 0.623918015992842, 0.813480010620579
            0.0295360554650766, 0.637723521737313, 0.816356169874256
            0.0283856064319006, 0.651529027481784, 0.819232329127933
            0.0272351573987246, 0.665334533226255, 0.822108488381610
            0.0260847083655486, 0.679140038970726, 0.824984647635288
            0.0249342593323725, 0.692945544715197, 0.827860806888965
            0.0237838102991965, 0.706751050459668, 0.830736966142642
            0.0266655527057504, 0.720108485426683, 0.833613011150838
            0.0306921463243087, 0.733338700647239, 0.836489023721566
            0.0347187399428670, 0.746568915867795, 0.839365036292295
            0.0387453335614253, 0.759799131088351, 0.842241048863024
            0.0427719271799836, 0.773029346308907, 0.845117061433752
            0.0467985207985420, 0.786259561529463, 0.847993074004481
            0.0508251144171003, 0.799489776750019, 0.850869086575210
            0.0543243452656655, 0.809363074030517, 0.854323979373982
            0.0578024225830770, 0.819101718900442, 0.857802092167860
            0.0612804999004885, 0.828840363770368, 0.861280204961739
            0.0647585772179000, 0.838579008640293, 0.864758317755617
            0.0682366545353115, 0.848317653510219, 0.868236430549495
            0.0824559937383796, 0.857808410314568, 0.870103608574403
            0.119096861113240, 0.866781720537228, 0.868608088650981
            0.155737728488100, 0.875755030759887, 0.867112568727559
            0.192378595862961, 0.884728340982547, 0.865617048804137
            0.229019463237821, 0.893701651205206, 0.864121528880715
            0.265533027155021, 0.902640996049882, 0.862949007485066
            0.300573901346200, 0.911187417841895, 0.865513042991455
            0.335614775537380, 0.919733839633909, 0.868077078497844
            0.370655649728558, 0.928280261425922, 0.870641114004232
            0.405696523919738, 0.936826683217935, 0.873205149510621
            0.433865841459246, 0.942808169370125, 0.871789763960101
            0.454914399190940, 0.946131700194355, 0.866250640084070
            0.475962956922634, 0.949455231018585, 0.860711516208039
            0.497011514654327, 0.952778761842815, 0.855172392332008
            0.518060072386021, 0.956102292667045, 0.849633268455978
            0.539108630117715, 0.959425823491275, 0.844094144579947
            0.560157187849408, 0.962749354315505, 0.838555020703916
            0.580066858429233, 0.966072851664108, 0.835977425832054
            0.598346254038576, 0.969396301093586, 0.837639150546793
            0.616625649647919, 0.972719750523063, 0.839300875261532
            0.634905045257262, 0.976043199952540, 0.840962599976270
            0.653184440866605, 0.979366649382018, 0.842624324691009
            0.673520154277404, 0.980392000000000, 0.846402904593296
            0.694773342044808, 0.980392000000000, 0.851125969025088
            0.983313985433363, 0.932257299726477, 0.559478497486493
            0.981922775792278, 0.925301074138720, 0.535827578823388
            0.980531566151194, 0.918344848550964, 0.512176660160283
            0.978557115131722, 0.910919296764732, 0.493902028384655
            0.976517629016349, 0.903441412144347, 0.476226886539392
            0.974478142900976, 0.895963527523962, 0.458551744694129
            0.972438656785604, 0.888485642903576, 0.440876602848866
            0.970399170670231, 0.881007758283191, 0.423201461003604
            0.968136944177000, 0.873235784877218, 0.405686674398249
            0.964398057910118, 0.863514146647407, 0.389234964611490
            0.960659171643235, 0.853792508417595, 0.372783254824731
            0.956920285376353, 0.844070870187783, 0.356331545037971
            0.953181399109471, 0.834349231957972, 0.339879835251212
            0.949442512842589, 0.824627593728160, 0.323428125464453
            0.944874573762697, 0.813580230728528, 0.312448977711914
            0.940201011637427, 0.802363967647595, 0.302167045696416
            0.935527449512158, 0.791147704566662, 0.291885113680919
            0.930853887386888, 0.779931441485729, 0.281603181665421
            0.927211588459254, 0.769869765377762, 0.273590161942413
            0.924029411764706, 0.760323235294118, 0.266589470588235
            0.920847235070158, 0.750776705210474, 0.259588779234058
            0.917665058375610, 0.741230175126830, 0.252588087879880
            0.914482881681062, 0.731683645043185, 0.245587396525702
            0.911300704986514, 0.722137114959541, 0.238586705171524
            0.907917932650137, 0.711988894139077, 0.231521865300274
            0.904357030656664, 0.701306369763206, 0.224400061313328
            0.900796128663191, 0.690623845387335, 0.217278257326382
            0.897235226669718, 0.679941321011465, 0.210156453339435
            0.893674324676245, 0.669258796635594, 0.203034649352489
            0.890135681872749, 0.658622942993772, 0.195937261121023
            0.887534641456583, 0.649952955661823, 0.189868314205240
            0.884933601040416, 0.641282968329873, 0.183799367289457
            0.882332560624250, 0.632612980997924, 0.177730420373674
            0.879731520208083, 0.623942993665975, 0.171661473457891
            0.877815862745098, 0.614071686274510, 0.167704333333333
            0.876585588235294, 0.602999058823530, 0.165859000000000
            0.875355313725490, 0.591926431372549, 0.164013666666667
            0.874125039215686, 0.580853803921569, 0.162168333333333
            0.872894764705882, 0.569781176470588, 0.160323000000000
            0.871664490196079, 0.558708549019608, 0.158477666666667
            0.870203529411765, 0.547559019607843, 0.156632313725490
            0.867127764705882, 0.535871176470588, 0.154786823529412
            0.864052000000000, 0.524183333333333, 0.152941333333333
            0.860976235294118, 0.512495490196078, 0.151095843137255
            0.857900470588235, 0.500807647058824, 0.149250352941176
            0.854824705882353, 0.489119803921569, 0.147404862745098
            0.851748941176471, 0.477431960784314, 0.145559372549020
            0.848673294117647, 0.467128235294118, 0.143713882352941
            0.845597686274510, 0.457285882352941, 0.141868392156863
            0.842522078431373, 0.447443529411765, 0.140022901960784
            0.839446470588235, 0.437601176470588, 0.138177411764706
            0.836370862745098, 0.427758823529412, 0.136331921568628
            0.833295254901961, 0.417916470588235, 0.134486431372549
            0.830219588235294, 0.407843411764706, 0.132179588235294
            0.827143823529412, 0.397385843137255, 0.129103823529412
            0.824068058823529, 0.386928274509804, 0.126028058823529
            0.820992294117647, 0.376470705882353, 0.122952294117647
            0.817916529411765, 0.366013137254902, 0.119876529411765
            0.814840764705882, 0.355555568627451, 0.116800764705882
            0.811765000000000, 0.345098000000000, 0.113725000000000
            0.808074019607843, 0.332795098039216, 0.110649329411765
            0.804383039215686, 0.320492196078431, 0.107573658823529
            0.800692058823530, 0.308189294117647, 0.104497988235294
            0.797001078431373, 0.295886392156863, 0.101422317647059
            0.793310098039216, 0.283583490196078, 0.0983466470588235
            0.789619117647059, 0.271280588235294, 0.0952709764705882
            0.783621372549020, 0.258593176470588, 0.0921952666666667
            0.776239568627451, 0.245675058823530, 0.0891195333333334
            0.768857764705882, 0.232756941176471, 0.0860438000000000
            0.761475960784314, 0.219838823529412, 0.0829680666666667
            0.754094156862745, 0.206920705882353, 0.0798923333333333
            0.746712352941176, 0.194002588235294, 0.0768166000000000
            0.739176784313726, 0.182007176470588, 0.0740484392156863
            0.731179921568627, 0.172779882352941, 0.0722029960784314
            0.723183058823529, 0.163552588235294, 0.0703575529411765
            0.715186196078431, 0.154325294117647, 0.0685121098039216
            0.707189333333333, 0.145098000000000, 0.0666666666666667
            0.699192470588235, 0.135870705882353, 0.0648212235294118
            0.691195607843137, 0.126643411764706, 0.0629757803921569
            0.680507450980392, 0.116877954901961, 0.0600538333333333
            0.669434823529412, 0.107035617647059, 0.0569781000000000
            0.658362196078431, 0.0971932803921569, 0.0539023666666667
            0.647289568627451, 0.0873509431372549, 0.0508266333333333
            0.636216941176471, 0.0775086058823532, 0.0477509000000001
            0.625144313725490, 0.0676662686274510, 0.0446751666666667
            0.613691364975110, 0.0597867741056274, 0.0454382145521159
            0.601858094925330, 0.0538701223168821, 0.0500400436563477
            0.590024824875549, 0.0479534705281369, 0.0546418727605795
            0.578191554825769, 0.0420368187393916, 0.0592437018648112
            0.566358284775989, 0.0361201669506464, 0.0638455309690430
            0.554525014726209, 0.0302035151619011, 0.0684473600732748
            0.541021979829645, 0.0262513071718408, 0.0725876948451947
            0.526066399231170, 0.0240079891502823, 0.0763265709266007
            0.511110818632695, 0.0217646711287238, 0.0800654470080068
            0.496155238034220, 0.0195213531071652, 0.0838043230894128
            0.481199657435745, 0.0172780350856067, 0.0875431991708188
            0.467112788202072, 0.0151070949156310, 0.0907753134538653
            0.455148071471673, 0.0131129647255190, 0.0927694724590995
            0.443183354741274, 0.0111188345354070, 0.0947636314643337
            0.431218638010876, 0.00912470434529506, 0.0967577904695678
            0.419253921280476, 0.00713057415518300, 0.0987519494748020
            0.407289204550077, 0.00513644396507100, 0.100746108480036
            0.390137698491358, 0.00373893472295468, 0.100499908250079
            0.364899897944644, 0.00327156788258534, 0.0967609491305841
            0.339662097397930, 0.00280420104221601, 0.0930219900110889
            0.314424296851217, 0.00233683420184667, 0.0892830308915937
            0.289186496304503, 0.00186946736147734, 0.0855440717720984
            0.263948695757789, 0.00140210052110800, 0.0818051126526032
            0.238710895211075, 0.000934733680738669, 0.0780661535331080
            0.213473094664361, 0.000467366840369335, 0.0743271944136128
            0.188235294117647, 0, 0.0705882352941176];
        
        % Interpolate into 256-element colormap
        [x,y]=meshgrid([1 2 3],linspace(1,length(r),size(Data,1)));
        [xn,yn]=meshgrid([1 2 3],1:length(r));

        Data=interp2(x,y,Data,xn,yn);
end

end

function colmap_demo

clf;
axp=[.1 .1 .2 .08];
dx=[.29 0 0 0];
dy=[0 .15 0 0];

%  dx . dy
examples={...
    0,6,'s_colmap(''jet'',256)',      {'Perceptually uniform jet replacement','with diverging luminance'}
    1,6,'s_colmap(''blue'',256)',     'Good for bathymetry'
    2,6,'s_colmap(''land'',256)',     'Land from coastal wetlands to mountains'
    0,5,'s_colmap(''diverging'',256)',{'Currents, echo-sounder images','diverging luminance with a ''zero'''}
    1,5,'s_colmap(''water'',256)',    'Another bathymetry map'
    2,5,'s_colmap(''gland'',256)',    'Land with more green'
    0,4,'s_colmap(''odv'',256)',      'Isoluminant'
    1,4,'s_colmap(''green'',256)',    'Chlorophyll? Land?'
    2,4,'s_colmap(''bland'',256)',    'Land without green'
    1,3,'s_colmap(''chlorophyl'',256)','Enhanced Chlorophyll emphasizing high values'
    0,3,'s_colmap(''odv'',256).*repmat(1-.2*cos([0:255]''/64*2*pi).^4,1,3)',      'Isoluminant (with added shading)'
    0,1,'s_colmap(''jet'',''step'',10)',{'Banded continuous map (256 colours)','sort of like contouring'}
    1,1,'s_colmap(''jet'',10)',        'A few discrete steps (10 colours)'
    2,3,'[s_colmap(''blues'',64);s_colmap(''gland'',128)]',{'Complex water + land example','must use ''caxis'' to get coastline correct'}
    2,1,'s_colmap(''BOD'',256)',{'Blue-orange diverging'}
    0,0,'s_colmap(''mBOD'',256)',{'Modified BOD with white removed, colorblind-friendly jet alternative'}
    };


for k=1:size(examples,1)
    ax(k)=axes('pos',axp+dx*examples{k,1}+dy*examples{k,2});
    imagesc(1:256);
    set(gca,'ytick',[],'xtick',[]);%[1 64:64:256]);
    eval(['colormap(ax(k),' examples{k,3} ');']);
    title([char('a'+k-1) ') ' examples{k,3}],'interp','none');
    xlabel(examples{k,4});
end

% print -dpng ./doc/exColmaps
end














