# -*- coding: utf-8 -*-
"""Probability Calibration Curve.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1mJg3ExONPgGsWMRfx5hbAUxId8T2AOiM
"""

# Commented out IPython magic to ensure Python compatibility.
import sys
import requests
import numpy as np
from io import BytesIO

# Check if running in a Jupyter environment
try:
    get_ipython
    IN_JUPYTER = True
except NameError:
    IN_JUPYTER = False

if IN_JUPYTER:
    # Install necessary packages using pip magic
#     %pip install scikit-learn xgboost
else:
    # If running as a script, ensure packages are installed before running
    import subprocess
    import sys
    try:
        import sklearn
        import xgboost
    except ImportError:
        subprocess.check_call([sys.executable, "-m", "pip", "install", "scikit-learn", "xgboost"])

import matplotlib as mpl
import matplotlib.pyplot as plt

mpl.rcParams["figure.figsize"] = (10, 10)
mpl.rcParams["axes.grid"] = False
mpl.rcParams["font.size"] = 16

# GitLab raw URLs (ensure they point directly to raw .npy files)
url_train = "https://gitlab.com/mirsakhawathossain/exoplanet-ml/-/raw/main/PCA_Robust/X_train_pca.npy"
url_test = "https://gitlab.com/mirsakhawathossain/exoplanet-ml/-/raw/main/PCA_Robust/X_test_pca.npy"
url_y_train = "https://gitlab.com/mirsakhawathossain/exoplanet-ml/-/raw/main/PCA_Robust/y_train.npy"
url_y_test = "https://gitlab.com/mirsakhawathossain/exoplanet-ml/-/raw/main/PCA_Robust/y_test.npy"

# Function to load numpy arrays directly from URL with error handling
def load_numpy_from_url(url):
    response = requests.get(url)
    try:
        response.raise_for_status()
    except requests.HTTPError as e:
        print(f"Failed to fetch {url}: {e}")
        raise
    return np.load(BytesIO(response.content))

# Load datasets
try:
    X_train = load_numpy_from_url(url_train)
    X_test = load_numpy_from_url(url_test)
    y_train = load_numpy_from_url(url_y_train)
    y_test = load_numpy_from_url(url_y_test)
except Exception as e:
    print(f"Error loading data: {e}")
    sys.exit(1)

from sklearn.model_selection import RepeatedStratifiedKFold
cv = RepeatedStratifiedKFold(n_splits=10, n_repeats=10, random_state=41)

import xgboost as xgb

# Create an instance of the XGBoost classifier with updated parameters
xgb_classifier = xgb.XGBClassifier(
    colsample_bytree=0.8840923286188427,
    learning_rate=0.02136758382220562,
    max_depth=10,
    n_estimators=426,
    reg_alpha=0,
    reg_lambda=1,
    subsample=0.8834671223572841,
    random_state=43,
    n_jobs=-1,
    use_label_encoder=False,
    eval_metric='logloss'
)

# Fit the classifier
try:
    xgb_classifier.fit(X_train, np.ravel(y_train))
except Exception as e:
    print(f"Error during model fitting: {e}")
    sys.exit(1)

from sklearn.calibration import calibration_curve
from sklearn.metrics import brier_score_loss

# Probability estimate
prob_pos = xgb_classifier.predict_proba(X_test)[:, 1]
# Brier Score
b_score = brier_score_loss(y_test, prob_pos)
print("Brier Score:", b_score)

# True and Predicted Probabilities for calibration curve
true_pos, pred_pos = calibration_curve(y_test, prob_pos, n_bins=10)

# Plot the first calibration curve
plt.figure()
plt.plot(pred_pos, true_pos, marker='o', linewidth=1, label='XGBoost')
plt.plot([0, 1], [0, 1], linestyle='--', label='Perfectly Calibrated')
plt.title('Probability Calibration Curve')
plt.xlabel('Predicted Probability')
plt.ylabel('True Probability')
plt.legend(loc='best')
# Save the plot before showing
plt.savefig("calibration_curve_xgb.png", dpi=300)
plt.savefig("calibration_curve_xgb.eps", format="eps", dpi=300)
plt.savefig("calibration_curve_xgb.pdf", format="pdf", dpi=300)
plt.savefig("calibration_curve_xgb.jpeg", format="jpeg", dpi=300)
plt.show()
plt.close()

from sklearn.calibration import CalibratedClassifierCV

# Create a calibrated classifier using isotonic regression
xgb_classifier_calib = CalibratedClassifierCV(xgb_classifier, method='isotonic', cv=5, n_jobs=-1)

# Fit the calibrated classifier
try:
    xgb_classifier_calib.fit(X_train, np.ravel(y_train))
except Exception as e:
    print(f"Error during calibration fitting: {e}")
    sys.exit(1)

# Probability estimate after calibration
prob_pos_cal = xgb_classifier_calib.predict_proba(X_test)[:, 1]
# Brier Score after calibration
b_score_cal = brier_score_loss(y_test, prob_pos_cal)
print("Brier Score after calibration:", b_score_cal)

# True and Predicted Probabilities for calibration curve after calibration
true_pos_cal, pred_pos_cal = calibration_curve(y_test, prob_pos_cal, n_bins=10)

# Plot the calibrated calibration curve
plt.figure()
plt.plot(pred_pos_cal, true_pos_cal, marker='o', linewidth=1, label='XGBoost Calibrated')
plt.plot([0, 1], [0, 1], linestyle='--', label='Perfectly Calibrated')
plt.title('Probability Calibration Curve (Calibrated)')
plt.xlabel('Predicted Probability')
plt.ylabel('True Probability')
plt.legend(loc='best')
# Save the plot before showing
plt.savefig("calibration_curve_xgb_calibrated.png", dpi=300)
plt.savefig("calibration_curve_xgb_calibrated.eps", format="eps", dpi=300)
plt.savefig("calibration_curve_xgb_calibrated.pdf", format="pdf", dpi=300)
plt.savefig("calibration_curve_xgb_calibrated.jpeg", format="jpeg", dpi=300)
plt.show()
plt.close()