package io.github.mianalysis.mia.module.images.process;

import java.util.Arrays;

import org.scijava.Priority;
import org.scijava.plugin.Plugin;

import deepimagej.DeepImageJ;
import deepimagej.tools.DijTensor;
import ij.IJ;
import ij.ImagePlus;
import ij.ImageStack;
import ij.measure.Calibration;
import ij.plugin.RGBStackConverter;
import io.github.mianalysis.mia.MIA;
import io.github.mianalysis.mia.module.Categories;
import io.github.mianalysis.mia.module.Category;
import io.github.mianalysis.mia.module.Module;
import io.github.mianalysis.mia.module.Modules;
import io.github.mianalysis.mia.object.Workspace;
import io.github.mianalysis.mia.object.image.Image;
import io.github.mianalysis.mia.object.image.ImageFactory;
import io.github.mianalysis.mia.object.measurements.Measurement;
import io.github.mianalysis.mia.object.parameters.BooleanP;
import io.github.mianalysis.mia.object.parameters.ChoiceP;
import io.github.mianalysis.mia.object.parameters.InputImageP;
import io.github.mianalysis.mia.object.parameters.OutputImageP;
import io.github.mianalysis.mia.object.parameters.ParameterState;
import io.github.mianalysis.mia.object.parameters.Parameters;
import io.github.mianalysis.mia.object.parameters.SeparatorP;
import io.github.mianalysis.mia.object.parameters.text.MessageP;
import io.github.mianalysis.mia.object.parameters.text.StringP;
import io.github.mianalysis.mia.object.refs.ImageMeasurementRef;
import io.github.mianalysis.mia.object.refs.collections.ImageMeasurementRefs;
import io.github.mianalysis.mia.object.refs.collections.MetadataRefs;
import io.github.mianalysis.mia.object.refs.collections.ObjMeasurementRefs;
import io.github.mianalysis.mia.object.refs.collections.ObjMetadataRefs;
import io.github.mianalysis.mia.object.refs.collections.ParentChildRefs;
import io.github.mianalysis.mia.object.refs.collections.PartnerRefs;
import io.github.mianalysis.mia.object.system.Status;
import io.github.mianalysis.mia.process.deepimagej.PrepareDeepImageJ;

/**
 * Uses <a href="https://deepimagej.github.io/deepimagej/">DeepImageJ</a> to run
 * Tensorflow and Pytorch models from the
 * <a href="https://bioimage.io/#/">BioImage Model Zoo</a>. This module will
 * detect and run any models already installed in the active copy of Fiji.
 */
@Plugin(type = Module.class, priority = Priority.LOW, visible = true)
public class ApplyDeepImageJModel extends Module {

    /**
    * 
    */
    public static final String INPUT_SEPARATOR = "Image input/output";

    /**
     * Image from the workspace to apply deep learning model to.
     */
    public static final String INPUT_IMAGE = "Input image";

    /**
     * Converts a composite image to RGB format. This should be set to match the
     * image-type used for generation of the model.
     */
    public static final String CONVERT_TO_RGB = "Convert to RGB";

    /**
     * Final image generated by model, which will be stored in the workspace with
     * this name.
     */
    public static final String OUTPUT_IMAGE = "Output image";

    /**
    * 
    */
    public static final String MODEL_SEPARATOR = "Model controls";

    /**
     * Model to apply to input image. This can be any model currently installed in
     * MIA. When using MIA's GUI, the available modules will automatically appear as
     * options.
     */
    public static final String MODEL = "Model";

    /**
     * If post-processing routines are available for the chosen model this option
     * will be visible. Note: If pre-processing routines are available, these will
     * always be applied.
     */
    public static final String USE_POSTPROCESSING = "Use postprocessing";

    /**
    * 
    */
    public static final String PATCH_SIZE = "Patch size";

    /**
    * 
    */
    public static final String AXES_ORDER = "Axes order";

    private String currModelName = "";

    public interface Models {
        String[] ALL = PrepareDeepImageJ.getAvailableModels();

    }

    public interface FormatsBoth {
        String PYTORCH = "Pytorch";
        String TENSORFLOW = "Tensorflow";

        String[] ALL = new String[] { PYTORCH, TENSORFLOW };

    }

    public interface Measurements {
        String CLASSIFICATION_CLASS = "CLASSIFICATION // CLASS";
        String CLASSIFICATION_PROBABILITY = "CLASSIFICATION // PROBABILITY";
    }

    public ApplyDeepImageJModel(Modules modules) {
        super("Apply DeepImageJ model", modules);
    }

    @Override
    public Category getCategory() {
        return Categories.IMAGES_PROCESS;
    }

    @Override
    public Status process(Workspace workspace) {
        // Getting parameters
        String inputImageName = parameters.getValue(INPUT_IMAGE, workspace);
        boolean convertToRGB = parameters.getValue(CONVERT_TO_RGB, workspace);
        String outputImageName = parameters.getValue(OUTPUT_IMAGE, workspace);
        String modelName = parameters.getValue(MODEL, workspace);
        // String preprocessing = parameters.getValue(PREPROCESSING, workspace);
        boolean usePostprocessing = parameters.getValue(USE_POSTPROCESSING, workspace);
        // String postprocessing = parameters.getValue(POSTPROCESSING, workspace);
        String patchSize = parameters.getValue(PATCH_SIZE, workspace);

        // Get input image
        Image inputImage = workspace.getImage(inputImageName);
        ImagePlus inputIpl = inputImage.getImagePlus();

        // Converting to RGB if requested
        if (convertToRGB) {
            inputIpl = inputIpl.duplicate();
            RGBStackConverter.convertToRGB(inputIpl);
        }

        // Running deep learning model
        DeepImageJ model = PrepareDeepImageJ.getModel(modelName);
        String outputType = model.params.outputList.get(0).tensorType;
        MIA.log.writeDebug(model);

        // Updating pre and post processing options
        boolean usePreprocessing = true;
        if (PrepareDeepImageJ.getPreprocessings(modelName).length == 0)
            usePreprocessing = false;
        if (PrepareDeepImageJ.getPostprocessings(modelName).length == 0)
            usePostprocessing = false;

        String format = PrepareDeepImageJ.getFormats(modelName)[0];
        PrepareDeepImageJ pDIJ = new PrepareDeepImageJ();

        ImageStack inputIst = inputIpl.getStack();
        ImagePlus outputIpl = null;
        ImageStack outputIst = null;

        // PrepareDeepImageJ.getOutputDimensions(modelName, outputIpl);

        int count = 0;
        for (int z = 0; z < inputIpl.getNSlices(); z++) {
            for (int t = 0; t < inputIpl.getNFrames(); t++) {
                int inputIdx = inputIpl.getStackIndex(1, z + 1, t + 1);
                ImagePlus tempIpl = new ImagePlus("Temp", inputIst.getProcessor(inputIdx));
                Object modelOutput = pDIJ.runModel(tempIpl, model, format, usePreprocessing, usePostprocessing,
                        patchSize);

                if (outputType.equals("image")) {
                    ImagePlus tempOutputIpl = (ImagePlus) modelOutput;

                    // If it hasn't already been created (i.e. this is the first slice), create
                    // output ImagePlus
                    if (outputIpl == null) {
                        int width = tempOutputIpl.getWidth();
                        int height = tempOutputIpl.getHeight();
                        int nChannels = tempOutputIpl.getNChannels();
                        int nSlices = inputIpl.getNSlices();
                        int nFrames = inputIpl.getNFrames();

                        outputIpl = IJ.createHyperStack(outputImageName, width, height, nChannels, nSlices, nFrames,
                                32);

                        Calibration inputCal = inputIpl.getCalibration();
                        Calibration outputCal = new Calibration();
                        outputCal.pixelHeight = inputCal.pixelHeight;
                        outputCal.pixelWidth = inputCal.pixelWidth;
                        outputCal.pixelDepth = inputCal.pixelDepth;
                        outputCal.setUnit(inputCal.getUnits());
                        outputCal.setTimeUnit(inputCal.getTimeUnit());
                        outputCal.fps = inputCal.fps;
                        outputCal.frameInterval = inputCal.frameInterval;
                        outputIpl.setCalibration(outputCal);

                        outputIst = outputIpl.getStack();

                    }

                    ImageStack tempIst = tempOutputIpl.getStack();
                    for (int c = 0; c < tempOutputIpl.getNChannels(); c++) {
                        int tempOutputIdx = tempOutputIpl.getStackIndex(c + 1, 1, 1);
                        int outputIdx = outputIpl.getStackIndex(c + 1, z + 1, t + 1);
                        outputIst.setProcessor(tempIst.getProcessor(tempOutputIdx), outputIdx);
                    }
                } else if (outputType.equals("list")) {
                    float[] modelOutputArray = (float[]) modelOutput;
                    int classificationClass = 0;
                    double classificationProbability = 0;

                    if (model.params.outputList.get(0).recommended_patch[1] == 1) {
                        // Binary classification
                        classificationClass = (int) Math.round(modelOutputArray[1]);
                        if (classificationClass == 0)
                            classificationProbability = modelOutputArray[1];
                        else
                            classificationProbability = 1 - modelOutputArray[1];

                    } else {
                        // Categorical classification
                        classificationClass = (int) Math.round(modelOutputArray[0]);
                    }

                    inputImage.addMeasurement(new Measurement(Measurements.CLASSIFICATION_CLASS, classificationClass));
                    inputImage.addMeasurement(
                            new Measurement(Measurements.CLASSIFICATION_PROBABILITY, classificationProbability));

                }

                writeProgressStatus(++count, inputIst.size(), "slices");

            }
        }

        if (outputType.equals("image")) {
            // It seems necessary to reapply the ImageStack
            outputIpl.setStack(outputIst);

            // Storing output image
            Image outputImage = ImageFactory.createImage(outputImageName, outputIpl);
            workspace.addImage(outputImage);

            if (showOutput)
                outputImage.show();
        } else if (outputType.equals("list")) {
            if (showOutput)
                inputImage.showMeasurements(this);
        }

        return Status.PASS;

    }

    @Override
    protected void initialiseParameters() {
        parameters.add(new SeparatorP(INPUT_SEPARATOR, this));
        parameters.add(new InputImageP(INPUT_IMAGE, this));
        parameters.add(new BooleanP(CONVERT_TO_RGB, this, false,
                "Converts a composite image to RGB format.  This should be set to match the image-type used for generation of the model."));
        parameters.add(new OutputImageP(OUTPUT_IMAGE, this));

        parameters.add(new SeparatorP(MODEL_SEPARATOR, this));
        parameters.add(new ChoiceP(MODEL, this, "", Models.ALL));
        parameters.add(new BooleanP(USE_POSTPROCESSING, this, false));
        parameters.add(new StringP(PATCH_SIZE, this, ""));
        parameters.add(new MessageP(AXES_ORDER, this, ParameterState.MESSAGE, 20));

        addParameterDescriptions();

    }

    @Override
    public Parameters updateAndGetParameters() {
        Workspace workspace = null;
        String modelName = parameters.getValue(MODEL, workspace);
        DeepImageJ model = PrepareDeepImageJ.getModel(modelName);

        Parameters returnedParameters = new Parameters();
        returnedParameters.add(parameters.getParameter(INPUT_SEPARATOR));
        returnedParameters.add(parameters.getParameter(INPUT_IMAGE));
        returnedParameters.add(parameters.getParameter(CONVERT_TO_RGB));

        if (model != null) {
            String outputType = model.params.outputList.get(0).tensorType;
            if (outputType.equals("image"))
                returnedParameters.add(parameters.getParameter(OUTPUT_IMAGE));
        }

        returnedParameters.add(parameters.getParameter(MODEL_SEPARATOR));
        returnedParameters.add(parameters.getParameter(MODEL));

        String[] postprocessingChoices = PrepareDeepImageJ.getPostprocessings(modelName);
        if (postprocessingChoices.length > 0)
            returnedParameters.add(parameters.getParameter(USE_POSTPROCESSING));

        if (!currModelName.equals(modelName)) {
            // We don't know the actual image size, so creating a small one.
            ImagePlus testIpl = IJ.createHyperStack("Test", 10, 10, 1, 1, 1, 8);
            String patchSize = PrepareDeepImageJ.getOptimalPatch(modelName, testIpl);
            parameters.getParameter(PATCH_SIZE).setValue(patchSize);
        }
        returnedParameters.add(parameters.getParameter(PATCH_SIZE));
        returnedParameters.add(parameters.getParameter(AXES_ORDER));
        ((MessageP) parameters.get(AXES_ORDER))
                .setValue("Selected model requires input image axes to be in the order: "
                        + PrepareDeepImageJ.getAxes(modelName));

        // PrepareDeepImageJ.getOutputDimensions(modelName);

        currModelName = modelName;

        return returnedParameters;

    }

    @Override
    public ImageMeasurementRefs updateAndGetImageMeasurementRefs() {
        Workspace workspace = null;

        ImageMeasurementRefs returnedRefs = new ImageMeasurementRefs();

        String modelName = parameters.getValue(MODEL, workspace);
        DeepImageJ model = PrepareDeepImageJ.getModel(modelName);

        if (model != null) {
            String outputType = model.params.outputList.get(0).tensorType;
            if (outputType.equals("list")) {
                ImageMeasurementRef ref = imageMeasurementRefs.getOrPut(Measurements.CLASSIFICATION_CLASS);
                ref.setImageName(parameters.getValue(INPUT_IMAGE, workspace));
                returnedRefs.add(ref);

                ref = imageMeasurementRefs.getOrPut(Measurements.CLASSIFICATION_PROBABILITY);
                ref.setImageName(parameters.getValue(INPUT_IMAGE, workspace));
                returnedRefs.add(ref);
            }
        }

        return returnedRefs;

    }

    @Override
    public ObjMeasurementRefs updateAndGetObjectMeasurementRefs() {
        return null;
    }

    @Override
    public ObjMetadataRefs updateAndGetObjectMetadataRefs() {
        return null;
    }

    @Override
    public MetadataRefs updateAndGetMetadataReferences() {
        return null;
    }

    @Override
    public ParentChildRefs updateAndGetParentChildRefs() {
        return null;
    }

    @Override
    public PartnerRefs updateAndGetPartnerRefs() {
        return null;
    }

    @Override
    public boolean verify() {
        return true;
    }

    @Override
    public String getVersionNumber() {
        return "1.0.0";
    }

    @Override
    public String getDescription() {
        return "Uses DeepImageJ to run Tensorflow and Pytorch models from the BioImage Model Zoo.  This module will detect and run any models already installed in the active copy of Fiji.  For more information on DeepImageJ and the BioImage Model Zoo, please go to <a href=\"https://deepimagej.github.io/deepimagej/\">DeepImageJ</a> and <a href=\"https://bioimage.io/#/\">BioImage Model Zoo</a>.";
    }

    protected void addParameterDescriptions() {
        parameters.get(INPUT_IMAGE).setDescription("Image from the workspace to apply deep learning model to.");
        parameters.get(OUTPUT_IMAGE).setDescription(
                "Final image generated by model, which will be stored in the workspace with this name.");
        parameters.get(MODEL).setDescription(
                "Model to apply to input image.  This can be any model currently installed in MIA.  When using MIA's GUI, the available modules will automatically appear as options.");
        parameters.get(USE_POSTPROCESSING).setDescription(
                "If post-processing routines are available for the chosen model this option will be visible.  Note: If pre-processing routines are available, these will always be applied.");

    }
}
