#include <AMReX_REAL.H>

#include <Derive.H>
#include <Castro.H>

#ifdef DIFFUSION
#include <diffusion_util.H>
#endif

using namespace amrex;

#ifdef __cplusplus
extern "C"
{
#endif

    // These routines are called in an MFIter loop, so we do not
    // need to explicitly synchronize after GPU kernels.

    void ca_derpres(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                    const FArrayBox& datfab, const Geometry& /*geom*/,
                    Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

        eos_rep_t eos_state;
        eos_state.rho  = dat(i,j,k,URHO);
        eos_state.T = dat(i,j,k,UTEMP);
        eos_state.e = dat(i,j,k,UEINT) * rhoInv;
        for (int n = 0; n < NumSpec; n++) {
          eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
        }
#if NAUX_NET > 0
        for (int n = 0; n < NumAux; n++) {
          eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
        }
#endif

        eos(eos_input_re, eos_state);

        der(i,j,k,0) = eos_state.p;
      });
    }

    void ca_dereint1(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                     const FArrayBox& datfab, const Geometry& /*geom*/,
                     Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        Real rhoInv = 1.0_rt/dat(i,j,k,URHO);
        Real ux = dat(i,j,k,UMX)*rhoInv;
        Real uy = dat(i,j,k,UMY)*rhoInv;
        Real uz = dat(i,j,k,UMZ)*rhoInv;

        der(i,j,k,0) = dat(i,j,k,UEDEN)*rhoInv -
          0.5_rt * (ux*ux + uy*uy + uz*uz);
      });
    }

    void ca_dereint2(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                     const FArrayBox& datfab, const Geometry& /*geom*/,
                     Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        der(i,j,k,0) = dat(i,j,k,UEINT) / dat(i,j,k,URHO);
      });
    }

    void ca_derlogden(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                      const FArrayBox& datfab, const Geometry& /*geom*/,
                      Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {
        der(i,j,k,0) = std::log10(dat(i,j,k,0));
      });
    }

    void ca_deruplusc(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                      const FArrayBox& datfab, const Geometry& /*geom*/,
                      Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {


        Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

        eos_rep_t eos_state;
        eos_state.rho  = dat(i,j,k,URHO);
        eos_state.T = dat(i,j,k,UTEMP);
        eos_state.e = dat(i,j,k,UEINT) * rhoInv;
        for (int n = 0; n < NumSpec; n++) {
          eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
        }
#if NAUX_NET > 0
        for (int n = 0; n < NumAux; n++) {
          eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
        }
#endif

        eos(eos_input_re, eos_state);

        der(i,j,k,0) = dat(i,j,k,UMX) / dat(i,j,k,URHO) + eos_state.cs;

      });
    }

    void ca_deruminusc(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                       const FArrayBox& datfab, const Geometry& /*geom*/,
                       Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {


        Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

        eos_rep_t eos_state;
        eos_state.rho  = dat(i,j,k,URHO);
        eos_state.T = dat(i,j,k,UTEMP);
        eos_state.e = dat(i,j,k,UEINT) * rhoInv;
        for (int n = 0; n < NumSpec; n++) {
          eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
        }
#if NAUX_NET > 0
        for (int n = 0; n < NumAux; n++) {
          eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
        }
#endif

        eos(eos_input_re, eos_state);

        der(i,j,k,0) = dat(i,j,k,UMX) / dat(i,j,k,URHO) - eos_state.cs;

      });
    }

    void ca_dersoundspeed(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                          const FArrayBox& datfab, const Geometry& /*geom*/,
                          Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {


        Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

        eos_rep_t eos_state;
        eos_state.rho  = dat(i,j,k,URHO);
        eos_state.T = dat(i,j,k,UTEMP);
        eos_state.e = dat(i,j,k,UEINT) * rhoInv;
        for (int n = 0; n < NumSpec; n++) {
          eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
        }
#if NAUX_NET > 0
        for (int n = 0; n < NumAux; n++) {
          eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
        }
#endif

        eos(eos_input_re, eos_state);

        der(i,j,k,0) = eos_state.cs;

      });
    }


    void ca_dergamma1(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                      const FArrayBox& datfab, const Geometry& /*geom*/,
                      Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {


        Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

        eos_rep_t eos_state;
        eos_state.rho  = dat(i,j,k,URHO);
        eos_state.T = dat(i,j,k,UTEMP);
        eos_state.e = dat(i,j,k,UEINT) * rhoInv;
        for (int n = 0; n < NumSpec; n++) {
          eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
        }
#if NAUX_NET > 0
        for (int n = 0; n < NumAux; n++) {
          eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
        }
#endif

        eos(eos_input_re, eos_state);

        der(i,j,k,0) = eos_state.gam1;

      });
    }

    void ca_dermachnumber(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                          const FArrayBox& datfab, const Geometry& /*geom*/,
                          Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {


        Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

        eos_rep_t eos_state;
        eos_state.rho  = dat(i,j,k,URHO);
        eos_state.T = dat(i,j,k,UTEMP);
        eos_state.e = dat(i,j,k,UEINT) * rhoInv;
        for (int n = 0; n < NumSpec; n++) {
          eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
        }
#if NAUX_NET > 0
        for (int n = 0; n < NumAux; n++) {
          eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
        }
#endif

        eos(eos_input_re, eos_state);

        der(i,j,k,0) = std::sqrt(dat(i,j,k,UMX)*dat(i,j,k,UMX) +
                                 dat(i,j,k,UMY)*dat(i,j,k,UMY) +
                                 dat(i,j,k,UMZ)*dat(i,j,k,UMZ)) /
          dat(i,j,k,URHO) / eos_state.cs;

      });
    }

    void ca_derentropy(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                       const FArrayBox& datfab, const Geometry& /*geom*/,
                       Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {


        Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

        eos_t eos_state;
        eos_state.rho  = dat(i,j,k,URHO);
        eos_state.T = dat(i,j,k,UTEMP);
        eos_state.e = dat(i,j,k,UEINT) * rhoInv;
        for (int n = 0; n < NumSpec; n++) {
          eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
        }
#if NAUX_NET > 0
        for (int n = 0; n < NumAux; n++) {
          eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
        }
#endif

        eos(eos_input_re, eos_state);

        der(i,j,k,0) = eos_state.s;
      });
    }

#ifdef DIFFUSION
    void ca_dercond(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                    const FArrayBox& datfab, const Geometry& /*geom*/,
                    Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      fill_temp_cond(bx, dat, der);

    }

    void ca_derdiffcoeff(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                    const FArrayBox& datfab, const Geometry& /*geom*/,
                    Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      fill_temp_diff_coeff(bx, dat, der);

    }

    void ca_derdiffterm(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                        const FArrayBox& datfab, const Geometry& geom,
                        Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      const Box& obx = amrex::grow(bx, 1);

      FArrayBox coeff_cc;
      coeff_cc.resize(obx, 1, The_Async_Arena());
      Array4<Real> const coeff_arr = coeff_cc.array();

      auto const dat = datfab.array();
      auto const der = derfab.array();

      fill_temp_cond(obx, dat, coeff_arr);

      auto dx = geom.CellSizeArray();
      auto problo = geom.ProbLoArray();
      const int coord_type = geom.Coord();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        // (k grad T)_{i+1/2}
        Real kgradT_xhi = 0.5_rt * (coeff_arr(i+1,j,k) + coeff_arr(i,j,k)) *
          (dat(i+1,j,k,UTEMP) - dat(i,j,k,UTEMP)) / dx[0];

        // (k grad T)_{i-1/2}
        Real kgradT_xlo = 0.5_rt * (coeff_arr(i,j,k) + coeff_arr(i-1,j,k)) *
          (dat(i,j,k,UTEMP) - dat(i-1,j,k,UTEMP)) / dx[0];

#if AMREX_SPACEDIM >= 2
        // (k grad T)_{j+1/2}
        Real kgradT_yhi = 0.5_rt * (coeff_arr(i,j+1,k) + coeff_arr(i,j,k)) *
          (dat(i,j+1,k,UTEMP) - dat(i,j,k,UTEMP)) / dx[1];

        // (k grad T)_{j-1/2}
        Real kgradT_ylo = 0.5_rt * (coeff_arr(i,j,k) + coeff_arr(i,j-1,k)) *
          (dat(i,j,k,UTEMP) - dat(i,j-1,k,UTEMP)) / dx[1];
#endif

#if AMREX_SPACEDIM == 3
        // (k grad T)_{k+1/2}
        Real kgradT_zhi = 0.5_rt * (coeff_arr(i,j,k+1) + coeff_arr(i,j,k)) *
          (dat(i,j,k+1,UTEMP) - dat(i,j,k,UTEMP)) / dx[2];

        // (k grad T)_{k-1/2}
        Real kgradT_zlo = 0.5_rt * (coeff_arr(i,j,k) + coeff_arr(i,j,k-1)) *
          (dat(i,j,k,UTEMP) - dat(i,j,k-1,UTEMP)) / dx[2];
#endif

        if (coord_type == 0) {
          // Cartesian
          der(i,j,k,0) = (kgradT_xhi - kgradT_xlo)/dx[0];
#if AMREX_SPACEDIM >= 2
          der(i,j,k,0) += (kgradT_yhi - kgradT_ylo)/dx[1];
#endif
#if AMREX_SPACEDIM == 3
          der(i,j,k,0) += (kgradT_zhi - kgradT_zlo)/dx[2];
#endif

        } else if (coord_type == 1) {
          // axisymmetric coords (2-d)
          Real rm = static_cast<Real>(i)*dx[0] + problo[0];
          Real r = (static_cast<Real>(i) + 0.5_rt)*dx[0] + problo[0];
          Real rp = (static_cast<Real>(i) + 1.0_rt)*dx[0] + problo[0];

          der(i,j,k,0) = (rp*kgradT_xhi - rm*kgradT_xlo)/(r*dx[0]);
#if AMREX_SPACEDIM == 2
          der(i,j,k,0) += (kgradT_yhi - kgradT_ylo)/dx[1];
#endif

        } else if (coord_type == 2) {
          // spherical coords (1-d) and (2-d)
          Real rm = static_cast<Real>(i)*dx[0] + problo[0];
          Real r = (static_cast<Real>(i) + 0.5_rt)*dx[0] + problo[0];
          Real rp = (static_cast<Real>(i) + 1.0_rt)*dx[0] + problo[0];

          der(i,j,k,0) = (rp*rp*kgradT_xhi - rm*rm*kgradT_xlo)/(r*r*dx[0]);
#if AMREX_SPACEDIM == 2
          Real sinm = std::sin(static_cast<Real>(j)*dx[1] + problo[1]);
          Real sinc = std::sin((static_cast<Real>(j) + 0.5_rt)*dx[1] + problo[1]);
          Real sinp = std::sin((static_cast<Real>(j) + 1.0_rt)*dx[1] + problo[1]);

          der(i,j,k,0) += (sinp*kgradT_yhi - sinm*kgradT_ylo)/(r*r*sinc*dx[1]);
#endif
        }
      });
    }
#endif

#ifdef REACTIONS
    void ca_derenuctimescale(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                             const FArrayBox& datfab, const Geometry& geom,
                             Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      auto dx = geom.CellSizeArray();

      Real dd = 0.0_rt;
#if AMREX_SPACEDIM == 1
      dd = dx[0];
#elif AMREX_SPACEDIM == 2
      dd = amrex::min(dx[0], dx[1]);
#else
      dd = amrex::min(dx[0], dx[1], dx[2]);
#endif

      int enuc_comp = datfab.nComp()-1;

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        // the nuclear energy (rho H_nuc) is tacked onto the end of
        // the input state, after the NUM_STATE conserved state
        // quantities
        Real enuc = std::abs(dat(i,j,k,enuc_comp)) / dat(i,j,k,URHO);

        if (enuc > 1.e-100_rt) {

          Real rhoInv = 1.0_rt / dat(i,j,k,URHO);

          // calculate the sound speed
          eos_rep_t eos_state;
          eos_state.rho  = dat(i,j,k,URHO);
          eos_state.T = dat(i,j,k,UTEMP);
          eos_state.e = dat(i,j,k,UEINT) * rhoInv;
          for (int n = 0; n < NumSpec; n++) {
            eos_state.xn[n] = dat(i,j,k,UFS+n) * rhoInv;
          }
#if NAUX_NET > 0
          for (int n = 0; n < NumAux; n++) {
            eos_state.aux[n] = dat(i,j,k,UFX+n) * rhoInv;
          }
#endif

          eos(eos_input_re, eos_state);

          Real t_e = eos_state.e / enuc;
          Real t_s = dd / eos_state.cs;

          der(i,j,k,0) = t_s/t_e;

        } else {
          der(i,j,k,0) = 0.0_rt;
        }

      });
    }

    void ca_derenuc(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                    const FArrayBox& datfab, const Geometry& /*geom*/,
                    Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {
      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {
          // The derive data is (rho, rho_enuc)
          Real enuc = dat(i,j,k,1) / dat(i,j,k,0);

          der(i,j,k,0) = enuc;
      });
    }
#endif

    void ca_dervel(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                   const FArrayBox& datfab, const Geometry& /*geom*/,
                   Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {
             der(i,j,k,0) = dat(i,j,k,1) / dat(i,j,k,0);
      });
    }


    void ca_dermagvel(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                      const FArrayBox& datfab, const Geometry& /*geom*/,
                      Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        Real deninv = 1.0_rt/dat(i,j,k,0);

        der(i,j,k,0) = std::sqrt( (dat(i,j,k,1) * dat(i,j,k,1) +
                                   dat(i,j,k,2) * dat(i,j,k,2) +
                                   dat(i,j,k,3) * dat(i,j,k,3)) ) * deninv;
      });
    }


    void ca_dermaggrav(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                       const FArrayBox& datfab, const Geometry& /*geom*/,
                       Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        der(i,j,k,0) = std::sqrt(dat(i,j,k,0)*dat(i,j,k,0) +
                                 dat(i,j,k,1)*dat(i,j,k,1) +
                                 dat(i,j,k,2)*dat(i,j,k,2));

      });
    }

    void ca_derradialvel(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                         const FArrayBox& datfab, const Geometry& geom,
                         Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      // our input dat is rho, UMX, UMY, UMZ

      auto const dat = datfab.array();
      auto const der = derfab.array();

      auto dx = geom.CellSizeArray();
      auto problo = geom.ProbLoArray();
      auto geomdata = geom.data();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {
        GpuArray<Real, 3> loc = {0.0};
        loc[0] = problo[0] + (static_cast<Real>(i) + 0.5_rt) * dx[0] - problem::center[0];
#if AMREX_SPACEDIM >= 2
        loc[1] = problo[1] + (static_cast<Real>(j) + 0.5_rt) * dx[1] - problem::center[1];
#endif
#if AMREX_SPACEDIM == 3
        loc[2] = problo[2] + (static_cast<Real>(k) + 0.5_rt) * dx[2] - problem::center[2];
#endif

        if (domain_is_plane_parallel) {
#if AMREX_SPACEDIM == 2
          // the radial velocity is just the horizontal velocity
          der(i,j,k,0) = dat(i,j,k,1)/dat(i,j,k,0);
#elif AMREX_SPACEDIM == 3
          // the velocity in the x-y plane decomposed into r, phi unit vectors is:
          // v_cyl = ( u cos phi + v sin phi) e_r +
          //         (-u sin phi + v cos phi) e_phi
          // where e_r and e_phi are the cylindrical unit vectors

          // we need the distance in the x-y plane from the origin
          Real r = std::sqrt(loc[0]*loc[0] + loc[1]*loc[1]);
          der(i,j,k,0) = (dat(i,j,k,1)*loc[0] + dat(i,j,k,2)*loc[1]) / (dat(i,j,k,0)*r);
#endif
        } else {
          Real r = distance(geomdata, loc);

          der(i,j,k,0) = (dat(i,j,k,1)*loc[0] +
                          dat(i,j,k,2)*loc[1] +
                          dat(i,j,k,3)*loc[2]) / ( dat(i,j,k,0)*r );
        }

      });
    }


    void ca_dercircvel(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                       const FArrayBox& datfab, const Geometry& geom,
                       Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      // our input dat is rho, UMX, UMY, UMZ

      auto const dat = datfab.array();
      auto const der = derfab.array();

      auto dx = geom.CellSizeArray();
      auto problo = geom.ProbLoArray();
      auto geomdata = geom.data();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        GpuArray<Real, 3> loc = {0.0};
        loc[0] = problo[0] + (static_cast<Real>(i) + 0.5_rt) * dx[0] - problem::center[0];
#if AMREX_SPACEDIM >= 2
        loc[1] = problo[1] + (static_cast<Real>(j) + 0.5_rt) * dx[1] - problem::center[1];
#endif
#if AMREX_SPACEDIM == 3
        loc[2] = problo[2] + (static_cast<Real>(k) + 0.5_rt) * dx[2] - problem::center[2];
#endif

        if (domain_is_plane_parallel) {
#if AMREX_SPACEDIM == 2
          // the circumferential velocity is just the out-of-plane velocity
          der(i,j,k,0) = dat(i,j,k,3)/dat(i,j,k,0);
#elif AMREX_SPACEDIM == 3
          // the velocity in the x-y plane decomposed into r, phi unit vectors is:
          // v_cyl = ( u cos phi + v sin phi) e_r +
          //         (-u sin phi + v cos phi) e_phi
          // where e_r and e_phi are the cylindrical unit vectors

          // we need the distance in the x-y plane from the origin
          Real r = std::sqrt(loc[0]*loc[0] + loc[1]*loc[1]);
          der(i,j,k,0) = (-dat(i,j,k,1)*loc[1] + dat(i,j,k,2)*loc[0]) / (dat(i,j,k,0)*r);
#endif
        } else {
          Real r = distance(geomdata, loc);

          // we really mean just the velocity component that is
          // perpendicular to radial, and in general 3-d (e.g. a
          // sphere), the sign doesn't make sense, so we compute this
          // such that v_r^2 + v_c^2 = v^2
          Real vtot2 = (dat(i,j,k,1)*dat(i,j,k,1) +
                        dat(i,j,k,2)*dat(i,j,k,2) +
                        dat(i,j,k,3)*dat(i,j,k,3))/(dat(i,j,k,0)*dat(i,j,k,0));

          Real vr = (dat(i,j,k,1)*loc[0] +
                     dat(i,j,k,2)*loc[1] +
                     dat(i,j,k,3)*loc[2]) / ( dat(i,j,k,0)*r );

          der(i,j,k,0) = std::sqrt(amrex::max(vtot2 - vr*vr, 0.0_rt));
        }

      });
    }


  void ca_dermagmom(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                    const FArrayBox& datfab, const Geometry& /*geom*/,
                    Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    auto const dat = datfab.array();
    auto const der = derfab.array();

    amrex::ParallelFor(bx,
                       [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
                       {

                         der(i,j,k,0) = std::sqrt(dat(i,j,k,0)*dat(i,j,k,0) +
                                                  dat(i,j,k,1)*dat(i,j,k,1) +
                                                  dat(i,j,k,2)*dat(i,j,k,2));

                       });
  }

  void ca_derangmomx (const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                      const FArrayBox& datfab, const Geometry& geom,
                      Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    int idir = 0;
    auto dx     = geom.CellSizeArray();
    auto problo = geom.ProbLoArray();

    auto const dat = datfab.array();
    auto const L = derfab.array();

    amrex::ParallelFor(bx,
                       [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
                       {
                         Real loc[3];

                         //loc calculated like sum_utils.cpp
                         //This might be equivalent and more modular: position(i, j, k, geom, loc);
                         loc[0] = problo[0] + (0.5_rt + i) * dx[0];

#if AMREX_SPACEDIM >= 2
                         loc[1] = problo[1] + (0.5_rt + j) * dx[1];
#else
                         loc[1] = 0.0_rt;
#endif

#if AMREX_SPACEDIM == 3
                         loc[2] = problo[2] + (0.5_rt + k) * dx[2];
#else
                         loc[2] = 0.0_rt;
#endif

                         for (int dir = 0; dir < AMREX_SPACEDIM; ++dir) {
                           loc[dir] -= problem::center[dir];
                         }

                         // Explicitly computing only the required cross-product as in inertial_to_rotational_velocity
                         if (idir == 0) { // cross_product(loc, mom): ang_mom(1)->x)
                           L(i,j,k,0) = loc[1] * dat(i,j,k,3) - loc[2] * dat(i,j,k,2);
                         }
                         else if (idir == 1) { // cross_product(loc, mom): ang_mom(2)->y)
                           L(i,j,k,0) = loc[2] * dat(i,j,k,1) - loc[0] * dat(i,j,k,3);
                         }
                         else { // cross_product(loc, mom): ang_mom(3)->z)
                           L(i,j,k,0) = loc[0] * dat(i,j,k,2) - loc[1] * dat(i,j,k,1);
                         }

                       });

  }

  void ca_derangmomy (const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                      const FArrayBox& datfab, const Geometry& geom,
                      Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    int idir = 1;
    auto dx     = geom.CellSizeArray();
    auto problo = geom.ProbLoArray();

    auto const dat = datfab.array();
    auto const L = derfab.array();

    amrex::ParallelFor(bx,
                       [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
                       {
                         Real loc[3];

                         loc[0] = problo[0] + (0.5_rt + i) * dx[0];

#if AMREX_SPACEDIM >= 2
                         loc[1] = problo[1] + (0.5_rt + j) * dx[1];
#else
                         loc[1] = 0.0_rt;
#endif

#if AMREX_SPACEDIM == 3
                         loc[2] = problo[2] + (0.5_rt + k) * dx[2];
#else
                         loc[2] = 0.0_rt;
#endif
                         for (int dir = 0; dir < AMREX_SPACEDIM; ++dir) {
                           loc[dir] -= problem::center[dir];
                         }

                         if (idir == 0) { // cross_product(loc, mom): ang_mom(1)->x)
                           L(i,j,k,0) = loc[1] * dat(i,j,k,3) - loc[2] * dat(i,j,k,2);
                         }
                         else if (idir == 1) { // cross_product(loc, mom): ang_mom(2)->y)
                           L(i,j,k,0) = loc[2] * dat(i,j,k,1) - loc[0] * dat(i,j,k,3);
                         }
                         else { // cross_product(loc, mom): ang_mom(3)->z)
                           L(i,j,k,0) = loc[0] * dat(i,j,k,2) - loc[1] * dat(i,j,k,1);
                         }

                       });

  }

  void ca_derangmomz (const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                      const FArrayBox& datfab, const Geometry& geom,
                      Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    int idir = 2;
    auto dx     = geom.CellSizeArray();
    auto problo = geom.ProbLoArray();

    auto const dat = datfab.array();
    auto const L = derfab.array();

    amrex::ParallelFor(bx,
                       [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
                       {
                         Real loc[3];

                         loc[0] = problo[0] + (0.5_rt + i) * dx[0];

#if AMREX_SPACEDIM >= 2
                         loc[1] = problo[1] + (0.5_rt + j) * dx[1];
#else
                         loc[1] = 0.0_rt;
#endif

#if AMREX_SPACEDIM == 3
                         loc[2] = problo[2] + (0.5_rt + k) * dx[2];
#else
                         loc[2] = 0.0_rt;
#endif

                         for (int dir = 0; dir < AMREX_SPACEDIM; ++dir) {
                           loc[dir] -= problem::center[dir];
                         }

                         if (idir == 0) { // cross_product(loc, mom): ang_mom(1)->x)
                           L(i,j,k,0) = loc[1] * dat(i,j,k,3) - loc[2] * dat(i,j,k,2);
                         }
                         else if (idir == 1) { // cross_product(loc, mom): ang_mom(2)->y)
                           L(i,j,k,0) = loc[2] * dat(i,j,k,1) - loc[0] * dat(i,j,k,3);
                         }
                         else { // cross_product(loc, mom): ang_mom(3)->z)
                           L(i,j,k,0) = loc[0] * dat(i,j,k,2) - loc[1] * dat(i,j,k,1);
                         }

                       });

  }

  void ca_derkineng (const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                     const FArrayBox& datfab, const Geometry& /*geom*/,
                     Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    auto const dat = datfab.array();
    auto const kineng = derfab.array();

    amrex::ParallelFor(bx,
                       [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
                       {
                         kineng(i,j,k,0) = 0.5_rt / dat(i,j,k,0) * ( dat(i,j,k,1)*dat(i,j,k,1) +
                                                                     dat(i,j,k,2)*dat(i,j,k,2) +
                                                                     dat(i,j,k,3)*dat(i,j,k,3) );
                       });

  }

  void ca_dernull(Real* /*der*/, const int* /*der_lo*/, const int* /*der_hi*/, const int* /*nvar*/,
                  const Real* /*data*/, const int* /*data_lo*/, const int* /*data_hi*/, const int* /*ncomp*/,
                  const int* /*lo*/, const int* /*hi*/,
                  const int* /*domain_lo*/, const int* /*domain_hi*/,
                  const Real* /*delta*/, const Real* /*xlo*/,
                  const Real* /*time*/, const Real* /*dt*/, const int* /*bcrec*/,
                  const int* /*level*/, const int* /*grid_no*/)
  {

    // This routine is used by particle_count.  Yes it does nothing.

  }

  void ca_derspec(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                  const FArrayBox& datfab, const Geometry& /*geom*/,
                  Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {
             der(i,j,k,0) = dat(i,j,k,1) / dat(i,j,k,0);
      });
    }


  void ca_derye(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                const FArrayBox& datfab, const Geometry& /*geom*/,
                Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        Real sum = 0.0_rt;
        Real xn;
        for (int n = 0; n < NumSpec; n++) {
          xn = dat(i,j,k,1+n) / dat(i,j,k,0);
          sum += xn * zion[n] / aion[n];
        }
        der(i,j,k,0) = sum;
      });
    }

  void ca_derabar(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                  const FArrayBox& datfab, const Geometry& /*geom*/,
                  Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        Real sum = 0.0_rt;
        Real xn;
        for (int n = 0; n < NumSpec; n++) {
          xn = dat(i,j,k,1+n)/dat(i,j,k,0);
          sum += xn/aion[n];
        }
        der(i,j,k,0) = 1.0_rt / sum;
      });
    }

  void ca_dermagvort(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                     const FArrayBox& datfab, const Geometry& geom,
                     Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      auto dx = geom.CellSizeArray();

      const int coord_type = geom.Coord();

#if AMREX_SPACEDIM == 2
      auto problo = geom.ProbLoArray();
#endif

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        // Calculate vorticity.
        if (coord_type == 0) {
          // Cartesian

          // dv/dx and dw/dx
          Real vx = 0.5_rt * (dat(i+1,j,k,2) / dat(i+1,j,k,0) -
                              dat(i-1,j,k,2) / dat(i-1,j,k,0)) / dx[0];
          Real wx = 0.5_rt * (dat(i+1,j,k,3) / dat(i+1,j,k,0) -
                              dat(i-1,j,k,3) / dat(i-1,j,k,0)) / dx[0];

          Real uy = 0.0;
          Real wy = 0.0;

#if AMREX_SPACEDIM >= 2
          // du/dy and dw/dy
          uy = 0.5_rt * (dat(i,j+1,k,1) / dat(i,j+1,k,0) -
                         dat(i,j-1,k,1) / dat(i,j-1,k,0)) / dx[1];
          wy = 0.5_rt * (dat(i,j+1,k,3) / dat(i,j+1,k,0) -
                         dat(i,j-1,k,3) / dat(i,j-1,k,0)) / dx[1];
#endif

          Real uz = 0.0;
          Real vz = 0.0;

#if AMREX_SPACEDIM == 3
          // du/dz and dv/dz
          uz = 0.5_rt * (dat(i,j,k+1,1) / dat(i,j,k+1,0) -
                         dat(i,j,k-1,1) / dat(i,j,k-1,0)) / dx[2];
          vz = 0.5_rt * (dat(i,j,k+1,2) / dat(i,j,k+1,0) -
                         dat(i,j,k-1,2) / dat(i,j,k-1,0)) / dx[2];
#endif

          // curl in Cartesian coords
         Real v1 = wy - vz;
         Real v2 = uz - wx;
         Real v3 = vx - uy;
         der(i,j,k,0) = std::sqrt(v1*v1 + v2*v2 + v3*v3);

        } else if (coord_type == 1) {
#if AMREX_SPACEDIM == 2
          // 2-d axisymmetric -- the coordinate ordering is r, z, phi

          Real r = (static_cast<Real>(i) + 0.5_rt)*dx[0] + problo[0];
          Real rm1 = (static_cast<Real>(i) - 0.5_rt)*dx[0] + problo[0];
          Real rp1 = (static_cast<Real>(i) + 1.5_rt)*dx[0] + problo[0];

          // dv_r/dz
          Real vr_z = 0.5_rt * (dat(i,j+1,k,1) / dat(i,j+1,k,0) -
                                dat(i,j-1,k,1) / dat(i,j-1,k,0)) / dx[1];

          // dv_phi/dz
          Real vphi_z = 0.5_rt * (dat(i,j+1,k,3) / dat(i,j+1,k,0) -
                                  dat(i,j-1,k,3) / dat(i,j-1,k,0)) / dx[1];

          // d (r v_phi)/dr
          Real rvphi_r = 0.5_rt * (rp1 * dat(i+1,j,k,3) / dat(i+1,j,k,0) -
                                   rm1 * dat(i-1,j,k,3) / dat(i-1,j,k,0)) / dx[0];

          // dv_z/dr
          Real vz_r = 0.5_rt * (dat(i+1,j,k,2) / dat(i+1,j,k,0) -
                                dat(i-1,j,k,2) / dat(i-1,j,k,0)) / dx[0];

          der(i,j,k,0) = std::sqrt(vphi_z*vphi_z +
                                   (vr_z - vz_r)*(vr_z - vz_r) +
                                   (rvphi_r/r)*(rvphi_r/r));
#else
          // for 1-d axisymmetric, we just set vorticity to 0
          der(i,j,k,0) = 0.0_rt;
#endif

        } else if (coord_type == 2) {
#if AMREX_SPACEDIM == 2
          // 2D spherical -- the coordinate ordering is r, theta, phi
          // Ignore d/dphi for azimuthal symmetry

          Real r = (static_cast<Real>(i) + 0.5_rt)*dx[0] + problo[0];
          Real rm1 = (static_cast<Real>(i) - 0.5_rt)*dx[0] + problo[0];
          Real rp1 = (static_cast<Real>(i) + 1.5_rt)*dx[0] + problo[0];

          Real sinc = std::sin((static_cast<Real>(j) + 0.5_rt)*dx[1] + problo[1]);
          Real sinm1 = std::sin((static_cast<Real>(j) - 0.5_rt)*dx[1] + problo[1]);
          Real sinp1 = std::sin((static_cast<Real>(j) + 1.5_rt)*dx[1] + problo[1]);

          // d (sin(theta) v_phi)/dtheta
          Real sinvphi_theta = 0.5_rt * (sinp1 * dat(i+1,j,k,3) / dat(i+1,j,k,0) -
                                         sinm1 * dat(i-1,j,k,3) / dat(i-1,j,k,0)) / dx[1];

          // d (r v_phi)/dr
          Real rvphi_r = 0.5_rt * (rp1 * dat(i+1,j,k,3) / dat(i+1,j,k,0) -
                                   rm1 * dat(i-1,j,k,3) / dat(i-1,j,k,0)) / dx[0];

          // d (r v_theta)/dr
          Real rvtheta_r = 0.5_rt * (rp1 * dat(i+1,j,k,2) / dat(i+1,j,k,0) -
                                   rm1 * dat(i-1,j,k,2) / dat(i-1,j,k,0)) / dx[0];

          // dv_r/dtheta
          Real vr_theta = 0.5_rt * (dat(i+1,j,k,1) / dat(i+1,j,k,0) -
                                    dat(i-1,j,k,1) / dat(i-1,j,k,0)) / dx[1];

          der(i,j,k,0) = std::sqrt((sinvphi_theta/sinc)*(sinvphi_theta/sinc) +
                                   (rvtheta_r - vr_theta)*(rvtheta_r - vr_theta) +
                                   (rvphi_r*rvphi_r)) / r;
#else
          // 1-d spherical -- we don't really have a vorticity in this case
          der(i,j,k,0) = 0.0;
#endif
        }
      });
    }

  void ca_derdivu(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                         const FArrayBox& datfab, const Geometry& geom,
                         Real /*time*/, const int* /*bcrec*/, int /*level*/)
    {

      auto const dat = datfab.array();
      auto const der = derfab.array();

      auto dx = geom.CellSizeArray();

      auto problo = geom.ProbLoArray();

      const int coord_type = geom.Coord();

      amrex::ParallelFor(bx,
      [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
      {

        Real uhi = dat(i+1,j,k,1) / dat(i+1,j,k,0);
        Real ulo = dat(i-1,j,k,1) / dat(i-1,j,k,0);

#if AMREX_SPACEDIM >= 2
        Real vhi = dat(i,j+dg1,k,2) / dat(i,j+dg1,k,0);
        Real vlo = dat(i,j-dg1,k,2) / dat(i,j-dg1,k,0);
#endif

#if AMREX_SPACEDIM == 3
        Real whi = dat(i,j,k+dg2,3) / dat(i,j,k+dg2,0);
        Real wlo = dat(i,j,k-dg2,3) / dat(i,j,k-dg2,0);
#endif

        if (coord_type == 0) {
          // Cartesian divergence

          der(i,j,k,0) = 0.5_rt * (uhi - ulo) / dx[0];
#if AMREX_SPACEDIM >= 2
          der(i,j,k,0) += 0.5_rt * (vhi - vlo) / dx[1];
#endif
#if AMREX_SPACEDIM == 3
          der(i,j,k,0) += 0.5_rt * (whi - wlo) / dx[2];
#endif
        } else if (coord_type == 1) {
          // axisymmetric divergence -- defined only for 2-d axisymmetric

          Real r = (static_cast<Real>(i) + 0.5_rt)*dx[0] + problo[0];
          Real rm1 = (static_cast<Real>(i) - 0.5_rt)*dx[0] + problo[0];
          Real rp1 = (static_cast<Real>(i) + 1.5_rt)*dx[0] + problo[0];

          der(i,j,k,0) = 0.5_rt * (rp1*uhi - rm1*ulo) / (r*dx[0]);
#if AMREX_SPACEDIM >= 2
          der(i,j,k,0) += 0.5_rt * (vhi - vlo) / dx[1];
#endif

        } else if (coord_type == 2) {
          // Spherical geometry

          Real r = (static_cast<Real>(i) + 0.5_rt)*dx[0] + problo[0];
          Real rm1 = (static_cast<Real>(i) - 0.5_rt)*dx[0] + problo[0];
          Real rp1 = (static_cast<Real>(i) + 1.5_rt)*dx[0] + problo[0];

          der(i,j,k,0) = 0.5_rt * (rp1*rp1*uhi - rm1*rm1*ulo) / (r*r*dx[0]);
#if AMREX_SPACEDIM == 2
          Real sinc = std::sin((static_cast<Real>(j) + 0.5_rt)*dx[1] + problo[1]);
          Real sinm1 = std::sin((static_cast<Real>(j) - 0.5_rt)*dx[1] + problo[1]);
          Real sinp1 = std::sin((static_cast<Real>(j) + 1.5_rt)*dx[1] + problo[1]);

          der(i,j,k,0) += 0.5_rt * (vhi*sinp1 - vlo*sinm1) / (r*sinc*dx[1]);
#endif
        }

      });
  }

  void ca_derstate(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                   const FArrayBox& datfab, const Geometry& /*geom*/,
                   Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    auto const dat = datfab.array();
    auto const der = derfab.array();

    amrex::ParallelFor(bx,
    [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
    {

      // density
      der(i,j,k,0) = dat(i,j,k,0);

      // temperature
      der(i,j,k,1) = dat(i,j,k,1);

      // (rho X)_1 = X_1
      der(i,j,k,2) = dat(i,j,k,2) / dat(i,j,k,0);

    });

  }

#ifdef MHD
  void ca_dermagcenx(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                     const FArrayBox& datfab, const Geometry& /*geom*/,
                     Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    auto const dat = datfab.array();
    auto const der = derfab.array();

    amrex::ParallelFor(bx,
    [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
    {
      der(i,j,k,0) = 0.5_rt * (dat(i,j,k,0) + dat(i+1,j,k,0));
    });

  }

  void ca_dermagceny(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                     const FArrayBox& datfab, const Geometry& /*geom*/,
                     Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    auto const dat = datfab.array();
    auto const der = derfab.array();

    amrex::ParallelFor(bx,
    [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
    {
      der(i,j,k,0) = 0.5_rt * (dat(i,j,k,0) + dat(i,j+1,k,0));
    });

  }

  void ca_dermagcenz(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                     const FArrayBox& datfab, const Geometry& /*geom*/,
                     Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    auto const dat = datfab.array();
    auto const der = derfab.array();

    amrex::ParallelFor(bx,
    [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
    {
      der(i,j,k,0) = 0.5_rt * (dat(i,j,k,0) + dat(i,j,k+1,0));
    });

  }

  void ca_derdivb(const Box& bx, FArrayBox& derfab, int /*dcomp*/, int /*ncomp*/,
                  const FArrayBox& datfab, const Geometry& geom,
                  Real /*time*/, const int* /*bcrec*/, int /*level*/)
  {

    auto const dat = datfab.array();
    auto const der = derfab.array();

    auto dx = geom.CellSizeArray();

    amrex::ParallelFor(bx,
    [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
    {
      Real dBx = dat(i+1,j,k,0) - dat(i,j,k,0);
      der(i,j,k,0) = dBx / dx[0];

#if AMREX_SPACEDIM >= 2
      Real dBy = dat(i,j+1,k,1) - dat(i,j,k,1);
      der(i,j,k,0) += dBy / dx[1];
#endif

#if AMREX_SPACEDIM == 3
      Real dBz = dat(i,j,k+1,2) - dat(i,j,k,2);
      der(i,j,k,0) += dBz / dx[2];
#endif
    });
  }

#endif

#ifdef __cplusplus
}
#endif
