#!/bin/bash

# Enhanced Test Runner - Convenient Wrapper Script
# Provides easy access to common testing scenarios

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
ENHANCED_RUNNER="$SCRIPT_DIR/enhanced_test_runner.py"

# Function to print colored output
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Function to check if enhanced runner exists
check_runner() {
    if [[ ! -f "$ENHANCED_RUNNER" ]]; then
        print_error "Enhanced test runner not found at: $ENHANCED_RUNNER"
        exit 1
    fi
}

# Function to show usage
show_usage() {
    cat << EOF
Enhanced Test Runner - Quick Start Script

USAGE:
    $0 [COMMAND] [OPTIONS]

COMMANDS:
    start          Interactive testing with prompts
    auto           Fully automated testing (no prompts)
    resume         Resume from previous session
    retry          Retry failed tests from previous session
    skip           Skip failed tests and continue with remaining
    fresh          Start fresh (ignore previous progress)
    dry-run        Test configuration without execution
    help           Show this help message

OPTIONS:
    --silent       Suppress real-time test output
    --phase PHASE  Run specific phase only (regular|advanced)

EXAMPLES:
    $0 start                    # Interactive mode with prompts
    $0 auto                     # Fully automated testing
    $0 auto --silent            # Automated silent mode
    $0 resume                   # Resume from interruption
    $0 retry --silent           # Retry failures silently
    $0 dry-run                  # Test configuration
    
LONG-RUNNING SESSIONS:
    # Use screen/tmux for background execution
    screen -S testing
    $0 auto --silent
    # Detach: Ctrl+A, D
    # Reattach: screen -r testing

MONITORING:
    # Check progress while running
    tail -f tests/comprehensive_test_log_*.csv
    
    # Check organized results
    ls -la runs/organized_results/

EOF
}

# Main execution
main() {
    check_runner
    
    # Check if no arguments provided
    if [[ $# -eq 0 ]]; then
        show_usage
        exit 0
    fi
    
    # Parse command
    COMMAND="$1"
    shift
    
    # Parse additional options
    EXTRA_ARGS=()
    while [[ $# -gt 0 ]]; do
        case $1 in
            --silent)
                EXTRA_ARGS+=("--silent")
                shift
                ;;
            --phase)
                PHASE="$2"
                print_warning "Phase-specific execution not yet implemented in base runner"
                print_info "Will run all phases but you can cancel advanced phase when prompted"
                shift 2
                ;;
            *)
                print_error "Unknown option: $1"
                show_usage
                exit 1
                ;;
        esac
    done
    
    # Execute based on command
    case $COMMAND in
        start)
            print_info "Starting interactive testing session"
            python3 "$ENHANCED_RUNNER" "${EXTRA_ARGS[@]}"
            ;;
        auto)
            print_info "Starting automated testing session"
            python3 "$ENHANCED_RUNNER" -y "${EXTRA_ARGS[@]}"
            ;;
        resume)
            print_info "Resuming from previous session"
            python3 "$ENHANCED_RUNNER" --resume-mode resume "${EXTRA_ARGS[@]}"
            ;;
        retry)
            print_info "Retrying failed tests from previous session"
            python3 "$ENHANCED_RUNNER" --resume-mode retry "${EXTRA_ARGS[@]}"
            ;;
        skip)
            print_info "Skipping failed tests and continuing with remaining"
            python3 "$ENHANCED_RUNNER" --resume-mode skip "${EXTRA_ARGS[@]}"
            ;;
        fresh)
            print_info "Starting fresh (ignoring previous progress)"
            python3 "$ENHANCED_RUNNER" --resume-mode fresh "${EXTRA_ARGS[@]}"
            ;;
        dry-run)
            print_info "Testing configuration (dry run mode)"
            python3 "$ENHANCED_RUNNER" --dry-run "${EXTRA_ARGS[@]}"
            ;;
        help|--help|-h)
            show_usage
            ;;
        *)
            print_error "Unknown command: $COMMAND"
            echo
            show_usage
            exit 1
            ;;
    esac
    
    # Check exit status
    EXIT_CODE=$?
    if [[ $EXIT_CODE -eq 0 ]]; then
        print_success "Command completed successfully"
        print_info "Check results in:"
        print_info "  - CSV log: tests/comprehensive_test_log_*.csv"
        print_info "  - Organized results: runs/organized_results/"
        print_info "  - Checkpoint: tests/testing_checkpoint.json"
    else
        print_error "Command failed with exit code: $EXIT_CODE"
        if [[ -f "tests/testing_checkpoint.json" ]]; then
            print_info "Progress saved. You can resume with: $0 resume"
        fi
    fi
    
    exit $EXIT_CODE
}

# Handle Ctrl+C gracefully
trap 'print_warning "Interrupted by user. Progress saved to checkpoint."; exit 130' INT

# Run main function
main "$@" 