import yaml
import os
from pathlib import Path

def load_config(config_path=None):
    """
    Load configuration from scripts/config.yml with dynamic project root detection.
    
    Args:
        config_path: Optional explicit path to config file. If None, will dynamically
                    find project root and construct path to scripts/config.yml.
    
    Returns:
        dict: Configuration dictionary loaded from YAML file.
    """
    if config_path is None:
        # Dynamically find project root by traversing up from current file
        current_file = Path(__file__).resolve()
        
        # Search for project root markers going up the directory tree
        for parent in current_file.parents:
            # Check for common project root markers
            if any(marker.exists() for marker in [
                parent / '.git',
                parent / 'pyproject.toml',
                parent / 'setup.py',
                parent / 'requirements.txt',
                parent / 'start.py'  # Specific to this project
            ]):
                project_root = parent
                break
        else:
            # Fallback: assume we're in scripts/utils/ and go up two levels
            project_root = current_file.parent.parent.parent
        
        # Construct absolute path to config.yml
        config_path = project_root / 'scripts' / 'config.yml'
    
    with open(config_path, 'r') as file:
        config = yaml.safe_load(file)
    return config