import pandas as pd
import os
from glob import glob

def combine_results(time):
    # Define base directory
    base_dir = os.path.join("runs", f'{time}-all')

    # Collect all CSV files in the base directory and subdirectories
    all_files = glob(os.path.join(base_dir, "**", "*.csv"), recursive=True)

    # Organize by filenames (ignoring parent folder)
    files_by_name = {}
    for file_path in all_files:
        filename = os.path.basename(file_path)
        if filename not in files_by_name:
            files_by_name[filename] = []
        files_by_name[filename].append(file_path)

    for filename, file_paths in files_by_name.items():
        if len(file_paths) > 1:  # Only process repeating files
            combined_df = pd.DataFrame()
            iteration_zero_processed = False
            
            for file_path in file_paths:
                # Get model and framework
                normalized_path = os.path.normpath(file_path)
                parent_dir = os.path.basename(os.path.dirname(normalized_path))
                model, framework = parent_dir.split('_')
                
                # Read CSV
                df = pd.read_csv(file_path)
                df.columns = df.columns.str.strip()  # Strip any leading/trailing spaces
                
                # Fill null values in 'percentage change' column with 0
                df['percentage change'] = df['percentage change'].fillna(0)
                
                # Ensure model and framework columns exist
                df['model'] = None
                df['framework'] = None
                
                # Process rows where iteration == 0
                if iteration_zero_processed:
                    df = df[df['iteration'] != 0]
                else:
                    df.loc[df['iteration'] == 0, ['model', 'framework']] = None
                    iteration_zero_processed = True
                
                # Include model and framework for all other iterations
                df.loc[df['iteration'] != 0, 'model'] = model
                df.loc[df['iteration'] != 0, 'framework'] = framework
                
                # Append to combined DataFrame
                combined_df = pd.concat([combined_df, df], ignore_index=True)
            
            # Save
            output_file = os.path.join(base_dir, f"combined_{filename}")
            combined_df.to_csv(output_file, index=False)
            print(f"Combined file saved as '{output_file}'")