import os
import sys
import shutil
import argparse
import subprocess
from datetime import datetime
import pandas as pd
from crewai import Crew, Process
from tasks import optimize_task, optimitize_again
from agents import create_optimizer
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '../../')))
import scripts.utils.notebook_converter as nc
from scripts.utils.visualize import plot_results
from scripts.utils.diff import write_code_diff
from scripts.utils.save_run_results import save_results
from scripts.utils.parse_agent_output import parse
from scripts.utils.config_loader import load_config
from scripts.utils.artifact_utils import determine_execution_mode, ExecutionMode

def main(input_path, model, iterations=1):
    # Load configuration
    config = load_config()

    time = datetime.now().strftime("%Y%m%d_%H%M%S")
    
    dir_name = time
    run_dir = os.path.join('runs', dir_name)
    
    os.makedirs(run_dir, exist_ok=True)

    if os.path.isfile(input_path) and input_path.endswith('.ipynb'):
        files = [input_path]
    else:
        files = [f for f in os.listdir(input_path) if os.path.isfile(os.path.join(input_path, f)) and f.endswith('.ipynb')]

    for file in files:
        file = os.path.join(input_path, file) if os.path.isdir(input_path) else file
        
        # Determine execution mode and framework
        mode = determine_execution_mode(file)
        
        # Determine the framework and select the appropriate interpreter
        framework = "scope_rl" if any(keyword in file for keyword in ["scope_rl", "basic", "rtb", "rec"]) else "obp"
        notebook_interpreter = config['settings']['interpreter_map'].get(framework, config['settings']['notebook_interpreter'])
        
        filename = os.path.basename(file).replace(".ipynb", ".py")
        new_filename = f'0-{filename}'
        file_to_run = os.path.join(run_dir, new_filename)

        # Convert our Jupyter notebook to .py file so it can be ran externally
        nc.convert_notebook_to_script(file, file_to_run)

        # Run the code once for initial set of results
        print("Running notebook for initial results...")
        subprocess.run(f"{notebook_interpreter} {file_to_run}", shell=True)
        # Save results from generated file out.csv and deleted it afterwards
        print("Saving initial results...")
        csv_file_path = f'{os.path.join(run_dir, filename[:-3])}_results.csv'
        save_results("out.csv", csv_file_path)
        initial_df = pd.read_csv(csv_file_path)
        print("Initial CSV File Contents:\n", initial_df)
        csv_string = initial_df.to_csv(index=False)

        optimizer = create_optimizer(model)
        optimize_task.agent = optimizer
        optimitize_again.agent = optimizer
        
        for i in range(iterations):
            print("///////////////////////////////////////////////////////////")
            print("Run no: ", i+1)
            print("///////////////////////////////////////////////////////////")
            prev_file = file_to_run
            file_to_run = file_to_run.replace(f'{i}-{filename}', f'{i+1}-{filename}')
            shutil.copy(prev_file, file_to_run)
            
            df = pd.read_csv(csv_file_path)
            csv_string = df.to_csv(index=False)
            
            task = optimize_task if i == 0 else optimitize_again
            crew = Crew(
                agents=[optimizer],
                tasks=[task],
                process=Process.sequential,
            )
            result = crew.kickoff(inputs={'file_path': file_to_run, 'results_file': csv_string})
            code = parse(result.raw)
            
            with open(file_to_run, "w") as f:
                f.write(code)
            print("File generated with agent's changes")
            # Compare the previous iteration with the current iteration
            write_code_diff(prev_file, file_to_run)
            print("Running notebook")
            try:
                subprocess.run(f"{notebook_interpreter} {file_to_run}", shell=True, check=True)
            except Exception as e:
                print("Error: Failed to run notebook", e)
                return
            save_results("out.csv", f'{os.path.join(run_dir, filename[:-3])}_results.csv')
            
            # Visualize the recorded results  
        plot_results(f'{os.path.join(run_dir, filename[:-3])}_results.csv')

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Use Crew to optimize hyperparameters in a notebook or py file.")
    parser.add_argument("input_path", type=str, help="Path to the input notebook or .py file")
    parser.add_argument("model", type=str, help="Name of the model to run")
    parser.add_argument("-n", "--iterations", type=int, default=1, help="Number of iterations to run the agent (default: 1)")

    args = parser.parse_args()

    main(args.input_path, args.model, args.iterations)
