import os
import shutil
import argparse
import subprocess
from datetime import datetime
import pandas as pd
import autogen
from autogen import ConversableAgent
import sys
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '../../')))
from scripts.utils.load_prompts import load_prompts
import scripts.utils.notebook_converter as nc
from scripts.utils.visualize import plot_results
from scripts.utils.diff import write_code_diff
from scripts.utils.save_run_results import save_results
from scripts.utils.parse_agent_output import parse
from scripts.utils.config_loader import load_config

def main(input_path, model, iterations=1):
    # Load configuration
    config = load_config()
    notebook_interpreter = config['settings']['notebook_interpreter']
    
    config_list = autogen.config_list_from_json(
        "scripts/API_KEYS",
        filter_dict={
            "model": [model],
        },
    )
    
    llm_config={"config_list": config_list,
                 "cache_seed": None,}
    prompts = load_prompts()

    time = datetime.now().strftime("%Y%m%d_%H%M%S")
    
    dir_name = time
    run_dir = os.path.join('runs', dir_name)
    
    os.makedirs(run_dir, exist_ok=True)

    if os.path.isfile(input_path) and input_path.endswith('.ipynb'):
        files = [input_path]
    else:
        files = [f for f in os.listdir(input_path) if os.path.isfile(os.path.join(input_path, f)) and f.endswith('.ipynb')]

    for file in files:
        file = os.path.join(input_path, file) if os.path.isdir(input_path) else file
        
        filename = os.path.basename(file).replace(".ipynb", ".py")
        new_filename = f'0-{filename}'
        file_to_run = os.path.join(run_dir, new_filename)

        # Convert our Jupyter notebook to .py file so it can be ran externally
        nc.convert_notebook_to_script(file, file_to_run)

        # Run the code once for initial set of results
        print("Running notebook for initial results...")
        subprocess.run(f"{notebook_interpreter} {file_to_run}", shell=True)
        # Save results from generated file out.csv and deleted it afterwards
        print("Saving initial results...")
        csv_file_path = f'{os.path.join(run_dir, filename[:-3])}_results.csv'
        save_results("out.csv", csv_file_path)
        initial_df = pd.read_csv(csv_file_path)
        print("Initial CSV File Contents:\n", initial_df)

        # Set up the generative model and start chat
        message_history = []
        
        # Initial prompt message
        intial_prompt = f"""
            {prompts['system_message']} The results that it generates are in the csv file and 
            are also provided. {prompts['whole_code']}
        """
        
        # Prompt for all further iterations
        improvement_prompt = """
            Implementing the changes you suggested yields the following results. Keeping in mind 
            what changes you suggested earlier and the results that were generated, make some
            further changes to the code that could help decrease the relative error estimation
            even further. Please output the entire code I give you and do it in one codeblock.
        """
        agent = ConversableAgent(
            "chatbot",
            llm_config=llm_config,
            code_execution_config=False,
            function_map=None,
            human_input_mode="NEVER",
        )
        
        for i in range(iterations):
            print("///////////////////////////////////////////////////////////")
            print("Run no: ", i+1)
            print("///////////////////////////////////////////////////////////")
            prev_file = file_to_run
            file_to_run = file_to_run.replace(f'{i}-{filename}', f'{i+1}-{filename}')
            shutil.copy(prev_file, file_to_run)
            with open(file_to_run, "r") as f:
                notebook_content = f.read()
            df = pd.read_csv(csv_file_path)
            csv_string = df.to_csv(index=False)
            
            message = intial_prompt if i == 0 else improvement_prompt
            message_history.append({"role": "user", "content": 
                f"""
                {message}
                {notebook_content}
                Run results:
                {csv_string}
                """})
            chat_response = agent.generate_reply(messages = message_history)
            try:
                agent_message = chat_response["content"]
            except:
                agent_message = chat_response
            print(agent_message)
            message_history.append({"role": "assistant", "content": agent_message})
            
            code = parse(agent_message)
            
            with open(file_to_run, "w") as f:
                f.write(code)
            print("File generated with agent's changes")
            # Compare the previous iteration with the current iteration
            write_code_diff(prev_file, file_to_run)
            print("Running notebook")
            try:
                subprocess.run(f"{notebook_interpreter} {file_to_run}", shell=True, check=True)
            except Exception as e:
                print("Error: Failed to run notebook", e)
                return
            save_results("out.csv", f'{os.path.join(run_dir, filename[:-3])}_results.csv')
            
            # Visualize the recorded results  
        plot_results(f'{os.path.join(run_dir, filename[:-3])}_results.csv')

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Use AutoGen to optimize hyperparameters in a notebook or py file.")
    parser.add_argument("input_path", type=str, help="Path to the input notebook or .py file")
    parser.add_argument("model", type=str, help="Name of the model to run")
    parser.add_argument("-n", "--iterations", type=int, default=1, help="Number of iterations to run the agent (default: 1)")

    args = parser.parse_args()

    main(args.input_path, args.model, args.iterations)
