"""Task definitions for the two-agent optimisation framework.

Defines the task workflow for the iterative optimisation process:
1. Analysis Task: Examine code and identify improvement opportunities
2. Coding Task: Implement the suggested improvements  
3. Evaluation Task: Run improved code and capture results
4. Summary Task: Analyse all iterations and identify best performers
"""

import sys
import os
from typing import Dict, Any, List, Tuple
from dataclasses import dataclass

# Add scripts to path for utilities
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '../../../')))
from scripts.utils.load_prompts import load_prompts

@dataclass
class TaskResult:
    """Standard result structure for task outputs."""
    task_type: str
    iteration: int
    success: bool
    output: Any
    error_message: str = ""

class TwoAgentTasks:
    """Task orchestrator for the two-agent framework.
    
    Manages the workflow of analysis, coding, evaluation, and summarisation tasks
    across multiple iterations.
    """
    
    def __init__(self, agents: Dict[str, Dict[str, Any]]):
        """Initialise task orchestrator with agent instances.
        
        Args:
            agents: Dictionary containing initialised agent instances
        """
        self.agents = agents
        self.prompts = load_prompts()
        
    def execute_analysis_task(self, code: str, framework: str, iteration: int) -> TaskResult:
        """Execute code analysis task to identify improvement opportunities.
        
        Args:
            code: Python code to analyse
            framework: Detected framework type ('obp', 'scope_rl', 'unknown')
            iteration: Current iteration number
            
        Returns:
            TaskResult with analysis output
        """
        try:
            analyse_agent = self.agents['analyse']
            result = analyse_agent['analyse'](code, framework)
            
            return TaskResult(
                task_type="analysis",
                iteration=iteration,
                success=True,
                output=result.analysis
            )
            
        except Exception as e:
            return TaskResult(
                task_type="analysis",
                iteration=iteration,
                success=False,
                output=None,
                error_message=f"Analysis task failed: {str(e)}"
            )
    
    def execute_coding_task(self, original_code: str, analysis: str, framework: str, iteration: int) -> TaskResult:
        """Execute code generation task to implement improvements.
        
        Args:
            original_code: Original Python code
            analysis: Analysis output from previous task
            framework: Detected framework type for context
            iteration: Current iteration number
            
        Returns:
            TaskResult with improved code
        """
        try:
            pcoder_agent = self.agents['pcoder']
            result = pcoder_agent['generate_code'](original_code, analysis, framework)
            
            return TaskResult(
                task_type="coding",
                iteration=iteration,
                success=True,
                output=result.code
            )
            
        except Exception as e:
            return TaskResult(
                task_type="coding",
                iteration=iteration,
                success=False,
                output=None,
                error_message=f"Coding task failed: {str(e)}"
            )
    
    def execute_evaluation_task(self, code_path: str, results_path: str, iteration: int) -> TaskResult:
        """Execute code evaluation task by running the improved code.
        
        Args:
            code_path: Path to the code file to execute
            results_path: Path where results will be saved
            iteration: Current iteration number
            
        Returns:
            TaskResult with execution results
        """
        try:
            import subprocess
            import os
            
            # Run code using subprocess
            result = subprocess.run(
                f"python {code_path}",
                shell=True,
                capture_output=True,
                text=True,
                timeout=300  # 5-minute timeout
            )
            
            if result.returncode == 0:
                # Read results if they were generated
                if os.path.exists(results_path):
                    with open(results_path, 'r') as f:
                        results_content = f.read()
                else:
                    results_content = "No results file generated"
                
                return TaskResult(
                    task_type="evaluation",
                    iteration=iteration,
                    success=True,
                    output=results_content
                )
            else:
                return TaskResult(
                    task_type="evaluation",
                    iteration=iteration,
                    success=False,
                    output=None,
                    error_message=f"Code execution failed: {result.stderr}"
                )
                
        except Exception as e:
            return TaskResult(
                task_type="evaluation",
                iteration=iteration,
                success=False,
                output=None,
                error_message=f"Evaluation task failed: {str(e)}"
            )
    
    def execute_summary_task(self, all_results: str) -> TaskResult:
        """Execute summary task to analyse all iteration results.
        
        Args:
            all_results: Combined results from all iterations
            
        Returns:
            TaskResult with summary analysis
        """
        try:
            summariser_agent = self.agents['summariser']
            result = summariser_agent['summarise'](all_results)
            
            return TaskResult(
                task_type="summary",
                iteration=-1,  # Summary spans all iterations
                success=True,
                output=result.summary
            )
            
        except Exception as e:
            return TaskResult(
                task_type="summary",
                iteration=-1,
                success=False,
                output=None,
                error_message=f"Summary task failed: {str(e)}"
            )
    
    def get_task_workflow(self) -> List[str]:
        """Get the standard task workflow for two-agent optimisation.
        
        Returns:
            List of task types in execution order
        """
        return ["analysis", "coding", "evaluation"]
    
    def get_framework_specific_prompts(self, framework: str) -> Dict[str, str]:
        """Get framework-specific prompts for task execution.
        
        Args:
            framework: Detected framework type
            
        Returns:
            Dictionary of framework-specific prompt modifications
        """
        framework_prompts = {
            'obp': {
                'analysis_focus': 'OBP estimators and hyperparameter tuning',
                'coding_focus': 'Optimise OBP-specific parameters and estimators'
            },
            'scope_rl': {
                'analysis_focus': 'SCOPE-RL policy optimisation and OPE parameters',
                'coding_focus': 'Tune CreateOPEInput and policy head parameters'
            },
            'unknown': {
                'analysis_focus': 'General machine learning optimisations',
                'coding_focus': 'Improve model performance and accuracy'
            }
        }
        
        return framework_prompts.get(framework, framework_prompts['unknown'])
    
    def validate_task_dependencies(self, task_sequence: List[str]) -> bool:
        """Validate that task sequence follows proper dependencies.
        
        Args:
            task_sequence: List of tasks to validate
            
        Returns:
            True if sequence is valid, False otherwise
        """
        valid_sequences = [
            ["analysis", "coding", "evaluation"],
            ["analysis", "coding"],  # If evaluation is handled externally
            ["summary"]  # Final summary task
        ]
        
        return task_sequence in valid_sequences or task_sequence == ["analysis", "coding", "evaluation", "summary"] 