"""IO manager for two-agent framework.

Handles working directory setup, file operations, and diff generation.
"""
import os
import difflib
from pathlib import Path
from datetime import datetime
from typing import Optional


class IOManager:
    """Manages file operations and working directories for two-agent framework."""
    
    def __init__(self, input_path: str):
        """Initialise IO manager with input path."""
        self.input_path = input_path
        self.run_dir: Optional[Path] = None
        self.results_dir: Optional[Path] = None
        self.docs_dir: Optional[Path] = None
        self.code_file: Optional[Path] = None
        self.result_file: Optional[Path] = None
    
    def setup_working_directories(self) -> None:
        """Set up working directories for the optimisation run."""
        # Check if we're being called from within an existing run directory
        input_dir = os.path.dirname(os.path.abspath(self.input_path))
        
        # If input_path is in a 'runs/' directory, use that as our base
        if 'runs' in input_dir and os.path.basename(input_dir).startswith(('20', 'gemini', 'gpt', 'mistral')):
            # We're being called by start.py from within a run directory
            self.run_dir = Path(input_dir) / "two_agent_results"
            print(f"Detected execution from run directory: {input_dir}")
            print(f"Creating two-agent subdirectory: {self.run_dir}")
        else:
            # Standalone execution - create our own timestamped directory
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            base_name = os.path.splitext(os.path.basename(self.input_path))[0]
            self.run_dir = Path(f"runs/two_agent_{base_name}_{timestamp}")
            print(f"Standalone execution - creating new directory: {self.run_dir}")
        
        # Create subdirectory structure
        self.results_dir = self.run_dir / "Results"
        self.docs_dir = self.run_dir / "R-doc"
        
        # Create directories
        for directory in [self.run_dir, self.results_dir, self.docs_dir]:
            directory.mkdir(parents=True, exist_ok=True)
            
        # Set up file paths
        self.code_file = self.results_dir / "pycode.py"
        self.result_file = self.results_dir / "result.txt"
        
        print(f"Two-agent working directory: {self.run_dir}")
    
    def write_file(self, file_path: str, content: str) -> None:
        """Write content to a file."""
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(content)
    
    def empty_file(self, file_path: str) -> None:
        """Clear the contents of a file."""
        with open(file_path, 'w') as f:
            f.write('')
    
    def generate_code_diff(self, old_code: str, new_code: str, iteration: int) -> str:
        """Generate unified diff between old and new code.
        
        Args:
            old_code: Previous iteration's code
            new_code: Current iteration's code  
            iteration: Current iteration number
            
        Returns:
            Unified diff as a string
        """
        old_lines = old_code.splitlines(keepends=True)
        new_lines = new_code.splitlines(keepends=True)
        
        diff = difflib.unified_diff(
            old_lines,
            new_lines,
            fromfile=f"iteration_{iteration-1}.py" if iteration > 1 else "original.py",
            tofile=f"iteration_{iteration}.py",
            lineterm='',
            n=3  # Context lines
        )
        
        diff_text = ''.join(diff)
        
        # Add summary header
        summary = f"""# Code Diff - Iteration {iteration}

## Summary
- Previous: {'iteration_{}'.format(iteration-1) if iteration > 1 else 'original'}.py
- Current: iteration_{iteration}.py
- Generated: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}

## Changes
"""
        
        return summary + diff_text 