"""Executor for two-agent framework.

Handles framework detection, interpreter selection, and code execution.
"""
import os
import subprocess
from pathlib import Path
from typing import Optional


def detect_notebook_framework(script_path: str) -> str:
    """Detect framework based on Python script content and path patterns.
    
    Args:
        script_path: Path to Python script (converted from notebook)
    
    Returns:
        'scope_rl' for SCOPE-RL scripts
        'obp' for Open Bandit Pipeline scripts  
        'unknown' if cannot determine
    """
    try:
        # Read the Python script content (converted from notebook)
        with open(script_path, 'r', encoding='utf-8') as f:
            script_content = f.read()
        
        # Check for SCOPE-RL patterns in the script
        scope_patterns = ['import scope_rl', 'from scope_rl', 'scope_rl.', 'SyntheticDataset', 'BasicEnv', 'RECEnv', 'RTBEnv']
        if any(pattern.lower() in script_content.lower() for pattern in scope_patterns):
            return 'scope_rl'
        
        # Check for OBP patterns in the script
        obp_patterns = ['import obp', 'from obp', 'obp.', 'OpenBanditDataset']
        if any(pattern.lower() in script_content.lower() for pattern in obp_patterns):
            return 'obp'
        
        # Fallback: Check path patterns (basic/rec/rtb are usually SCOPE-RL)
        if any(keyword in script_path.lower() for keyword in ['/basic/', '/rec/', '/rtb/', 'basicenv', 'recenv', 'rtbenv']):
            return 'scope_rl'
            
        return 'unknown'
        
    except Exception as e:
        print(f"Warning: Framework detection failed for {script_path}: {e}")
        return 'unknown'


def pick_interpreter(script_path: str, config_settings: dict) -> str:
    """Select appropriate Python interpreter based on detected framework.
    
    Args:
        script_path: Path to the converted Python script
        config_settings: Settings section from config.yml
        
    Returns:
        Path to the appropriate Python interpreter
    """
    framework = detect_notebook_framework(script_path)
    
    print(f"Framework detection for {os.path.basename(script_path)}: {framework}")
    
    # Use interpreter mapping from config.yml
    interpreter_map = config_settings.get('interpreter_map', {})
    
    if framework == 'scope_rl':
        interpreter = interpreter_map.get('scope_rl', config_settings.get('notebook_interpreter'))
    elif framework == 'obp':
        interpreter = interpreter_map.get('obp', config_settings.get('notebook_interpreter'))
    else:
        # Fallback to default interpreter
        interpreter = config_settings.get('notebook_interpreter')
    
    print(f"Selected interpreter: {interpreter}")
    return interpreter


class CodeExecutor:
    """Executes Python code using appropriate interpreter."""
    
    def __init__(self, interpreter: str, run_dir: Path):
        """Initialise executor with interpreter and working directory.
        
        Args:
            interpreter: Path to Python interpreter
            run_dir: Working directory for execution
        """
        self.interpreter = interpreter
        self.run_dir = run_dir
    
    def run_code(self, code_path: str) -> Optional[str]:
        """Run Python code using the appropriate interpreter.
        
        Args:
            code_path: Path to Python file to execute
            
        Returns:
            Path to out.csv if successful and CSV is generated, None otherwise
        """
        try:
            # Use the appropriate interpreter for the detected framework
            cmd = f"{self.interpreter} {code_path}"
            
            # Set working directory to the run directory, not the code's directory
            result = subprocess.run(
                cmd,
                shell=True,
                capture_output=True,
                text=True,
                timeout=1200,  # 20-minute timeout
                cwd=self.run_dir  # Use the main run directory as CWD
            )
            
            if result.returncode == 0:
                # Check for out.csv and return its path
                # The CSV will be in the run_dir because of the cwd change
                csv_path = os.path.join(self.run_dir, "out.csv")
                if os.path.exists(csv_path):
                    return csv_path
                else:
                    # Look for out.csv in common locations
                    possible_paths = [
                        os.path.join(self.run_dir, "out.csv"),
                        "out.csv",  # Current working directory
                        os.path.join(self.run_dir, "..", "out.csv"),  # Parent directory
                    ]
                    
                    for path in possible_paths:
                        if os.path.exists(path):
                            return os.path.abspath(path)
                    
                    print("Warning: Code executed successfully but no out.csv was found")
                    return None
            else:
                print(f"Code execution error: {result.stderr}")
                return None
                
        except subprocess.TimeoutExpired:
            print("Code execution timed out after 5 minutes")
            return None
        except Exception as e:
            print(f"Error running code: {str(e)}")
            return None 