import autogen
from autogen import AssistantAgent, UserProxyAgent
import argparse
import sys
import os
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '../../')))
from scripts.utils import (
    load_prompts,
    apply_diff,
    read_file,
    parse
)

def main(input_path, model, option):
    print(f"Running {model}...with option={option}")
    
    config_list = autogen.config_list_from_json(
        "scripts/API_KEYS",
        filter_dict={
            "model": [model],
        },
    )
    
    llm_config={"config_list": config_list}

    prompts = load_prompts()

    if option=="whole_code":
        prompt = prompts['whole_code']
    else:   # option=="manual_patch", option=="agent_applies"
        prompt = prompts['diff']   

    # Declare agents
    if option=="agent_applies":
        user_proxy = UserProxyAgent(
            name="User",
            llm_config=False,
            is_termination_msg=lambda msg: msg.get("content") is not None and "TERMINATE" in msg["content"],
            human_input_mode="NEVER",
            code_execution_config=False,
            )
        edit_applier = AssistantAgent(
            name="Applier",
            system_message="""
            You are a helpful AI assistant that can accurately apply a given code diff to a file. If 
            you want the user to save the code in a file before executing it, put # filename: <filename>
            inside the code block as the first line. Don't include multiple code blocks in one response.
            Do not ask users to copy and paste the result. Instead, use 'print' function for the output
            when relevant. Reply with 'TERMINATE' when you are done.
            """, 
            llm_config=llm_config,
            max_consecutive_auto_reply=3)  
            
    edit_suggester = AssistantAgent(
        name="Suggester", 
        system_message=f"""
        You are a helpful AI assistant that can suggest changes to 
        hyperparameters of machine learning models to improve their 
        performance and output. {prompt} Reply with 'TERMINATE' when you are 
        done.
        """, 
        llm_config=llm_config, 
        max_consecutive_auto_reply=3)

    corpus = read_file(input_path)

    # Start the chat
    if option=="agent_applies":
        chat_result = autogen.initiate_chats(
        [
            {
                "sender": user_proxy,
                "recipient": edit_suggester,
                "message": """The following code performs offline policy
                evaluation and calculates the relative error estimation with 
                various estimators at the end. Please modify the 
                hyperparameters of the file to decrease the relative error 
                estimation.:\n"""+corpus,
                "clear_history": True,
                "silent": False,
                "summary_method": "last_msg",
            },
            {
                "sender": user_proxy,
                "recipient": edit_applier,
                "message": """
                I want you to apply the given diff to the given code. The diff may have 
                incorrect line numbers, so I want you to apply it by checking if it fits into
                place it is being applied to. Here is the code:\n
                """+corpus,
                "clear_history": False,
                "silent": False,
                "summary_method": "last_msg",
            }
        ])
        
        llm_response = chat_result[-1].chat_history[-1]["content"]
        with open(input_path, "w") as f:
            f.write(parse(llm_response))
            
    else:
        reply = edit_suggester.generate_reply(
            messages =[{"content": 
                """The following code performs offline policy
                evaluation and calculates the relative error estimation with 
                various estimators at the end. Please modify the 
                hyperparameters of the file to decrease the relative error 
                estimation.:\n"""+corpus
                + corpus, 
                "role": "user"}])
        # Gemini returns the response as a dictionary while GPT returns the 
        # response as a string. So we need to handle both cases.
        print(reply)
        try:
            parsed_response = parse(reply['content'])
        except:
            parsed_response = parse(reply)
        if option=="whole_code":
            with open(input_path, "w") as f:
                f.write(parsed_response)
            print("Whole code written to file.")
        else:   # option=="manual_patch"  
            print("Applying diff to file...")
            apply_diff(input_path, parsed_response)
        
if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Modify hyperparameters of a notebook to decrease relative error estimation.")
    parser.add_argument("input_path", type=str, help="Path to the input notebook file")
    parser.add_argument("model", type=str, help="Specify the model to use")
    parser.add_argument("-opt", "--option", type=str, default="manual_patch", choices=['manual_patch', 'whole_code', 'agent_applies'], help="Choose how to apply the agent's changes")
    args = parser.parse_args()

    main(args.input_path, args.model, args.option)