import os
import joblib
import pandas as pd
from sklearn.datasets import load_breast_cancer

# Path to weights (can be overridden by ENV var FRA_RIG_WEIGHTS)
PATH = os.getenv("FRA_RIG_WEIGHTS", "models/fra_rig_breast.joblib")

def rig_predict_proba(system, X_new: pd.DataFrame):
    \"\\"Compute block indices (S_R, S_I, S_G) and probability using the trained head.\"\\"
    import numpy as np
    prep = system["prep"]
    head = system["head"]
    num_cols = system["num_cols"]
    blocks = system["blocks"]          # {'R': [...], 'I': [...], 'G': [...]}
    block_w = system["block_weights"]  # {'R': {col: w, ...}, ...}

    # flatten weights per column
    flat_w = {}
    for b in ['R','I','G']:
        for c, v in block_w[b].items():
            flat_w[c] = float(v)

    # transform numeric features
    Z = prep.named_transformers_['num'].transform(X_new[num_cols])
    col_idx = {c:i for i,c in enumerate(num_cols)}

    def block_sum(cols):
        w = np.array([flat_w[c] for c in cols], dtype=float)
        Zb = Z[:, [col_idx[c] for c in cols]]
        return (Zb * w).sum(axis=1)

    S = pd.DataFrame({
        'S_R': block_sum(blocks['R']),
        'S_I': block_sum(blocks['I']),
        'S_G': block_sum(blocks['G']),
    }, index=X_new.index)

    proba = head.predict_proba(S)[:, 1]
    return proba, S

def main():
    if not os.path.exists(PATH):
        print(f"[!] Weights not found: {PATH}")
        print("    Download 'fra_rig_breast.joblib' from https://doi.org/10.5281/zenodo.17489822 ")
        print("    and place it at 'models/fra_rig_breast.joblib' or set FRA_RIG_WEIGHTS env var.")
        return

    system = joblib.load(PATH)

    ds = load_breast_cancer()
    X = pd.DataFrame(ds.data, columns=ds.feature_names)

    proba, S = rig_predict_proba(system, X.iloc[:5])
    print("Probabilities (first 5):", [round(float(p), 3) for p in proba])
    print("\nBlock indices (first 5):\n", S.round(3))

if __name__ == "__main__":
    main()
