from __future__ import annotations

import json
from pathlib import Path

import numpy as np
import pandas as pd
from scipy import stats

from .utils import ROOT


def run_tests(
    indices_path: Path = ROOT / "reports/indices.parquet",
    alpha_path: Path = ROOT / "reports/alpha_phi.parquet",
    out_path: Path = ROOT / "reports/rrce_summary.md",
) -> None:
    # 入力読み込み
    idx = pd.read_parquet(indices_path)
    alp = pd.read_parquet(alpha_path)

    # --- H1: α_φ < 1 ?（呼称で距離が縮むか）---
    a = alp["alpha_phi"].dropna().to_numpy(dtype=float)
    a = a[np.isfinite(a)]

    if a.size > 0:
        diffs = a - 1.0
        # すべて差が 0 の場合は Wilcoxon が定義できないのでスキップ
        if np.allclose(diffs, 0.0):
            stat, p = None, 1.0  # “差なし”扱い
        else:
            stat, p = stats.wilcoxon(diffs, alternative="less", zero_method="wilcox")
        md_alpha = float(np.median(a))
    else:
        stat, p, md_alpha = None, None, float("nan")

    # 参考値: E_score の中央値（H3 の事前確認）
    e = idx["E_score"].to_numpy(dtype=float)
    md_E = float(np.nanmedian(e)) if e.size > 0 else float("nan")

    # 出力整形
    summary = {
        "H1_wilcoxon_stat": None if stat is None else float(stat),
        "H1_p_less_than_1": None if p is None else float(p),
        "median_alpha_phi": md_alpha,
        "median_E_score": md_E,
        "n_alpha_phi": int(a.size),
    }

    # 出力先フォルダを確保
    out_path.parent.mkdir(parents=True, exist_ok=True)

    # Markdown（JSON埋め込み）で保存
    lines = [
        "# RRCE/ARI Summary (auto)",
        "",
        "```json",
        json.dumps(summary, indent=2),
        "```",
        "",
    ]
    out_path.write_text("\n".join(lines), encoding="utf-8")
    print(f"summary -> {out_path}")


if __name__ == "__main__":
    run_tests()