#========================================================================
# Copyright 2021 Science Technology Facilities Council
#
# This file is part of the SyneRBI Synergistic Image Reconstruction Framework (SIRF).
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0.txt
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#=========================================================================

import os
import unittest
import inspect
import functools
import sirf.STIR as pet
from sirf.Utilities import examples_data_path, DataContainerAlgebraTests

pet.AcquisitionData.set_storage_scheme('file')
pet.set_verbosity(0)

class TestSTIRImageDataAlgebra(unittest.TestCase, DataContainerAlgebraTests):
    def setUp(self):
        self.image1 = pet.ImageData(os.path.join(
            examples_data_path('PET'),'thorax_single_slice','emission.hv')
            )
        self.image2 = pet.ImageData(os.path.join(
            examples_data_path('PET'),'thorax_single_slice','emission.hv')
            )

    def tearDown(self):
        # shutil.rmtree(self.cwd)
        pass

class TestSTIRAcquisitionDataAlgebraFile(unittest.TestCase, DataContainerAlgebraTests):

    def setUp(self):
        path = os.path.join(
            examples_data_path('PET'), 'thorax_single_slice', 'template_sinogram.hs')
        if os.path.exists(path):
            template = pet.AcquisitionData(path)
            self.image1 = template.get_uniform_copy(0)
            self.image2 = template.get_uniform_copy(0)
            self.set_storage_scheme()
        
        
    def tearDown(self):
        pass

    def set_storage_scheme(self):
        pet.AcquisitionData.set_storage_scheme('file')
    

class TestSTIRAcquisitionDataAlgebraMemory(unittest.TestCase, DataContainerAlgebraTests):
    def setUp(self):
        pet.AcquisitionData.set_storage_scheme('file')
        path = os.path.join(
            examples_data_path('PET'), 'thorax_single_slice', 'template_sinogram.hs')
        if os.path.exists(path):
            template = pet.AcquisitionData(path)
            self.image1 = template.get_uniform_copy(0)
            self.image2 = template.get_uniform_copy(0)
            pet.AcquisitionData.set_storage_scheme('memory')

class TestSTIRAcquisitionDataSubsetAlgebra():
    def setUp(self):
        path = os.path.join(
            examples_data_path('PET'), 'thorax_single_slice', 'template_sinogram.hs')
        if os.path.exists(path):
            template = pet.AcquisitionData(path)
            prompts = template.get_uniform_copy(0)
            # create a staggered list of views for 2 subsets
            views = prompts.dimensions()[2]
            indices = list(range(views))
            num_batches = 2
            batches = [indices[i::num_batches] for i in range(num_batches)]
            
            self.image1 = prompts.get_subset(batches[0])
            self.image2 = prompts.get_subset(batches[0])
            
            
class TestSTIRAcquisitionDataSubsetAlgebraMemory(TestSTIRAcquisitionDataSubsetAlgebra, unittest.TestCase, DataContainerAlgebraTests):
    def setUp(self):
        pet.AcquisitionData.set_storage_scheme('memory')
        super().setUp()


class TestSTIRAcquisitionDataSubsetAlgebraFile(TestSTIRAcquisitionDataSubsetAlgebra, unittest.TestCase, DataContainerAlgebraTests):
    def setUp(self):
        pet.AcquisitionData.set_storage_scheme('file')
        super().setUp()
        
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        # wrap all test methods that would raise exception as
        # storage_scheme file does not work
        a = inspect.getmembers(DataContainerAlgebraTests, predicate=inspect.isfunction)
        def wrapper(self, func):
            try:
                func(self)
            except Exception as e:
                print (f"Caught exception in {func.__name__}: {e}")
                return
            raise AssertionError(f"Expected exception in {func.__name__} but none was raised.")
        
        for m in a:
            # test_division_by_datacontainer_zero and test_division_by_scalar_zero
            # do not raise exception, they just assert True. Skip them.
            if m[0] not in ["test_division_by_datacontainer_zero", "test_division_by_scalar_zero"]:
                self.__setattr__(m[0], functools.partial(wrapper, self, m[1]))

        self.__setattr__('setUp', functools.partial(wrapper, self, self.setUp))