'''Poisson noise generation demo.

Usage:
  generate_noisy_data [--help | options]

Options:
  -f <file>, --file=<file>     prompts file [default: my_forward_projection.hs]
  -p <path>, --path=<path>     path to data files, defaults to data/examples/PET
                               subfolder of SIRF root folder
  -o <file>, --output=<file>   output file for Poisson noisy data
  -r <seed>, --seed=<seed>     random generator seed [default: 1]
  -F <sf>, --sf=<sf>           scaling factor [default: 1.0] (use a higher value for lower relative noise)
  -m, --pm                     preserve mean
  -e <engn>, --engine=<engn>   reconstruction engine [default: STIR]
  -s <stsc>, --storage=<stsc>  acquisition data storage scheme [default: file]
  --non-interactive            do not show plots
'''

## SyneRBI Synergistic Image Reconstruction Framework (SIRF)
## Copyright 2015 - 2025 Rutherford Appleton Laboratory STFC
## Copyright 2015 - 2025 University College London.
##
## This is software developed for the Collaborative Computational
## Project in Synergistic Reconstruction for Biomedical Imaging (formerly CCP PETMR)
## (http://www.ccpsynerbi.ac.uk/).
##
## Licensed under the Apache License, Version 2.0 (the "License");
##   you may not use this file except in compliance with the License.
##   You may obtain a copy of the License at
##       http://www.apache.org/licenses/LICENSE-2.0
##   Unless required by applicable law or agreed to in writing, software
##   distributed under the License is distributed on an "AS IS" BASIS,
##   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
##   See the License for the specific language governing permissions and
##   limitations under the License.

__version__ = '0.1.0'
from docopt import docopt
args = docopt(__doc__, version=__version__)

from sirf.Utilities import error, examples_data_path, existing_filepath

# import engine module
import importlib
engine = args['--engine']
pet = importlib.import_module('sirf.' + engine)

# process command-line options
data_file = args['--file']
data_path = args['--path']
if data_path is None:
    data_path = examples_data_path('PET')
output_file = args['--output']
seed = int(args['--seed'])
sf = float(args['--sf'])
pm = bool(args['--pm'])
storage = args['--storage']
show_plot = not args['--non-interactive']

# select acquisition data storage scheme
# storage = 'file' (default):
#     all acquisition data generated by the script is kept in
#     scratch files deleted after the script terminates
# storage = 'memory':
#     all acquisition data generated by the script is kept in RAM
#     (avoid if data is very large)
scheme = pet.AcquisitionData.get_storage_scheme()
if scheme != storage:
    print('default storage scheme is %s' % repr(scheme))
    print('setting storage scheme to %s' % repr(storage))
    pet.AcquisitionData.set_storage_scheme(storage)
else:
    print('using default storage scheme %s' % repr(scheme))


def main():
    engine_version = pet.get_engine_version_string()
    print('using %s version %s as the reconstruction engine' % (engine, engine_version))

    # direct all engine's messages to files
    _ = pet.MessageRedirector('info.txt', 'warn.txt', 'errr.txt')

    # PET acquisition data to be read from this file
    prompts_file = existing_filepath(data_path, data_file)
    print('reading prompts from %s' % prompts_file)
    acq_data = pet.AcquisitionData(prompts_file)
    dim = acq_data.dimensions()
    print(f'acquisition data norm: {acq_data.norm()}')

    png = pet.PoissonNoiseGenerator(sf, pm)
    png.set_seed(seed)
    png.process(acq_data)
    noisy_data = png.get_output()
    print(f'noisy data norm: {noisy_data.norm()}')
    if show_plot:
        noisy_data.show(range(dim[1]//4), title='selected noisy sinograms')

    if output_file is not None:
        noisy_data.write(output_file)


try:
    main()
    print('\n=== done with %s' % __file__)

except error as err:
    print('%s' % err.value)

if scheme != storage:
    pet.AcquisitionData.set_storage_scheme(scheme)
