function acquisition_data(engine)
% USING_ACQUISITION_DATA Demo illustrating acquisitions pre-processing 
% and displaying.
%
% In MATLAB, there are also ISMRMRD tools available for examining 
% data before processing.
%
% SyneRBI Synergistic Image Reconstruction Framework (SIRF).
% Copyright 2015 - 2020 Rutherford Appleton Laboratory STFC.
% Copyright 2015 - 2019 University College London.
% 
% This is software developed for the Collaborative Computational
% Project in Synergistic Reconstruction for Biomedical Imaging (formerly CCP PETMR)
% (http://www.ccpsynerbi.ac.uk/).
% 
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
% http://www.apache.org/licenses/LICENSE-2.0
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distributed on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.

% Select and import SIRF MATLAB MR package so that SIRF MR objects can be 
% created in this function without using the prefix 'MR.'
if nargin < 1
    engine = [];
end
MR = set_up_MR(engine);
mr_data_path = sirf.Utilities.examples_data_path('MR');

% all acquisition data generated by this script will be stored in memory
% (the input data will remain in the input file)
AD = MR.AcquisitionData();

% read acquisitions from an HDF file
[filename, pathname] = uigetfile('*.h5', 'Select raw data file', mr_data_path);
acq_data = MR.AcquisitionData(fullfile(pathname, filename));
%acq_data = acq_data.clone(); % R2014b doesn't allow .clone() in the previous line

header = acq_data.get_header();
fprintf(header);

na = acq_data.number();
fprintf('%d acquisitions (readouts) found\n', na)

fprintf('sorting acquisitions...\n')
acq_data.sort()

[ns, nc, na] = acq_data.dimensions();
fprintf('raw data dimensions: %d samples, %d coils, %d acquisitions\n', ...
    ns, nc, na)

% clone acquisition data
cloned_acq_data = acq_data.clone();

% pre-process acquisition data
fprintf('processing acquisitions...\n')
processed_data = MR.preprocess_acquisition_data(acq_data);
processed_data.sort()

% display acquistion parameters
while true
    num = input('enter acquisition number (0 to stop this loop): ');
    if num < 1 || num > na
        break
    end
    a = acq_data.acquisition(num);
    fprintf('flags: %d\n', a.flags())
    fprintf('encode step 1: %d\n', a.kspace_encode_step_1())
    fprintf('slice: %d\n', a.slice())
    fprintf('repetition: %d\n', a.repetition())
    fprintf('sample_time_us: %f\n', a.sample_time_us())
    fprintf('sample_time_us: %f\n', a.info('sample_time_us'))
    fprintf('physiology time stamp: %d %d %d\n', a.physiology_time_stamp())
    fprintf('channel_mask: ')
    fprintf('%d ', a.channel_mask())
    fprintf('\n')
    fprintf('read dir: %f %f %f\n', a.read_dir())
end

% Data returned as complex array
acq_array0 = acq_data.as_array();
is = int32(ns/3);
ic = 1;
ia = int32(na/5);
fprintf('Value of one array element: %f\n', acq_array0(is, ic, ia))

acq_array0(is, ic, ia) = acq_array0(is, ic, ia)*10;

% Data can be replaced using fill method
acq_data.fill(acq_array0);

acq_array = acq_data.as_array();
cloned_acq_array = cloned_acq_data.as_array();
processed_array = processed_data.as_array();

fprintf('Value of same array element after replacement with 10x data: %f\n', ...
    acq_array(is, ic, ia))

acq_array = permute(acq_array, [1 3 2]);
cloned_acq_array = permute(cloned_acq_array, [1 3 2]);
processed_array = permute(processed_array, [1 3 2]);
title = 'Acquisition data (magnitude)';
sirf.Utilities.show_3D_array(abs(acq_array).^0.2, title, 'samples', 'readouts', 'coil');
sirf.Utilities.set_window(0.1, 0.1, 0.8, 0.8)
title = 'Cloned acquisition data (magnitude)';
sirf.Utilities.show_3D_array(abs(cloned_acq_array).^0.2, title, ...
    'samples', 'readouts', 'coil');
sirf.Utilities.set_window(0.1, 0.1, 0.8, 0.8)
title = 'Processed acquisition data (magnitude)';
sirf.Utilities.show_3D_array(abs(processed_array).^0.2, title, ...
    'samples', 'readouts', 'coil');
sirf.Utilities.set_window(0.1, 0.1, 0.8, 0.8)

end
