/*
    Copyright 2017 Zheyong Fan and GPUMD development team
    This file is part of GPUMD.
    GPUMD is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    GPUMD is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with GPUMD.  If not, see <http://www.gnu.org/licenses/>.
*/

#pragma once
#include "ensemble_lan.cuh"
#include "force/force.cuh"
#include "langevin_utilities.cuh"
#include "model/box.cuh"
#include "utilities/common.cuh"
#include "utilities/error.cuh"
#include "utilities/read_file.cuh"
#include <map>
#include <math.h>

class Ensemble_TI_Liquid : public Ensemble_LAN
{
public:
  Ensemble_TI_Liquid(const char** params, int num_params);
  virtual ~Ensemble_TI_Liquid(void);

  virtual void compute1(
    const double time_step,
    const std::vector<Group>& group,
    Box& box,
    Atom& atoms,
    GPU_Vector<double>& thermo);

  virtual void compute3(
    const double time_step,
    const std::vector<Group>& group,
    Box& box,
    Atom& atoms,
    GPU_Vector<double>& thermo,
    Force& force);

  double fe(double x, const double coef[4], const double sum_spline[106], int index);
  void find_thermo();
  double get_UF_sum();
  void add_UF_force(Force& force);
  void init();
  void find_lambda();
  double switch_func(double t);
  double dswitch_func(double t);

protected:
  FILE* output_file;
  double lambda = 0, dlambda = 0;
  int t_equil = -1, t_switch = -1;
  double pe, eUF;
  double sigma_sqrd = 1;
  double p = 1;
  double beta;
  // Force& force;
  //  this is the actual pressure, which may cause problems due to its fluctuation
  double pressure, avg_pressure = 0, V;
  // so I use the input pressure.
  double target_pressure;
  double E_diff = 0, E_ref = 0;
  bool auto_switch = true;
  GPU_Vector<double> gpu_eUF;
  std::vector<double> thermo_cpu;
  // GPU_Vector<int> g_NN;
  // GPU_Vector<int> g_NL;

  std::vector<double> sum_spline1 = {
    0.00254749763692896, 0.00502343076464539, 0.007503155352045, 0.009991644266977,
    0.012499194237399,   0.015014204802471,   0.017538325302583, 0.020054309791579,
    0.022553886421309,   0.025061302083425,   0.027575920148606, 0.030092356559897,
    0.032614547922250,   0.035137489104047,   0.037655980393329, 0.040177975858651,
    0.042702236066866,   0.045224262320203,   0.047752893726561, 0.050283763913754,
    0.052809030243105,   0.055342989844471,   0.057876077596860, 0.060408537176755,
    0.062949101861291,   0.065485523673407,   0.068021359289858, 0.070563779924634,
    0.073107372840564,   0.075648876622086,   0.078192831974852, 0.080739796945067,
    0.083287297938079,   0.085838826257298,   0.088390782142592, 0.090941487103934,
    0.093495934822125,   0.096052395113379,   0.098608702502556, 0.101165466768052,
    0.126828600249899,   0.152636424993590,   0.178584058988326, 0.204667858034673,
    0.230881327018910,   0.257212980165412,   0.283665721426921, 0.310239904601930,
    0.336934423372076,   0.363742640738165,   0.390663732387729, 0.417695452530482,
    0.444833522871054,   0.472072401125805,   0.499411308558241, 0.526853594435617,
    0.554389479161144,   0.582018619672601,   0.609739676016697, 0.637549235823260,
    0.665442176402622,   0.693417666359418,   0.721474846192127, 0.749610732177091,
    0.777826382292536,   0.806114602359870,   0.834474078551576, 0.862908778915283,
    0.891414262332787,   0.919986121662808,   0.948625377621246, 0.977323871599565,
    1.006088946254230,   1.034922010440430,   1.063810895855170, 1.092760251235910,
    1.209142505487360,   1.326345560852390,   1.444325827752690, 1.563010170883090,
    1.682347996050180,   1.802291319952330,   1.922779851243730, 2.043789271198770,
    2.165269938086670,   2.287201466881660,   2.409560723917550, 2.532304993309890,
    2.655415729723060,   2.778874280141550,   2.902655074118200, 3.026744286866460,
    3.151124869813240,   3.275769909623850,   3.400676019195220, 3.525827463966480,
    3.651209368838240,   3.776814577889110,   3.902626259276300, 4.028639540093830,
    4.154842804432210,   4.281225969699400,   4.407785922723560, 4.534511457426090,
    4.661396868277670,   4.788436685942290};

  std::vector<double> sum_spline25 = {
    0.0147306831690499,  0.0294723009167598, 0.044314585232149,  0.059266774566239,
    0.074364727404319,   0.089577633956651,  0.104900643259878,  0.120355574417470,
    0.135929438647708,   0.151628979580904,  0.167456382248253,  0.183402201706604,
    0.199474910655862,   0.215678934679661,  0.232002621416176,  0.248455108128911,
    0.265043028697512,   0.281764420032029,  0.298620944576488,  0.315609712431588,
    0.332730399767214,   0.349987208133893,  0.367379357200308,  0.384908094731203,
    0.402573836888694,   0.420379490808446,  0.438326494610401,  0.456409759693646,
    0.474634200869181,   0.493005523005278,  0.511517421640986,  0.530171576555525,
    0.548974779234038,   0.567920184019094,  0.587013926018609,  0.606257768799528,
    0.625644095881285,   0.645176700398732,  0.664860247998297,  0.684697734522109,
    0.891538683065501,   1.114251996782550,  1.353585176796290,  1.610009060676020,
    1.883922617339220,   2.175555391811240,  2.485017120623020,  2.812273942799810,
    3.157146676566280,   3.519357379844810,  3.898540114272880,  4.294231124991230,
    4.705893658921520,   5.132981533379270,  5.574894809889770,  6.031017598746010,
    6.500724418731780,   6.983386177801780,  7.478395797930240,  7.985156732556710,
    8.503079951132140,   9.031613826197340,  9.570230921441540,  10.118409261212300,
    10.675654205937700,  11.241514550570800, 11.815556143140700, 12.397363533483300,
    12.986549704634500,  13.582742724161200, 14.185604107077500, 14.794808595489200,
    15.410047788328500,  16.031036870139700, 16.657509119233900, 17.289210433454900,
    19.863737736039800,  22.505231774871200, 25.203309318443400, 27.949712402396300,
    30.737816881147200,  33.562254425354400, 36.418610175373900, 39.303207666649200,
    42.212978261551300,  45.145337904966100, 48.098069497800400, 51.069282074654800,
    54.057336100710400,  57.060793966792300, 60.078398032231200, 63.109042396913400,
    66.151754802071800,  69.205657384509900, 72.269965351582700, 75.343983500706900,
    78.427077996985700,  81.518678937913900, 84.618271051632900, 87.725376094437000,
    90.839563272672900,  93.960437796409500, 97.087634524555600, 100.220819343418000,
    103.359685953714000, 106.503948752097000};

  std::vector<double> sum_spline50 = {
    0.0185306768522294,  0.0372792103018056,  0.056212924124813,   0.075339433470916,
    0.094670677379331,   0.114211740478830,   0.133964254004940,   0.153906375048613,
    0.174068045752584,   0.194460932793556,   0.215053388420972,   0.235855692969847,
    0.256891191126303,   0.278154324237520,   0.299640300689610,   0.321354234314584,
    0.343299150716760,   0.365487119019348,   0.387906483714816,   0.410558306409225,
    0.433455852088645,   0.456601146873966,   0.479991362154924,   0.503625542771789,
    0.527513043039352,   0.551661018804711,   0.576062742828836,   0.600718010420542,
    0.625631652957410,   0.650811826229759,   0.676265005943332,   0.701987128356421,
    0.727984517596200,   0.754259780580181,   0.780805855020279,   0.807639270480600,
    0.834762176938773,   0.862162489540871,   0.889845550786647,   0.917826657161586,
    1.214448887520250,   1.542807048274960,   1.905534876211200,   2.304602702409210,
    2.741802500363830,   3.218477172736700,   3.735466642853960,   4.293207387699940,
    4.891740153250660,   5.530683263336780,   6.209331814398670,   6.926714254220380,
    7.681639969876430,   8.472726809276860,   9.298537215682430,   10.157573569421100,
    11.048244553949000,  11.969002729676400,  12.918317752301500,  13.894674586327600,
    14.896616854691400,  15.922754315291800,  16.971757396581600,  18.042356765856200,
    19.133356088574700,  20.243632522984800,  21.372124410801500,  22.517831333884600,
    23.679815766965000,  24.857207226191900,  26.049188131103400,  27.254989159736300,
    28.473897024782100,  29.705240886102400,  30.948394672645400,  32.202775378630600,
    37.322114460371100,  42.583053241284600,  47.962431706422500,  53.442139223302300,
    59.007828689933400,  64.648002555536600,  70.353324445686900,  76.116124076291200,
    81.930033771772100,  87.789687084536400,  93.690547065163100,  99.628737382230000,
    105.600908427574000, 111.604152813349000, 117.635927885104000, 123.693996112488000,
    129.776379922909000, 135.881321741619000, 142.007241588694000, 148.152728303473000,
    154.316505259635000, 160.497420758929000, 166.694429301703000, 172.906578294339000,
    179.132998436755000, 185.372888981918000, 191.625519385956000, 197.890214082006000,
    204.166353790251000, 210.453362095267000};

  std::vector<double> sum_spline75 = {
    0.0208656914702533,  0.0421335911164029,  0.063670659694271,   0.085446488482928,
    0.107476527532398,   0.129787697603331,   0.152365197690609,   0.175229483364329,
    0.198374365040979,   0.221794637876004,   0.245514188874677,   0.269525042404852,
    0.293833546997918,   0.318437614065975,   0.343350517320355,   0.368581943444779,
    0.394111175953036,   0.419959700202330,   0.446142318747512,   0.472650775786975,
    0.499481087590020,   0.526644185038990,   0.554147918489952,   0.581989928956324,
    0.610184053390845,   0.638738048797146,   0.667643378121671,   0.696907232776590,
    0.726535198866118,   0.756532069823965,   0.786907543422482,   0.817665686268779,
    0.848809929274073,   0.880338669814449,   0.912259079648739,   0.944582450263800,
    0.977309319721285,   1.010442352711650,   1.043992062405470,   1.077960731599310,
    1.441576940402560,   1.852126706766450,   2.314029292048210,   2.831417341879480,
    3.407857461071770,   4.046104044629470,   4.748227223037830,   5.515406855441720,
    6.347982573960320,   7.245631996300860,   8.207373767176780,   9.231679089428320,
    10.316652088214900,  11.460077516053600,  12.659525239148600,  13.912460856231600,
    15.216272280735500,  16.568350436187700,  17.966118918803700,  19.407059123602500,
    20.888758962307900,  22.408905181147100,  23.965279041874100,  25.555796098410200,
    27.178494538874400,  28.831516926070500,  30.513127703302500,  32.221699368081600,
    33.955707729659200,  35.713739822333300,  37.494475566759000,  39.296671350850400,
    41.119175505438800,  42.960917109134600,  44.820894179829500,  46.698173950942900,
    54.363744255600800,  62.246000442365100,  70.308622909130700,  78.523441176252200,
    86.868326598212700,  95.325623860809300,  103.881054222375000, 112.522964293432000,
    121.241707227290000, 130.029192070618000, 138.878578664985000, 147.784032584640000,
    156.740515555896000, 165.743666573692000, 174.789680900629000, 183.875199521387000,
    192.997250555642000, 202.153194471623000, 211.340671158141000, 220.557561700525000,
    229.801953680186000, 239.072114874521000, 248.366476952209000, 257.683606218045000,
    267.022194189537000, 276.381043170246000, 285.759051399038000, 295.155206415848000,
    304.568572552008000, 313.998283381448000};

  std::vector<double> sum_spline100 = {
    0.022751120037216,   0.045910532808262,   0.069358886314654,   0.093111074332860,
    0.117193421001055,   0.141590628241591,   0.166322354449552,   0.191384838705569,
    0.216783189118122,   0.242528215537895,   0.268620650085640,   0.295067156179220,
    0.321877113204317,   0.349060002629334,   0.376608779352532,   0.404526116361092,
    0.432826329629026,   0.461516598754625,   0.490608087814006,   0.520098744854098,
    0.549998589556064,   0.580313886093840,   0.611042422021131,   0.642198386422815,
    0.673785350112000,   0.705810878229726,   0.738282703229342,   0.771207048892062,
    0.804591961793084,   0.838436905388695,   0.872758381474059,   0.907561036175287,
    0.942846603767524,   0.978625367185550,   1.014903295330200,   1.051686407918390,
    1.088980922859230,   1.126797655939330,   1.165141004877230,   1.204018512792680,
    1.623677596465690,   2.104370910113270,   2.653008220727520,   3.275979576925120,
    3.978923839960030,   4.766450436575240,   5.641977302440640,   6.607574516351870,
    7.664068081251860,   8.811188730960130,   10.047707473241200,  11.371559331958000,
    12.780062993792400,  14.270100584286200,  15.838239772969900,  17.480824496419700,
    19.194110106141100,  20.974404874909700,  22.818033169025700,  24.721404291025700,
    26.681081620554200,  28.693777743176400,  30.756365300567500,  32.865885098080900,
    35.019567569746600,  37.214805116699300,  39.449156150761500,  41.720343675582300,
    44.026242064232500,  46.364873708591800,  48.734393689770700,  51.133080789475600,
    53.559334176702700,  56.011672897612300,  58.488711702450300,  60.989157573392400,
    71.202211031757100,  81.707105874834700,  92.454214656498000,  103.405240329003000,
    114.530211680502000, 125.805339025108000, 137.211452823616000, 148.732919115523000,
    160.356818930707000, 172.072353003607000, 183.870405493372000, 195.743198822087000,
    207.684039226795000, 219.687114803139000, 231.747350331965000, 243.860281653511000,
    256.021956470225000, 268.228858659096000, 280.477840724457000, 292.766082311122000,
    305.091036447450000, 317.450392248597000, 329.842058607862000, 342.264125899782000,
    354.714844033488000, 367.192616055157000, 379.695974154135000, 392.223562929583000,
    404.774131143285000, 417.346524791109000};

  std::vector<std::vector<double>> spline1 = {
    {-14.175243817292000, 1.036718109543200, 1.000000000000000, 0.000000000000000},
    {20.125731451941000, 0.779460795023800, 1.000643143286300, -0.000000535952739},
    {-21.719681990149000, 1.407141996654600, 0.997504737278147, 0.000004694723942},
    {43.540196508616000, -0.061205269567100, 1.008517341774810, -0.000022836787300},
    {-71.922704044525000, 3.402681747027300, 0.973878471608861, 0.000092626113253},
    {103.152219669624000, -3.162627892253700, 1.055944842099880, -0.000249317097126},
    {-173.217374634049000, 9.274003851411600, 0.869395365944898, 0.000683430283649},
    {165.589278866989000, -8.513345457392800, 1.180673978848970, -0.001132361624958},
    {-3.827740834339000, 1.651675724686500, 0.977373555207398, 0.000222974532652},
    {-99.974315529110000, 8.141569516583200, 0.831350944889726, 0.001318144110035},
    {66.317002951237000, -4.330279369442800, 1.143147167040380, -0.001280157741221},
    {15.909503723606000, -0.171660683163100, 1.028785153167680, -0.000231839280721},
    {-106.588617845915000, 10.853170258094300, 0.698040224929945, 0.003075610001657},
    {73.030567659913000, -6.659700328723900, 1.267208519001530, -0.003090379850786},
    {115.311147206967000, -11.099161181164300, 1.422589648836940, -0.004903159698865},
    {-243.593556489187000, 29.277617984653100, -0.091539569881214, 0.014023455535112},
    {161.930278750475000, -19.385242244106900, 1.854974839269210, -0.011930069920228},
    {134.477241488185000, -15.884979993165400, 1.706213693604210, -0.009822620356640},
    {-412.088844704072000, 57.901441642789500, -1.614175280013770, 0.039983214247630},
    {255.759737328844000, -37.266981296900800, 2.906324809621510, -0.031591370504929},
    {428.045095388187000, -63.109785005802200, 4.198464995066570, -0.053127040262347},
    {-1196.375318882160000, 192.736430241777000, -9.233461305431350, 0.181931669996367},
    {1525.206580141470000, -256.324583097121000, 15.464894428208100, -0.270871518453689},
    {-1164.169401683850000, 207.592773767747000, -11.210353591521800, 0.240404068591134},
    {283.490226592775000, -52.985959322046600, 4.424370393865790, -0.072290411116619},
    {378.541295313965000, -70.808034707269600, 5.538250105442220, -0.095496238441127},
    {-180.369007848853000, 38.179474409480500, -1.545937987146590, 0.057994503564964},
    {-426.607663919398000, 88.042802263766000, -4.911712617310870, 0.133724432743661},
    {513.250863527337000, -109.327488500049000, 8.904207736156160, -0.188647042170570},
    {-76.462990190578000, 18.935274683598300, -0.394842594658249, 0.036080007490778},
    {-146.336502766146000, 34.656815013100800, -1.573958119370910, 0.065557895608594},
    {-74.568598742743000, 17.970777327659100, -0.280790198749144, 0.032151057659198},
    {553.455697736353000, -132.755053827324000, 11.777276293649600, -0.289397382138102},
    {-1116.130992202070000, 280.467651932436000, -22.313596931530600, 0.648101631554353},
    {1235.285871074090000, -319.143648202987000, 28.653363579980300, -0.795962249605125},
    {-652.692492097983000, 176.450672129683000, -14.711139449128300, 0.468835755410543},
    {65.116097320299000, -17.357647013253100, 2.731609273735960, -0.054446706275385},
    {-94.523897183549000, 26.942451461564600, -1.366149835184660, 0.071900866249667},
    {332.019491414570000, -94.622414288899100, 10.182512411109400, -0.293806771549644},
    {-203.602068476343000, 62.046891979191300, -5.092744950029300, 0.202639092687359},
    {2.113572118540000, 0.332199800726900, 1.078724267817120, -0.003076547907521},
    {-0.816135495800000, 1.430840156104200, 0.941394223394965, 0.002645537276735},
    {1.175353864660000, 0.534669943897200, 1.075819755226020, -0.004075739314817},
    {-0.961727962842000, 1.656637903335900, 0.879475362324243, 0.007377683604453},
    {0.892773986718000, 0.543936733600000, 1.102015596271440, -0.007458331992027},
    {0.148232015969000, 1.046502563855400, 0.988938284463978, 0.001022466393532},
    {0.068185949408000, 1.106537113776000, 0.973929646983833, 0.002273186183543},
    {0.283568186410000, 0.928846768249400, 1.022794492003620, -0.002206091276604},
    {-0.654778695062000, 1.773358961574000, 0.769440834006258, 0.023129274523131},
    {1.603674593854000, -0.428632995119300, 1.485088219931620, -0.054399192285451},
    {-1.567247680359000, 2.900835392804300, 0.319774284158366, 0.081554100221428},
    {1.561508127579000, -0.619014891125700, 1.639718140632100, -0.083438881837791},
    {-1.790976829943000, 3.403967057901100, 0.030525361021391, 0.131120155443640},
    {2.778271192180000, -2.421824170306300, 2.506486633009560, -0.219641024754687},
    {-1.261563938766000, 3.031953256471200, 0.052286790959677, 0.148488951552796},
    {-2.060175437128000, 4.169974641637500, -0.488273366994286, 0.234077643228837},
    {4.887065687294000, -6.250887044995600, 4.722157476322240, -0.634327497323919},
    {-5.217911312043000, 9.664451728959600, -3.633395380004210, 0.827894252533207},
    {4.915491560891000, -7.055663011381600, 5.562667727183480, -0.858050650451206},
    {-5.752118931552000, 11.345965088083100, -5.018268430008800, 1.169962113010660},
    {7.555352165319000, -12.607482886285400, 9.353800354612240, -1.704451643913550},
    {-7.517321729709000, 15.653780666892600, -8.309489366124000, 1.975400381239840},
    {6.547374753507000, -11.772377475378000, 9.517513426351890, -1.887116890463270},
    {-3.955697284297000, 9.496343401175000, -4.838873165321420, 1.343070092663240},
    {-0.769705616371000, 2.805760898529000, -0.155465413469181, 0.250274950564380},
    {3.896279749827000, -7.342757272951300, 7.202210260853950, -1.527830004063700},
    {-0.242933382970000, 1.970472275841300, 0.217288099259571, 0.218400536334885},
    {-5.062786217923000, 13.176630117107100, -8.467484227721360, 2.461966720804950},
    {4.943998254687000, -10.839652617156600, 10.745541959689600, -2.661506929171310},
    {-0.255606800897000, 2.029369895414700, 0.128598386818230, 0.258152553368322},
    {-8.626851051063000, 23.376042733339000, -18.016073525417500, 5.399142928501780},
    {22.443932605168000, -58.184764364268600, 53.349632684989300, -15.415854716200200},
    {-26.812124169636000, 74.806588927702800, -66.342585277785000, 20.491810672632100},
    {13.358688873416000, -36.667417266766800, 36.770870452099400, -11.301504844082300},
    {3.492607075913000, -8.549084143882800, 10.058453985359600, -2.842572962948000},
    {-6.549597177037000, 20.824363295996700, -18.580657268523300, 6.465138194564030},
    {0.432583115381000, -0.122177581260100, 2.365883608733510, -0.517042097854896},
    {0.220924125406000, 0.576297085658400, 1.597561475123160, -0.235323982197766},
    {-0.468509617007000, 3.058258558344300, -1.380792292099940, 0.956017524691461},
    {0.638544342622000, -1.259251884207200, 4.231971283217070, -1.476180024612580},
    {-0.640927753484000, 4.114530919439200, -3.291324641887960, 2.034691407103130},
    {0.454258871316000, -0.813808892160700, 4.101185075511870, -1.661563451596760},
    {-0.061983831777000, 1.664156082686600, 0.136441115756184, 0.452966660272947},
    {-0.339934344208000, 3.081703696082300, -2.273389827016550, 1.818537527844160},
    {0.738674808610000, -2.742785729134000, 8.210691138372710, -4.471911051389360},
    {-0.785182890232000, 5.943203154263300, -8.292687740081940, 5.980228904965210},
    {0.389997452319000, -1.107878901041400, 5.809476370527360, -3.421213835441010},
    {0.017247580954000, 1.240445288558900, 0.877995572366696, 0.030822723271454},
    {-0.051576076128000, 1.694681425299000, -0.121323928461336, 0.763657023878633},
    {-0.157272776449000, 2.423988657515100, -1.798730562558410, 2.049668776686320},
    {0.427151411728000, -1.783865497360200, 8.300119409142490, -6.029411200674480},
    {-0.521808059859000, 5.333330539542100, -9.492870683113070, 8.798080542871740},
    {0.128343117105000, 0.262151359225600, 3.692195185709710, -2.628976543441290},
    {0.614295961645000, -3.674066681554900, 14.319983895817000, -12.193986382537800},
    {-1.090873763694000, 10.649359011295300, -25.785608044163700, 25.237899428110800},
    {1.104581293134000, -8.451099983109700, 29.605723039610800, -28.307053952871100},
    {-0.814237208846000, 8.818266534713700, -22.202376513859700, 23.501045600599600},
    {0.329128042249000, -1.815030300471800, 10.760843675215600, -10.560948594778200},
    {0.213262439857000, -0.702720517510500, 7.201452369739290, -6.764264535603490},
    {-0.586912801689000, 7.219014373794700, -18.940272771567700, 21.991633119834000},
    {0.622161266901000, -5.113541125822200, 22.990415927130100, -25.529814072023500},
    {-0.250032265914000, 4.044490968739600, -9.062696403836790, 11.865483647438300},
    {-0.258332203250000, 4.134130291970400, -9.385397967467940, 12.252725523796100},
    {0.573511078920000, -5.099330140120000, 24.778405631266400, -29.882632247976400},
    {-0.509162112434000, 7.243144241310800, -22.122997018170300, 29.525811107976500},
    {-0.505012629187000, 7.194595287325500, -21.933656097627700, 29.279667911271300}};

  std::vector<std::vector<double>> spline25 = {
    {-1.498614095965000, 5.894146535239900, 1.000000000000000, 0.000000000000000},
    {39.388642287790000, 5.587492112362000, 1.000766636057190, -0.000000638863381},
    {-10.955155055115000, 6.342649072505100, 0.996990851256481, 0.000005654111287},
    {103.132777932515000, 3.775670580284100, 1.016243189948130, -0.000042476735442},
    {-144.590356675209000, 11.207364618515800, 0.941926249565816, 0.000205246399165},
    {204.751848768723000, -1.892968085631600, 1.105680408367660, -0.000477062595842},
    {-86.749838400593000, 11.224607836987700, 0.908916769528367, 0.000506755598354},
    {14.273104834572000, 5.920903317141600, 1.001731598625670, -0.000034664238047},
    {111.974219061821000, 0.058836463506600, 1.118972935698380, -0.000816273151865},
    {-83.851581082186000, 13.277077973227000, 0.821562501729670, 0.001414305102901},
    {24.324905267648000, 5.163841496989700, 1.024393413635590, -0.000275952496315},
    {164.892760011242000, -6.433006519356500, 1.343306734085110, -0.003199324600436},
    {-131.263945313620000, 20.221096959880900, 0.543683629707989, 0.004796906443335},
    {-45.209778755577000, 11.830815720472000, 0.816367769988769, 0.001842828256960},
    {368.103060335160000, -31.567032384054800, 2.335292453647180, -0.015877959719054},
    {-342.695262585834000, 48.397778944557000, -0.663387971175752, 0.021605545591232},
    {258.223590008626000, -23.712483366778400, 2.221022521277670, -0.016853260974813},
    {-128.877497447420000, 25.642905283867800, 0.123418503625190, 0.012862795941930},
    {61.523199778577000, -0.061188841641400, 1.280102739273090, -0.004487467592788},
    {25.856698333797000, 5.021287614240100, 1.038685107618700, -0.000665021758260},
    {158.774806886371000, -14.916428668646300, 2.035570921763030, -0.017279785327332},
    {-269.099925879601000, 52.473841741994300, -1.502418274795600, 0.044635025612444},
    {402.700096631676000, -58.373161972365800, 4.594166929494180, -0.067135703132868},
    {-406.574060645612000, 81.226630157966200, -3.432821117999910, 0.086714901110768},
    {571.501745948807000, -94.827015029029300, 7.130397593219830, -0.124549473113627},
    {-550.973723148899000, 115.637135426790000, -6.023611810268880, 0.149492389459054},
    {254.642746646141000, -41.458076183241600, 4.187576944383140, -0.071750033558405},
    {279.420336565294000, -46.475538141870700, 4.526255626590650, -0.079370303908075},
    {-120.250492905286000, 37.455336046951200, -1.348905566626900, 0.057716790600335},
    {-543.829564948481000, 129.583784216346000, -8.028218058907980, 0.219133509163794},
    {1063.584752701840000, -232.084437254977000, 19.096898551441200, -0.458994406094936},
    {-608.061445859817000, 156.573303910609000, -11.024076388891700, 0.319130779863664},
    {-382.682969261622000, 102.482469527042000, -6.696809638206400, 0.203736999845391},
    {1474.793322904870000, -357.242912784165000, 31.230534402468200, -0.839264961273161},
    {-1739.690322354920000, 462.450416757082000, -38.443398608537800, 1.134829807372010},
    {958.431966511453000, -245.806684070341000, 23.529097713861800, -0.672701335364648},
    {84.122456309075000, -9.743116315698500, 2.283376615943900, -0.035329702427108},
    {5.814208254975000, 11.987422519313800, 0.273301773705291, 0.026647605208581},
    {-41.155289331414000, 25.373729331434800, -0.998397373446217, 0.066918078201713},
    {-107.417050928546000, 44.755294598596800, -2.888099986994580, 0.128333413142037},
    {40.477779409261000, 0.386845497254800, 1.548744923139590, -0.019561417195768},
    {26.232693351055000, 5.728752769081900, 0.881006514161216, 0.008261016511663},
    {31.944663186516000, 3.158366343124500, 1.266564478054820, -0.011016881683016},
    {31.397101902993000, 3.445836016974100, 1.216257285131120, -0.008082295429132},
    {26.088241201117000, 6.631152438100300, 0.579194000905811, 0.034388590185893},
    {30.725293292966000, 3.501142276102300, 1.283446287355370, -0.018430331297826},
    {19.578009626707000, 11.861605025796200, -0.806669400068083, 0.155745975987461},
    {23.847884200569000, 8.338958502360000, 0.162058393876862, 0.066945928209175},
    {16.163861570823000, 15.254578869131400, -1.912627716154420, 0.274414539212290},
    {18.229469515910000, 13.240611122670700, -1.258088198554520, 0.203506091472284},
    {6.756340365689000, 25.287396730402200, -5.474463161260360, 0.695416503787941},
    {12.461649021839000, 18.868924492233700, -3.067536071947060, 0.394550617623765},
    {11.292423546206000, 20.271995062992700, -3.628764300250630, 0.469381048064239},
    {-3.017903205614000, 38.517661671562800, -11.383172608892800, 1.567922225121860},
    {13.395189275718000, 16.359986821764800, -1.412218926483840, 0.072279172760537},
    {-7.750373898443000, 46.492414344944200, -15.725121999993800, 2.338488826066230},
    {9.838626320675000, 20.108914016267400, -2.533371835655470, 0.139863798676534},
    {-3.965411385695000, 41.850273403801100, -13.947585514111000, 2.137351192406300},
    {3.806379219907000, 29.026818904558200, -6.894685539527850, 0.844319530399478},
    {-9.006665491393000, 51.129321031550400, -19.603624262548800, 3.280199452311900},
    {13.944122746579000, 9.817902203201400, 5.183227034460220, -1.677170807089820},
    {-14.904545496256000, 63.909155158517800, -28.623806062612400, 5.365961088133610},
    {4.121099235135000, 26.809147932305300, -4.508801365574600, 0.141043403775504},
    {-6.945931438509000, 49.219885046434200, -19.636048917611700, 3.544674102983880},
    {13.896546516003000, 5.450681341960500, 11.002393675520000, -3.604295835413540},
    {-21.225854623935000, 81.841903821324900, -44.381242622018600, 9.780082936491470},
    {15.365271976702000, -0.488131030109000, 17.366283516555900, -5.656798598151950},
    {-14.443233281008000, 68.816643694069300, -36.344916894682900, 8.218594841418240},
    {5.383661147357000, 21.232097065994000, 1.722720407775870, -1.932775105903700},
    {1.583788689368000, 30.636781399516900, -6.036144167379920, 0.200912652263981},
    {-9.837215899097000, 59.760343100102100, -30.791171612877600, 7.214837095155050},
    {5.544274897443000, 19.383929759185600, 4.538190060424110, -3.089560059557870},
    {-3.287083684040000, 43.228597929190400, -16.922011292580700, 3.348500346343740},
    {2.336859842262000, 27.622154643703500, -2.486051253507320, -1.102587332369960},
    {-7.180355696414000, 54.746218928929600, -28.253912324470900, 7.057235340101460},
    {3.514162960710000, 23.464751856844000, 2.245518070811220, -2.855079538364780},
    {-0.815274052562000, 36.453062896659200, -10.742792969005400, 1.474357474907830},
    {-0.419894516671000, 35.148310428219000, -9.307565253721340, 0.948107312637104},
    {-0.247097880896000, 34.526242539429500, -8.561083787173960, 0.649514726018123},
    {-0.098413959299000, 33.946375245199300, -7.807256304672620, 0.322856150266714},
    {-0.030246282398000, 33.660071002216900, -7.406430364497210, 0.135804044851477},
    {-0.160050910966000, 34.244191830775000, -8.282611607335380, 0.573894666271086},
    {-0.079150073458000, 33.855867810734000, -7.661293175270250, 0.242524835836476},
    {0.187501204070000, 32.495946295342100, -5.349426599103610, -1.067532890658150},
    {-0.023454741674000, 33.635108402358800, -7.399918391734230, 0.162762184920613},
    {-0.219542238327000, 34.752807133278900, -9.523545980480630, 1.507726324458940},
    {0.529403595294000, 30.259132131553200, -0.536195977029207, -4.483840344508670},
    {-0.559111842881000, 37.116779392058500, -14.937255224091700, 5.596901128435870},
    {0.406083476344000, 30.746490285175500, -0.922619188948725, -4.680498630669230},
    {-0.320251962654000, 35.758204814259500, -12.449562605841500, 4.156824655615030},
    {0.386934374359000, 30.666463187766300, -0.229382702257264, -5.619319267252890},
    {-0.115585534820000, 34.435362506608200, -9.651630999360630, 2.232554313665470},
    {-0.230292235005000, 35.330074768053800, -11.977882879120600, 4.248639276125300},
    {0.253354474853000, 31.412536418201700, -1.400529334524110, -5.270978914007720},
    {0.112674335435000, 32.594249589312700, -4.709326213629090, -2.182768493515300},
    {-0.394551816441000, 37.007117110635700, -17.506642025464900, 10.187970124592000},
    {0.567832930255000, 28.345654390363400, 8.477746135352890, -15.796418036226800},
    {-0.440179904490000, 37.720173753495700, -20.583263890358300, 14.233292323675800},
    {0.020886687747000, 33.293934468018900, -6.419298176830060, -0.874937770756713},
    {0.299633153361000, 30.534344458442000, 2.687348854776830, -10.892249505527800},
    {-0.322419301079000, 36.879279493733400, -18.885430265214000, 13.556900163795100},
    {0.185044050897000, 31.550914297975500, -0.236152080058281, -8.200591052223380},
    {-0.076256902463000, 34.372964594268700, -10.395533146716000, 3.990666227768300},
    {0.194983559112000, 31.362195470785100, 0.744312610179325, -9.748476872409580},
    {-0.290677334272000, 36.898729655357500, -20.294517291197500, 16.900707669336300},
    {-0.284274221883000, 36.823813240413500, -20.002343272917200, 16.520881445573400}};

  std::vector<std::vector<double>> spline50 = {
    {36.084207286535000, 7.367165481783600, 1.000000000000000, 0.000000000000000},
    {20.636978140445000, 7.483019700379300, 0.999710364453511, 0.000000241362955},
    {45.732680151716000, 7.106584170210300, 1.001592542104360, -0.000002895599796},
    {27.549101252475000, 7.515714695443200, 0.998524063165109, 0.000004775597552},
    {41.837314838631000, 7.087068287858200, 1.002810527240960, -0.000009512616034},
    {34.317639393322000, 7.369056117057500, 0.999285679375968, 0.000005174250070},
    {-67.491872412759000, 11.950484148331100, 0.930564258906866, 0.000348781352416},
    {327.112250258779000, -8.766232291924600, 1.293106796611340, -0.001766050117527},
    {-277.174728623209000, 27.490986440994700, 0.567962421952950, 0.003068245713529},
    {74.405864234279000, 3.759296423114300, 1.101925447355260, -0.000936476976989},
    {236.120871686083000, -8.369329135770800, 1.405141086327380, -0.003463273968423},
    {-2.070150978639000, 11.281430234068400, 0.864745203656813, 0.001490354956057},
    {-170.777867771718000, 26.465124745446000, 0.409234368315474, 0.006045463309471},
    {245.181622065757000, -14.090925513707500, 1.727306001737950, -0.008233646052606},
    {-47.478220492252000, 16.638357954884500, 0.651781080337197, 0.004314144697070},
    {77.787259902351000, 2.545991410491800, 1.180244825751920, -0.002291652120614},
    {245.538780883407000, -17.584191107235400, 1.985452126461020, -0.013027749463402},
    {-666.867183434479000, 98.747569343294600, -2.958647692686490, 0.057013664641187},
    {1004.707552853000000, -126.915020055516000, 7.196168830260010, -0.095308583203011},
    {-612.705427977004000, 103.566329712760000, -3.751695283733110, 0.078032598601880},
    {449.454959055801000, -55.757728342160500, 4.214507619012920, -0.054737449777220},
    {-495.623208247195000, 93.092083008061500, -3.600107476873740, 0.082018314400797},
    {600.314673933155000, -87.737667551696500, 6.345528803912970, -0.100318350746960},
    {-144.899487483770000, 40.811775292722800, -1.046064159641140, 0.041353847721160},
    {242.438475999762000, -28.909058134312600, 3.137185845980970, -0.042311152391281},
    {-750.806416514534000, 157.324359212118000, -8.502402738170980, 0.200180276445218},
    {1133.235190058030000, -210.063754069532000, 15.377824625136300, -0.317224649759774},
    {-857.596743716867000, 193.079712519884000, -11.834359369649200, 0.295049490122899},
    {1166.841384812340000, -232.052294471250000, 17.924881119730200, -0.399332787962620},
    {-1061.403995536660000, 252.591075754658000, -17.211763221648100, 0.449802783620689},
    {488.745797335354000, -96.192627641545400, 8.947014533067090, -0.204166660247191},
    {334.715206195994000, -60.380515201644600, 6.171575818974800, -0.132467826799808},
    {-427.965022119849000, 122.662739594157000, -8.471884564689260, 0.258024450097899},
    {-365.722317713163000, 107.257670253502000, -7.200966344085110, 0.223074199031283},
    {2473.785492972120000, -616.816821471245000, 54.345365452518400, -1.520738535205810},
    {-3493.861254179220000, 949.690449655982000, -82.724020771114200, 2.477118562983470},
    {2327.176323744730000, -621.989696383485000, 58.727192372437900, -1.766417831323090},
    {-734.485640796406000, 227.621498776680000, -19.861843179877300, 0.656744098206622},
    {1750.753439440010000, -480.671639090701000, 47.426004917524000, -1.474037758211090},
    {-1624.675316962510000, 506.641272157037000, -48.837003929130500, 1.654510029305190},
    {105.387443025830000, -12.377555839463900, 3.064878870519590, -0.075552730683151},
    {60.881221008023000, 4.312277417213900, 0.978649713434863, 0.011373484195380},
    {75.711666542077000, -2.361423073110200, 1.979704786983460, -0.038679269482050},
    {80.249981623655000, -4.744038490938600, 2.396662485103460, -0.063001801872384},
    {65.687702963322000, 3.993328705261300, 0.649189045863380, 0.053496427410294},
    {69.465510523003000, 1.443308602477200, 1.222943568989750, 0.010464838175820},
    {58.391470944673000, 9.748838286223900, -0.853438851946789, 0.183496706587187},
    {55.091165698389000, 12.471590114408700, -1.602195604697650, 0.252132742256021},
    {34.802586261662000, 30.731311607462300, -7.080112052613710, 0.799924387047623},
    {42.916089255186000, 22.820646188774800, -4.509145791539820, 0.521403042097903},
    {18.056256717443000, 48.923470353404500, -13.645134249160000, 1.587268362153570},
    {28.075843875149000, 37.651434800985900, -9.418120917003130, 1.058891695633970},
    {-2.660432217863000, 74.534966112600000, -24.171533441648800, 3.026013365586730},
    {35.425084996041000, 25.975931664872000, -3.533943801363970, 0.102354833212981},
    {-18.880867766006000, 99.288967893635800, -36.524810104307800, 5.050984778654570},
    {9.354066067840000, 59.054187180403500, -17.413289265521800, 2.024993979180020},
    {4.288603494706000, 66.652381040103900, -21.212386195371700, 2.658176800821630},
    {-8.131520046595000, 86.214075617653200, -31.482275848585100, 4.455407490133960},
    {-5.068123307797000, 81.159470998634600, -28.702243308124300, 3.945734857716040},
    {-0.258386722999000, 72.862675389857800, -23.931585833077400, 3.031358841665350},
    {-6.551129799846000, 84.189612928181700, -30.727748356071600, 4.390591346264160},
    {-4.637894077587000, 80.602295948946100, -28.485675244049200, 3.923492781259470},
    {-9.162893889561000, 89.426045582295000, -34.221112505725200, 5.166170854622450},
    {0.639869636279000, 69.575449442465600, -20.821960111338700, 2.151361565885110},
    {-7.918184656455000, 87.547363457209600, -33.402299921660500, 5.086774188293760},
    {-4.787931010151000, 80.739061776500200, -28.466281203148200, 3.893902997987120},
    {-6.491691303063000, 84.572522435554400, -31.341376697440100, 4.612676871560430},
    {-2.534903777723000, 75.372991439140800, -24.211740175220500, 2.770854103320580},
    {-0.136693585637000, 69.617286978133500, -19.607176606414900, 1.542970484972490},
    {-9.420721880542000, 92.595257008020900, -38.564001881069000, 6.756097435501510},
    {-1.310018891581000, 71.912964386169100, -20.984053152493500, 1.775111962404720},
    {0.814397446720000, 66.336371498128900, -16.104534375459900, 0.351918985770362},
    {-10.504370894932000, 96.897046020591700, -43.609141445676800, 8.603301106835610},
    {6.655886133467000, 49.277332766783500, 0.439093314095444, -4.978237944094090},
    {-8.490373640598000, 92.444173122868700, -40.569405024186300, 8.007786529695390},
    {-0.057591570203000, 67.778285566965300, -16.520164657181500, 0.191783410419190},
    {-1.324435177597000, 71.578816389146100, -20.320695479362100, 1.458627017812530},
    {-0.867291239115000, 70.070241392157100, -18.661262982675300, 0.850168435694543},
    {-0.311849866023000, 68.070652449025500, -16.261756250917000, -0.109634257009005},
    {-0.341109296710000, 68.184764228704000, -16.410101564497700, -0.045351287791280},
    {0.217387052817000, 65.839079560691100, -13.126143029278200, -1.577865270894380},
    {-0.213538914505000, 67.778246413640700, -16.034893308703000, -0.123490131181684},
    {0.374768605201000, 64.954370319053500, -11.516691557367400, -2.533197731892040},
    {-0.060035506419000, 67.171871288316200, -15.286443205113300, -0.397005131503130},
    {-0.031626579414000, 67.018463082489300, -15.010308434628100, -0.562685993792228},
    {0.466541823960000, 64.178903183258700, -9.615144626089150, -3.979623072534070},
    {-0.196540716365000, 68.157398425209300, -17.572135109992700, 1.325037250069840},
    {0.237621041424000, 65.422179351138500, -11.828175054447600, -2.695734788809250},
    {-0.020443449239000, 67.125404989514800, -15.575271458877300, 0.052135907773802},
    {0.153652755341000, 65.924141177912900, -12.812364692194900, -2.066092613347810},
    {-0.047667572096000, 67.373647535462100, -16.291179950314300, 0.716959593148794},
    {0.209517533108000, 65.444759246433300, -11.468959227743600, -3.301557675659060},
    {-0.052902560446000, 67.491635976157800, -16.790838725030500, 1.310737888658800},
    {-0.202907291123000, 68.706674294636800, -20.071442184925900, 4.263281002566720},
    {0.544031724667000, 62.432386562002700, -2.503436533541810, -12.133524272066500},
    {-0.723719607299000, 73.461823150103000, -34.488802639024500, 18.785662963225500},
    {0.906346704380000, 58.791226344989100, 9.522987776313800, -25.226127452109500},
    {-0.903667210059000, 75.624355749276800, -42.659713376981700, 28.695997072966400},
    {0.908822135943000, 58.224458027653600, 13.019959332206300, -30.695653816827300},
    {-0.839621334205000, 75.534048382124200, -44.101688837541900, 32.138159169890300},
    {0.598163201221000, 60.868646120782100, 5.760678851007830, -24.372524210450800},
    {-0.309031470795000, 70.394190176946200, -27.578725345570700, 14.523447352228600},
    {0.092462681846000, 66.058053328432000, -11.968632690937700, -4.208663833309130},
    {0.181680743685000, 65.067732842017200, -8.304446891210490, -8.727826319629920},
    {-0.368185657248000, 71.336209812660000, -32.124659379645400, 21.444442832377600},
    {-0.338438114227000, 70.988163559304600, -30.767278991548300, 19.679848327837300}};

  std::vector<std::vector<double>> spline75 = {
    {68.998569518968000, 8.260028376202600, 1.000000000000000, 0.000000000000000},
    {10.524611443188000, 8.698583061770700, 0.998903613286080, 0.000000913655595},
    {59.148104708555000, 7.969230662789700, 1.002550375280990, -0.000005164281063},
    {34.644569722673000, 8.520560199972400, 0.998415403752115, 0.000005173147759},
    {99.134096400775000, 6.585874399629000, 1.017762261755550, -0.000059316378919},
    {-90.042715325680000, 13.680004839371500, 0.929085631258765, 0.000310169581484},
    {293.259164901158000, -3.568579770836200, 1.187814400411880, -0.000983474264281},
    {-293.533144278442000, 27.238016461093400, 0.648698966353102, 0.002161365767728},
    {375.006212212199000, -12.874344928345000, 1.450946194141870, -0.003186949084197},
    {-89.528504571206000, 18.481748454534700, 0.745434093027077, 0.002104391674164},
    {-24.108193925869000, 13.575225156134200, 0.868097175487100, 0.001082199320330},
    {262.810880273819000, -10.095598465340200, 1.519044825077650, -0.004884820800916},
    {-279.130527168760000, 38.679128204491800, 0.055803024982691, 0.009747597200033},
    {587.975228400665000, -45.863682963527300, 2.803444387943320, -0.020018517565374},
    {-454.850386432499000, 63.633006593954900, -1.028939746568560, 0.024692630670598},
    {12.658317328160000, 11.038277420880400, 0.943362597421746, 0.000038851370719},
    {849.965917120058000, -89.438634554147100, 4.962439076422840, -0.053548835015962},
    {-805.685185807622000, 121.656881069132000, -4.009120337566560, 0.073548256682222},
    {467.605226109526000, -50.237324539682200, 3.726118914830090, -0.042480332103728},
    {-309.921318631864000, 60.560208085965900, -1.536763884888190, 0.040848645558478},
    {829.260848420578000, -110.317116971900000, 7.007102368005120, -0.101549125323077},
    {-755.742875051765000, 139.320969474993000, -6.098897170456790, 0.127805866600006},
    {520.782651785367000, -71.305742453133900, 5.485571985590240, -0.084576067927523},
    {233.057067913195000, -21.673079235183700, 2.631693850558070, -0.029876737006073},
    {-244.646123440901000, 64.313495208554200, -2.527500616066240, 0.073307152326414},
    {-359.918974149593000, 85.927154716435000, -3.878354335308860, 0.101449938143970},
    {1240.805220038220000, -226.214063150189000, 16.410824826021700, -0.338148943684860},
    {-1228.709906002330000, 273.862749873024000, -17.344360053045200, 0.421342716094146},
    {1351.218403974040000, -267.922195222015000, 20.580586103607600, -0.463572694227750},
    {-873.619709899741000, 215.980094545533000, -14.502329904539700, 0.384264442635810},
    {714.652435626134000, -141.381138197789000, 12.299762551209400, -0.285787868757919},
    {-526.782032604273000, 147.252375665781000, -10.069334773217300, 0.292080478789771},
    {265.339694793880000, -42.856838909776300, 5.139402392827380, -0.113485845638088},
    {502.303253424684000, -101.505319670901000, 9.977902055620150, -0.246544586364890},
    {62.855291513274000, 10.553910616508900, 0.452867481190354, 0.023331393243954},
    {-767.484419483678000, 228.518084753208000, -18.618997755770700, 0.579594129321982},
    {1106.922386423290000, -277.571752841674000, 26.929087627768600, -0.786848432184197},
    {-148.733126206366000, 70.872651913055700, -5.302019812043820, 0.206944047210019},
    {-461.877881600400000, 160.118907200354000, -13.780414064337000, 0.475426531865966},
    {500.644652551613000, -121.418934039123000, 13.669525456513200, -0.416696502561707},
    {103.464226566351000, -2.264806243544400, 1.754112676955400, -0.019516076576448},
    {112.501817059314000, -5.653902678405600, 2.177749731313050, -0.037167620508018},
    {129.733577196538000, -13.408194740156700, 3.340893540575760, -0.095324810971155},
    {119.293570154258000, -7.927191042959500, 2.381717893566230, -0.039372898228931},
    {121.129486186607000, -9.028740662368900, 2.602027817448160, -0.054060226487729},
    {118.099525099410000, -6.983516928510900, 2.141852477330020, -0.019547075978861},
    {88.880413416211000, 14.930816833888100, -3.336730963269730, 0.437001544071115},
    {90.182981234580000, 13.856198383734200, -3.041210889477400, 0.409912203973487},
    {69.207821646432000, 32.733842013067700, -8.704503978277700, 0.976241512853539},
    {50.827652178691000, 50.654507244115600, -14.528720178368400, 1.607198267863380},
    {38.894049640771000, 63.184789908931700, -18.914319111054000, 2.118851476676700},
    {36.116149255827000, 66.309927841994600, -20.086245835952700, 2.265342317289050},
    {7.433993336213000, 100.728514945532000, -33.853680677367900, 4.101000296144440},
    {19.366437400527000, 85.514648763530900, -27.387787550017100, 3.184998769769690},
    {-0.113182939185000, 111.812136222142000, -39.221656906391900, 4.960079173225920},
    {-1.819305640447000, 114.243361071437000, -40.376488709805600, 5.142927542099520},
    {-0.974714502854000, 112.976474365046000, -39.743045356609200, 5.037353649899980},
    {-11.270636354584000, 129.192551281521000, -48.256485737758900, 6.527205716601220},
    {-7.971540059134000, 123.749042394029000, -45.262555849639300, 5.978318570446060},
    {-3.665603428180000, 116.321301705631000, -40.991604953808400, 5.159719648744780},
    {-17.023646221833000, 140.365778734201000, -55.418291170948000, 8.045056892172220},
    {-11.475011676202000, 129.962088961146000, -48.915985062789000, 6.690409786305930},
    {-1.204307083162000, 109.934215004717000, -35.897866991109900, 3.869817537442050},
    {-16.258159994368000, 140.418267149908000, -56.474602189114000, 8.499582956992940},
    {-9.807852919812000, 126.872622293338000, -46.992650789513600, 6.287127630419140},
    {-3.502428351294000, 113.158323856813000, -37.049784423033400, 3.884268258519980},
    {-14.979233658232000, 138.981135797425000, -56.416893378492700, 8.726045497384890},
    {-2.779037020628000, 110.615678614995000, -34.433664062110500, 3.047044590653070},
    {-3.840618260710000, 113.163473591197000, -36.471900043074100, 3.590574185577250},
    {-9.404089942200000, 126.933066002895000, -47.831813782734900, 6.714550463986710},
    {-12.418221952885000, 134.619102630141000, -54.364944915894600, 8.565604285048590},
    {9.102577729027000, 78.127003465118600, -4.934358146495670, -5.851650189360560},
    {-16.289688957113000, 146.686123517701000, -66.637566193823700, 12.659312224839100},
    {4.721778127077000, 88.379302359075200, -12.703756622097600, -3.970279059775710},
    {-9.749423569610000, 129.622227194633000, -51.884535215874700, 8.436967494919540},
    {0.323916123125000, 100.157708593388000, -23.156629579666300, -0.899601836846346},
    {-2.336117275985000, 108.137808790720000, -31.136729777001000, 1.760431562266020},
    {-1.007052875457000, 103.751896268979000, -26.312226003089500, -0.008553154833837},
    {-0.532131222184000, 102.042178317198000, -24.260564460952500, -0.829217771688257},
    {0.135957764275000, 99.436631270009000, -20.873353299605600, -2.297009274939210},
    {-0.278629835043000, 101.177899187141000, -23.311128383589800, -1.159380902413720},
    {0.398071576035000, 98.132742837288900, -18.743393858807700, -3.443248164806470},
    {0.404833530968000, 98.100285453612200, -18.691462044925400, -3.470945132210200},
    {0.148124299983000, 99.409502531632900, -20.917131077559300, -2.209732680385000},
    {0.166139269209000, 99.312221697818000, -20.742025576698000, -2.314795980898420},
    {0.372818622941000, 98.134149381544900, -18.503688175781800, -3.732409668143720},
    {0.179616239166000, 99.293363684195000, -20.822116781078700, -2.186790597947890},
    {-0.123823579543000, 101.205034542056000, -24.836625582582400, 0.623365563102255},
    {0.685118079161000, 95.866019594605800, -13.090792698186400, -7.990245218793070},
    {-0.258888737426000, 102.379666629056000, -28.072180877416500, 3.495485718613010},
    {0.163676870841000, 99.337194249535400, -20.770247166567500, -2.346061250066100},
    {0.020751253987000, 100.409136375936000, -23.450102482565200, -0.112848486736879},
    {0.230328113298000, 98.774436873309700, -19.199883775733700, -3.796371365994920},
    {-0.160563706982000, 101.940660617570000, -27.748687885224200, 3.897552332535040},
    {0.260426714271000, 98.404341079047900, -17.846993177370000, -5.344029394787450},
    {-0.275144149873000, 103.063807597110000, -31.359446079763900, 7.718008410872600},
    {0.237652884971000, 98.448634283516800, -17.513926138990500, -6.127511529894490},
    {0.008029609921000, 100.584130741480000, -24.133965158683400, 0.713195457128813},
    {-0.236270324149000, 102.929410108548000, -31.638859133283700, 8.718415696684970},
    {0.327051686259000, 97.352522205511500, -13.235129053251100, -11.525687391365100},
    {-0.271936420904000, 103.462200898574000, -34.008036609671300, 12.016941172586600},
    {0.170693997302000, 98.814581507414100, -17.741368740627900, -6.960838007945430},
    {-0.050839567939000, 101.207144012018000, -26.354593757208700, 3.375032011960770},
    {-0.037335725779000, 101.057251364044000, -25.799990959726800, 2.691021895094890},
    {-0.029817529321000, 100.971543924427000, -25.474302689181500, 2.278483419067830},
    {-0.023394156801000, 100.896390465943000, -25.181204201087600, 1.897455384534690}};

  std::vector<std::vector<double>> spline100 = {
    {69.740148090887000, 9.013272829772800, 1.000000000000000, 0.000000000000000},
    {14.185955727086000, 9.429929272501700, 0.998958358893177, 0.000000868034256},
    {95.331869000646000, 8.212740573398500, 1.005044302388690, -0.000009275204904},
    {30.748168270236000, 9.665873839832400, 0.994145802890440, 0.000017971043842},
    {27.446817918602000, 9.764914350381500, 0.993155397784950, 0.000021272394194},
    {150.368400055570000, 5.155355020245300, 1.050774889411650, -0.000218808820917},
    {-99.774818141114000, 16.411799839095200, 0.881928217128914, 0.000625424540496},
    {246.327672509992000, -1.758580920088300, 1.199909880414630, -0.001229468495337},
    {-98.598271898780000, 18.936975744437700, 0.785998747124120, 0.001529939059933},
    {172.396615084977000, 0.644820873033600, 1.197572231730720, -0.001556862074617},
    {-12.141788440090000, 14.485201137413800, 0.851562725121211, 0.001326550480462},
    {172.752938674279000, -0.768613849521300, 1.271042637261920, -0.002518682047494},
    {-1.137166257053000, 14.881495594299100, 0.801539353947290, 0.002176350785653},
    {-71.026673646060000, 21.695722564727000, 0.580076977408391, 0.004575526531490},
    {303.005460841989000, -17.577651556518100, 1.954645071651970, -0.011461101234685},
    {26.076830276409000, 13.576819382109400, 0.786352411453451, 0.003142557017797},
    {5.954418054159000, 15.991508848779200, 0.689764832786666, 0.004430391400020},
    {294.087897507610000, -20.745509781536000, 2.251088124575070, -0.017688355233649},
    {-468.787608087097000, 82.242683473749000, -2.383380571912740, 0.051828675213668},
    {954.448134843913000, -120.568409893920000, 7.250146363051560, -0.100702167923267},
    {-883.307331288993000, 155.094910026015000, -6.533019632945210, 0.129017265343346},
    {751.294790309628000, -102.354924125766000, 6.983096660023290, -0.107514769783602},
    {-1.463829947426000, 21.850248216647800, 0.151812181190493, 0.017725445661666},
    {-231.789870520346000, 61.581490215476200, -2.132734233742120, 0.061512585281208},
    {607.156112027524000, -89.428786643138400, 6.927882377774640, -0.119699746949125},
    {-416.095377590863000, 102.430867660309000, -5.063346016190790, 0.130117511258487},
    {543.609398336996000, -84.711563645624300, 7.100912018694900, -0.133441412830703},
    {-191.238215757891000, 64.095078208589400, -2.943536306464460, 0.092558674485381},
    {-184.800535297571000, 62.743165311921100, -2.848902403697610, 0.090350550087486},
    {1468.904356937770000, -296.937648749266000, 23.227956615738600, -0.539840209548890},
    {-2001.632892445850000, 483.933232362048000, -35.337359467610000, 0.924292692534822},
    {2040.699212907380000, -455.908982132577000, 37.500412155723400, -0.957349741067953},
    {-1230.347959376130000, 329.142339215465000, -25.303693552120000, 0.717426411141203},
    {1125.428624791740000, -253.912365366083000, 22.798319575857800, -0.605378949878190},
    {-870.614539854281000, 255.078641618651000, -20.465916017844500, 0.620441058610043},
    {1075.925534686350000, -255.888127948266000, 24.243676319260800, -0.683588717888869},
    {-369.695599015358000, 134.429578151195000, -10.884917229690600, 0.370269088579676},
    {-12.439138626235000, 35.290910393212000, -1.714590462077130, 0.087517346578257},
    {596.684153649391000, -138.309227905342000, 14.777422676285600, -0.434729736136564},
    {-215.801476095407000, 99.342818795006400, -8.393651876997920, 0.318330186845134},
    {151.893634718826000, -10.965714449263000, 2.637201447428990, -0.049364923969095},
    {157.174101095023000, -12.945889340336900, 2.884723308813180, -0.059678334860102},
    {177.819560901185000, -22.236346253109400, 4.278291845729060, -0.129356761705896},
    {175.343431300166000, -20.936378212574500, 4.050797438635470, -0.116086254625436},
    {176.293881897982000, -21.506648571263800, 4.164851510373220, -0.123689859407947},
    {158.343345108222000, -9.390036238174700, 1.438613735428020, 0.080777973712954},
    {137.013697669101000, 6.607199341165200, -2.560695159406900, 0.414053714949194},
    {120.487944214597000, 20.240945941131000, -6.309975474397390, 0.757737743823314},
    {95.704605473816000, 42.545950807835900, -13.001476934409400, 1.426887889824570},
    {75.724033887627000, 62.027008104368200, -19.332820555782000, 2.112783448806560},
    {46.597978979366000, 92.609365758042100, -30.036645734567800, 3.361563052998220},
    {44.204370192733000, 95.302175643004600, -31.046449441428900, 3.487788516355870},
    {18.029340248359000, 126.712211576253000, -43.610463814728000, 5.162990432795720},
    {16.071548818400000, 129.208395649450000, -44.671342045835800, 5.313281515535910},
    {-10.759375526914000, 165.430143515625000, -60.971128585615900, 7.758249496503090},
    {5.609153292608000, 142.104989947802000, -49.891680640897400, 6.004003571922250},
    {-2.214837645442000, 153.840976354874000, -55.759673844432000, 6.982002439177790},
    {-30.253802707966000, 198.002346328350000, -78.944393080507600, 11.039328305491100},
    {-1.173151524881000, 150.019271876257000, -52.553702131854300, 6.201034964904330},
    {-20.957591189790000, 184.147430298226000, -72.177393224487500, 9.962242424325880},
    {-11.758083723998000, 167.588316859803000, -62.241925161435300, 7.975148811715710},
    {-21.648473903406000, 186.132798446197000, -73.832226152934000, 10.389794851611900},
    {-11.990420676466000, 167.299594653659000, -61.590643687781700, 7.737451984161590},
    {-9.266643364481000, 161.783945596887000, -57.867580574458600, 6.899762783663480},
    {-24.018205899938000, 192.762226921346000, -79.552377501579700, 11.959548733324900},
    {-2.286133008329000, 145.494968382095000, -45.283615060621700, 3.677931143426580},
    {-19.371662068897000, 183.937408768378000, -74.115445350336400, 10.885888715855900},
    {-3.904018747198000, 147.975138045429000, -46.244685540051600, 3.685942431532570},
    {-13.377062903874000, 170.710444021455000, -64.432930320876600, 8.536141039753640},
    {-6.779729652862000, 154.382044225196000, -50.962000488957600, 4.831635335974510},
    {-12.275218479772000, 168.395540733814000, -62.873472521283800, 8.206552411800250},
    {-1.991396463018000, 141.400507939836000, -39.252818826553700, 1.317195084170920},
    {-2.367195612806000, 142.415165644261000, -40.166010760533900, 1.591152664364350},
    {-15.423021116112000, 178.645081415940000, -73.678682849339600, 11.924226558413600},
    {2.433680090962000, 127.753482975776000, -25.331664331182100, -3.385662639003420},
    {-5.642899267423000, 151.377477599053000, -48.365059088878600, 4.100190657248280},
    {-2.930628663751000, 143.240665788032000, -40.228247277853000, 1.387920053571440},
    {-1.167852616800000, 137.423504833096000, -33.829370227424900, -0.958334864918111},
    {-0.598160869185000, 135.372614541682000, -31.368301877725000, -1.942762204799400},
    {0.188095093802000, 132.306216286027000, -27.381984145366500, -3.670166555491020},
    {0.219183493869000, 132.175645005744000, -27.199184352967600, -3.755473125278460},
    {0.329467930794000, 131.679365039579000, -26.454764403722600, -4.127683099899970},
    {0.651145783070000, 130.135311348658000, -23.984278498251000, -5.445275582817150},
    {0.214948936730000, 132.359915264990000, -27.766105156011600, -3.302240476754780},
    {0.617358470220000, 130.186903784144000, -23.854684490489200, -5.649092876068300},
    {0.269317182165000, 132.170739126056000, -27.623971640118300, -3.261877681305090},
    {0.370472801319000, 131.563805411133000, -26.410104210276100, -4.071122634530060},
    {0.208781612355000, 132.582459901612000, -28.549278640293400, -2.573700533510600},
    {0.149930749558000, 132.970875596069000, -29.403793168093900, -1.947056546461810},
    {0.323965388839000, 131.770036585034000, -26.641863442716100, -4.064536002581390},
    {-0.016782304442000, 134.223419976656000, -32.529983582612100, 0.645960109337352},
    {0.204663828640000, 132.562573978537000, -28.377868587307800, -2.814135720087500},
    {0.004616990032000, 134.122939319687000, -32.434818474308500, 0.701887515320459},
    {-0.014101788634000, 134.274561426884000, -32.844198163765400, 1.070329235854940},
    {0.357260163764000, 131.155121026739000, -24.109765043352700, -7.081808343204460},
    {-0.413928865737000, 137.864465583400000, -43.566864257664600, 11.726720897294400},
    {0.227955298851000, 132.087508102098000, -26.235991813736200, -5.604151546657930},
    {0.156107670637000, 132.755691044488000, -28.307358935125800, -3.463738854573470},
    {-0.332385981142000, 137.445230101559000, -43.313883917751800, 12.543221126892800},
    {0.085436253676000, 133.308789976866000, -29.663631506258300, -2.472056525756170},
    {0.260640966394000, 131.521701907136000, -23.587532069169700, -9.358302554465250},
    {-0.318000119000000, 137.597433303778000, -44.852591957420000, 15.450933981831300},
    {0.111359509310000, 132.960349318029000, -28.159089608739600, -4.581268836565500},
    {-0.152437917495000, 135.888500755573000, -38.993249927669700, 8.780862223466980},
    {0.313392159433000, 130.578037878597000, -18.813490995166400, -16.780165757696200},
    {-0.141130719794000, 135.895955565553000, -39.553369974307900, 10.181676915205200}};
};
