package io.openems.edge.meter.socomec.singlephase;

import static io.openems.edge.bridge.modbus.api.ElementToChannelConverter.SCALE_FACTOR_1;
import static io.openems.edge.bridge.modbus.api.ElementToChannelConverter.SCALE_FACTOR_1_AND_INVERT_IF_TRUE;
import static io.openems.edge.bridge.modbus.api.ElementToChannelConverter.SET_ZERO_IF_TRUE;
import static io.openems.edge.bridge.modbus.api.ElementToChannelConverter.chain;
import static io.openems.edge.common.type.Phase.SinglePhase.L1;
import static io.openems.edge.common.type.Phase.SinglePhase.L2;
import static io.openems.edge.common.type.Phase.SinglePhase.L3;
import static org.osgi.service.component.annotations.ConfigurationPolicy.REQUIRE;
import static org.osgi.service.component.annotations.ReferenceCardinality.MANDATORY;
import static org.osgi.service.component.annotations.ReferencePolicy.STATIC;
import static org.osgi.service.component.annotations.ReferencePolicyOption.GREEDY;

import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.metatype.annotations.Designate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.openems.common.channel.AccessMode;
import io.openems.common.exceptions.OpenemsException;
import io.openems.common.types.MeterType;
import io.openems.edge.bridge.modbus.api.BridgeModbus;
import io.openems.edge.bridge.modbus.api.ModbusComponent;
import io.openems.edge.bridge.modbus.api.element.DummyRegisterElement;
import io.openems.edge.bridge.modbus.api.element.SignedDoublewordElement;
import io.openems.edge.bridge.modbus.api.element.UnsignedDoublewordElement;
import io.openems.edge.bridge.modbus.api.task.FC3ReadRegistersTask;
import io.openems.edge.common.component.OpenemsComponent;
import io.openems.edge.common.modbusslave.ModbusSlave;
import io.openems.edge.common.modbusslave.ModbusSlaveTable;
import io.openems.edge.common.taskmanager.Priority;
import io.openems.edge.common.type.Phase.SinglePhase;
import io.openems.edge.meter.api.ElectricityMeter;
import io.openems.edge.meter.api.SinglePhaseMeter;
import io.openems.edge.meter.socomec.AbstractSocomecMeter;
import io.openems.edge.meter.socomec.SocomecMeter;

@Designate(ocd = Config.class, factory = true)
@Component(//
		name = "Meter.Socomec.Singlephase", //
		immediate = true, //
		configurationPolicy = REQUIRE //
)
public class MeterSocomecSinglephaseImpl extends AbstractSocomecMeter implements MeterSocomecSinglephase, SocomecMeter,
		SinglePhaseMeter, ElectricityMeter, ModbusComponent, OpenemsComponent, ModbusSlave {

	private final Logger log = LoggerFactory.getLogger(MeterSocomecSinglephaseImpl.class);

	@Reference
	private ConfigurationAdmin cm;

	@Override
	@Reference(policy = STATIC, policyOption = GREEDY, cardinality = MANDATORY)
	protected void setModbus(BridgeModbus modbus) {
		super.setModbus(modbus);
	}

	private Config config;

	public MeterSocomecSinglephaseImpl() {
		super(//
				OpenemsComponent.ChannelId.values(), //
				ModbusComponent.ChannelId.values(), //
				ElectricityMeter.ChannelId.values(), //
				SinglePhaseMeter.ChannelId.values(), //
				SocomecMeter.ChannelId.values(), //
				MeterSocomecSinglephase.ChannelId.values() //
		);
	}

	@Activate
	private void activate(ComponentContext context, Config config) throws OpenemsException {
		if (super.activate(context, config.id(), config.alias(), config.enabled(), config.modbusUnitId(), this.cm,
				"Modbus", config.modbus_id())) {
			return;
		}
		this.config = config;
		this.identifySocomecMeter();
	}

	@Override
	@Deactivate
	protected void deactivate() {
		super.deactivate();
	}

	@Override
	public SinglePhase getPhase() {
		return this.config.phase();
	}

	@Override
	public MeterType getMeterType() {
		return this.config.type();
	}

	@Override
	protected void identifiedCountisE14() throws OpenemsException {
		this.modbusProtocol.addTask(//
				new FC3ReadRegistersTask(0xc558, Priority.HIGH, //
						m(new UnsignedDoublewordElement(0xc558)) //
								.m(ElectricityMeter.ChannelId.VOLTAGE, SCALE_FACTOR_1) //
								.m(ElectricityMeter.ChannelId.VOLTAGE_L1, chain(//
										SCALE_FACTOR_1, //
										SET_ZERO_IF_TRUE(this.config.phase() != L1))) //
								.m(ElectricityMeter.ChannelId.VOLTAGE_L2, chain(//
										SCALE_FACTOR_1, //
										SET_ZERO_IF_TRUE(this.config.phase() != L2))) //
								.m(ElectricityMeter.ChannelId.VOLTAGE_L3, chain(//
										SCALE_FACTOR_1, //
										SET_ZERO_IF_TRUE(this.config.phase() != L3))) //
								.build(), //
						new DummyRegisterElement(0xc55A, 0xc55D), //
						m(ElectricityMeter.ChannelId.FREQUENCY, new UnsignedDoublewordElement(0xc55E)), //
						m(new UnsignedDoublewordElement(0xc560)) //
								.m(ElectricityMeter.ChannelId.CURRENT, SCALE_FACTOR_1) //
								.m(ElectricityMeter.ChannelId.CURRENT_L1, SET_ZERO_IF_TRUE(this.config.phase() != L1)) //
								.m(ElectricityMeter.ChannelId.CURRENT_L2, SET_ZERO_IF_TRUE(this.config.phase() != L2)) //
								.m(ElectricityMeter.ChannelId.CURRENT_L3, SET_ZERO_IF_TRUE(this.config.phase() != L3)) //
								.build(), //
						new DummyRegisterElement(0xc562, 0xc567), //
						m(new SignedDoublewordElement(0xc568)) //
								.m(ElectricityMeter.ChannelId.ACTIVE_POWER,
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()))
								.m(ElectricityMeter.ChannelId.ACTIVE_POWER_L1, chain(//
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()),
										SET_ZERO_IF_TRUE(this.config.phase() != L1))) //
								.m(ElectricityMeter.ChannelId.ACTIVE_POWER_L2, chain(//
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()),
										SET_ZERO_IF_TRUE(this.config.phase() != L2))) //
								.m(ElectricityMeter.ChannelId.ACTIVE_POWER_L3, chain(//
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()),
										SET_ZERO_IF_TRUE(this.config.phase() != L3))) //
								.build(), //
						m(new SignedDoublewordElement(0xc56A)) //
								.m(ElectricityMeter.ChannelId.REACTIVE_POWER,
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()))
								.m(ElectricityMeter.ChannelId.REACTIVE_POWER_L1, chain(//
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()),
										SET_ZERO_IF_TRUE(this.config.phase() != L1))) //
								.m(ElectricityMeter.ChannelId.REACTIVE_POWER_L2, chain(//
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()),
										SET_ZERO_IF_TRUE(this.config.phase() != L2))) //
								.m(ElectricityMeter.ChannelId.REACTIVE_POWER_L3, chain(//
										SCALE_FACTOR_1_AND_INVERT_IF_TRUE(this.config.invert()),
										SET_ZERO_IF_TRUE(this.config.phase() != L3))) //
								.build()));
		if (this.config.invert()) {
			this.modbusProtocol.addTask(new FC3ReadRegistersTask(0xC702, Priority.LOW, //
					m(ElectricityMeter.ChannelId.ACTIVE_CONSUMPTION_ENERGY, new UnsignedDoublewordElement(0xC702),
							SCALE_FACTOR_1), //
					new DummyRegisterElement(0xC704, 0xC707), //
					m(ElectricityMeter.ChannelId.ACTIVE_PRODUCTION_ENERGY, new UnsignedDoublewordElement(0xC708),
							SCALE_FACTOR_1) //
			));
		} else {
			this.modbusProtocol.addTask(new FC3ReadRegistersTask(0xC702, Priority.LOW, //
					m(ElectricityMeter.ChannelId.ACTIVE_PRODUCTION_ENERGY, new UnsignedDoublewordElement(0xC702),
							SCALE_FACTOR_1), //
					new DummyRegisterElement(0xC704, 0xC707), //
					m(ElectricityMeter.ChannelId.ACTIVE_CONSUMPTION_ENERGY, new UnsignedDoublewordElement(0xC708),
							SCALE_FACTOR_1) //
			));
		}
	}

	@Override
	protected void identifiedCountisE23_E24_E27_E28() throws OpenemsException {
		this.thisIsNotASinglePhaseMeter();
	}

	@Override
	protected void identifiedCountisE34_E44() throws OpenemsException {
		this.thisIsNotASinglePhaseMeter();
	}

	@Override
	protected void identifiedDirisA10() throws OpenemsException {
		this.thisIsNotASinglePhaseMeter();
	}

	@Override
	protected void identifiedDirisA14() throws OpenemsException {
		this.thisIsNotASinglePhaseMeter();
	}

	@Override
	protected void identifiedDirisB30() throws OpenemsException {
		this.thisIsNotASinglePhaseMeter();
	}

	private void thisIsNotASinglePhaseMeter() {
		this.logError(this.log, "This is not a singlephase meter!");
		this.channel(MeterSocomecSinglephase.ChannelId.NOT_A_SINGLEPHASE_METER).setNextValue(true);
	}

	@Override
	public String debugLog() {
		return "L:" + this.getActivePower().asString();
	}

	@Override
	public ModbusSlaveTable getModbusSlaveTable(AccessMode accessMode) {
		return new ModbusSlaveTable(//
				OpenemsComponent.getModbusSlaveNatureTable(accessMode), //
				ElectricityMeter.getModbusSlaveNatureTable(accessMode), //
				SinglePhaseMeter.getModbusSlaveNatureTable(accessMode) //
		);
	}
}
