package io.openems.edge.battery.fenecon.home;

import static io.openems.edge.battery.fenecon.home.BatteryFeneconHome.ChannelId.BATTERY_HARDWARE_TYPE;
import static io.openems.edge.battery.fenecon.home.BatteryFeneconHome.ChannelId.NUMBER_OF_MODULES_PER_TOWER;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

import org.junit.Test;

import io.openems.common.test.DummyConfigurationAdmin;
import io.openems.edge.battery.api.Battery;
import io.openems.edge.bridge.modbus.test.DummyModbusBridge;
import io.openems.edge.common.channel.ChannelId;
import io.openems.edge.common.startstop.StartStopConfig;
import io.openems.edge.common.test.AbstractComponentTest.TestCase;
import io.openems.edge.common.test.ComponentTest;
import io.openems.edge.common.test.DummyComponentManager;
import io.openems.edge.common.test.DummySerialNumberStorage;

public class TowersAndModulesTest {

	private static final int TOWERS = 1;
	private static final int MODULES = 5;
	private static final int CELLS = 14;

	@Test
	public void testChannelsCreatedDynamicallyBasedOnRegister() throws Exception {
		var battery = new BatteryFeneconHomeImpl();
		var componentTest = new ComponentTest(battery) //
				.addReference("cm", new DummyConfigurationAdmin()) //
				.addReference("componentManager", new DummyComponentManager()) //
				.addReference("setModbus", new DummyModbusBridge("modbus0")) //
				.addReference("serialNumberStorage", new DummySerialNumberStorage()) //
				.activate(MyConfig.create() //
						.setId("battery0") //
						.setModbusId("modbus0") //
						.setModbusUnitId(0) //
						.setBatteryStartUpRelay("io0/Relay4") //
						.setStartStop(StartStopConfig.AUTO) //
						.build());

		// initial home (1 tower, each tower 5 modules)
		componentTest.next(new TestCase() //
				.input(BatteryFeneconHome.ChannelId.RACK_NUMBER_OF_BATTERY_BCU, 1) //
				.input(NUMBER_OF_MODULES_PER_TOWER, MODULES) //
				.input(BATTERY_HARDWARE_TYPE, BatteryFeneconHomeHardwareType.BATTERY_52));
		checkDynamicChannels(battery, TOWERS, MODULES, CELLS, BatteryFeneconHomeHardwareType.BATTERY_52);

		// add new module (1 tower, each tower 6 modules)
		componentTest.next(new TestCase() //
				.input(NUMBER_OF_MODULES_PER_TOWER, MODULES + 1));
		checkDynamicChannels(battery, TOWERS, MODULES + 1, CELLS, BatteryFeneconHomeHardwareType.BATTERY_52);

		// add new tower home (2 tower, each tower 6 modules)
		componentTest.next(new TestCase() //
				.input(BatteryFeneconHome.ChannelId.RACK_NUMBER_OF_BATTERY_BCU, 2));
		checkDynamicChannels(battery, TOWERS + 1, MODULES + 1, CELLS, BatteryFeneconHomeHardwareType.BATTERY_52);
	}

	@Test
	public void testSerialNumberFormatterForBms() {
		assertEquals("519100001009210104000035", //
				BatteryFeneconHomeImpl.buildSerialNumber("519100001009", 707002403));
	}

	@Test
	public void testSerialNumberFormatterForOldBms() {
		assertNull(BatteryFeneconHomeImpl.buildSerialNumber("519100001009", 0));
	}

	@Test
	public void testSerialNumberFormatterForBattery() {
		assertEquals("519110001210201219000039", //
				BatteryFeneconHomeImpl.buildSerialNumber("519110001210", 697499687));
	}

	/**
	 * Check if all dynamic channels depending on tower, modules and cells
	 * parameters are created. If channel not exists an exception will be thrown and
	 * the test fails.
	 *
	 * @param battery      the {@link Battery}
	 * @param towers       number of given towers
	 * @param modules      number of given modules
	 * @param cells        number of given cells
	 * @param hardwareType hardware type
	 */
	private static void checkDynamicChannels(Battery battery, int towers, int modules, int cells,
			BatteryFeneconHomeHardwareType hardwareType) {
		for (var tower = 0; tower < towers; tower++) {
			// check for each tower the serial number channel is existent
			battery.channel("Tower" + tower + "BmsSerialNumber");

			for (var module = 0; module < modules; module++) {
				// check for each tower and module the serial number channel is existent
				battery.channel("Tower" + tower + "Module" + module + "SerialNumber");

				// check for each tower, module and cell voltage
				for (var cell = 0; cell < hardwareType.cellsPerModule; cell++) {
					battery.channel(ChannelId.channelIdUpperToCamel(
							BatteryFeneconHomeImpl.generateCellVoltageChannelName(tower, module, cell)));
				}
				// check for each tower, module and temperature sensor
				for (var sensor = 0; sensor < hardwareType.tempSensorsPerModule; sensor++) {
					battery.channel(ChannelId.channelIdUpperToCamel(
							BatteryFeneconHomeImpl.generateTempSensorChannelName(tower, module, sensor + 1)));
				}
				// check for each tower, module and temperature balancing
				for (var balancing = 0; balancing < 2; balancing++) {
					battery.channel(ChannelId.channelIdUpperToCamel(
							BatteryFeneconHomeImpl.generateTempBalancingChannelName(tower, module, balancing + 1)));
				}
			}
		}
	}
}
