# Prime Wave Final Analysis Script (Up to 1,000,000)
# This script computes sequences and performs FFT analysis to identify Global Drift (DC)
# and Local Oscillations (Spectral Peaks) in the DeltaAlpha sequence.
# FIX: Corrected ValueError by ensuring all CSV writing is done inside the 'with open' block.

import numpy as np
import matplotlib.pyplot as plt
from math import atan
from scipy.signal import find_peaks
from scipy.signal.windows import hann
import os
import csv
import time

# ===============================================================
# 1. Configuration and Setup
# ===============================================================
# Target limit for prime generation (1 million)
PRIME_LIMIT = 1000000 
# Output folder setup
folder = r"C:\Users\allen\OneDrive\Desktop\Prime Wave Analysis\Output"
os.makedirs(folder, exist_ok=True)

# ===============================================================
# 2. Prime generation and Sequence Computation
# ===============================================================
def sieve_of_eratosthenes(limit):
    """Return a NumPy array of all primes <= limit."""
    sieve = np.ones(limit + 1, dtype=bool)
    sieve[:2] = False
    for num in range(2, int(limit**0.5) + 1):
        if sieve[num]:
            sieve[num*num:limit+1:num] = False
    return np.nonzero(sieve)[0]

def compute_sequences(primes):
    """Return gaps, 2nd diffs, alphas, and alpha diffs (optimized)."""
    gaps = np.diff(primes)
    second_diff = np.diff(gaps)
    
    # Optimized: Arctan(P_n / P_{n+1})
    alpha = np.arctan(primes[:-1] / primes[1:])
    
    delta_alpha = np.diff(alpha)
    return gaps, second_diff, alpha, delta_alpha

# ===============================================================
# 3. Analysis Function (Focus on DeltaAlpha)
# ===============================================================
def analyze_delta_alpha(delta_alpha, primes_count):
    """Performs FFT analysis focusing on the Windowed FFT for DeltaAlpha."""
    
    seq_name = "DeltaAlpha"
    results_file = os.path.join(folder, f"{seq_name}_final_analysis.csv")
    
    # --- 1) Quantify Global Drift (DC Component) ---
    avg_delta_alpha = np.mean(delta_alpha)
    
    print("-" * 50)
    print(f"[{seq_name} - N={primes_count:,}] **CRITICAL DATA POINTS**")
    print(f"1. Total Primes Analyzed: {primes_count:,}")
    print(f"2. Global Drift (Avg DeltaAlpha): {avg_delta_alpha:.10f}")
    
    # --- 2) Local Oscillations (Windowed FFT) ---
    seq = delta_alpha
    seq_detrend = seq - np.mean(seq)
    window = hann(len(seq_detrend))
    fft_vals_win = np.fft.fft(seq_detrend * window)
    
    fft_mag_win = np.abs(fft_vals_win[:len(seq)//2])
    fft_freq_win = np.fft.fftfreq(len(seq))[:len(seq)//2]
    
    # Find the top 5 peaks for reporting
    # Lower the height threshold for these expected weak signals
    win_peaks, _ = find_peaks(fft_mag_win, height=np.std(fft_mag_win) * 0.3) 
    win_peak_mags = fft_mag_win[win_peaks]
    win_peak_freqs = fft_freq_win[win_peaks]
    
    # Sort by magnitude (largest first) and take top 5
    win_idx_sort = np.argsort(win_peak_mags)[::-1][:5] 
    top_win_freqs = win_peak_freqs[win_idx_sort]
    top_win_mags = win_peak_mags[win_idx_sort]
    
    print("\n3. Local Oscillation Frequencies (Windowed FFT Top 5):")
    peak_results = []
    for f, m in zip(top_win_freqs, top_win_mags):
        period = 1/f if f > 1e-10 else np.inf
        peak_results.append([f, m, period])
        print(f"   Freq: {f:.5f}, Mag: {m:.2f}, Period: {period:.3f}")
        
    # --- 4) File Writing (Done safely inside one 'with' block) ---
    with open(results_file, "w", newline="") as csv_file:
        writer = csv.writer(csv_file)
        writer.writerow(["Metric", "Value"])
        writer.writerow(["Primes Count", primes_count])
        writer.writerow(["Average Delta Alpha (Global Drift)", avg_delta_alpha])
        writer.writerow([])
        writer.writerow(["Windowed FFT Top Peaks", "Frequency", "Magnitude", "Period"])
        writer.writerows(peak_results)
    
    # --- 5) Plot Generation (For Visual Confirmation) ---
    plt.figure(figsize=(12, 6))
    plt.plot(fft_freq_win, fft_mag_win, lw=0.8, color="#0057B8")
    
    # Annotate the top 3 peaks (the expected 0.33, 0.39, 0.47 group)
    for f, m in zip(top_win_freqs[:3], top_win_mags[:3]):
        plt.annotate(
            f"f={f:.3f}\nMag={m:.1f}",
            xy=(f, m),
            xytext=(f, m * 1.1),
            arrowprops=dict(facecolor='red', shrink=0.05, width=1, headwidth=5),
            ha='center',
            fontsize=9,
            color='red'
        )
        
    plt.title(f"{seq_name} – Windowed FFT (Hann) N={primes_count:,}")
    plt.xlabel("Frequency (f)")
    plt.ylabel("Magnitude")
    # Limit X-axis to the crucial range (0.2 to 0.55) where the peaks were found, for clarity
    plt.xlim(0.2, 0.55) 
    plt.grid(alpha=0.3)
    plt.tight_layout()
    plot_path = os.path.join(folder, f"{seq_name}_WindowedFFT.png")
    plt.savefig(plot_path)
    plt.close()
    
    print(f"\n[{seq_name}] Analysis results saved to CSV.")
    print(f"[{seq_name}] Visual confirmation plot saved to: {plot_path}")
    print("-" * 50)


# ===============================================================
# 4. Main function
# ===============================================================
def main():
    print(f"\n[Prime Wave Analysis] Generating primes up to {PRIME_LIMIT:,}...")
    start = time.time()
    
    # 1. Generate Primes
    primes = sieve_of_eratosthenes(PRIME_LIMIT)
    primes_count = len(primes)
    
    print(f"Found {primes_count:,} primes in {time.time()-start:.2f}s")
    
    # 2. Compute Delta Alpha
    sequence_start_time = time.time()
    gaps, second_diff, alpha, delta_alpha = compute_sequences(primes)
    print(f"Sequences computed in {time.time()-sequence_start_time:.2f}s")
    
    # 3. Run Analysis on Delta Alpha
    analysis_start_time = time.time()
    analyze_delta_alpha(delta_alpha, primes_count)
    print(f"Analysis completed in {time.time()-analysis_start_time:.2f}s")


# ===============================================================
# 5. Run script
# ===============================================================
if __name__ == "__main__":
    main()
