/*
 * Copyright 2018 The Hyve
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.radarbase.connect.rest.oura;

import static org.apache.kafka.common.config.ConfigDef.NO_DEFAULT_VALUE;

import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import okhttp3.Headers;
import okhttp3.HttpUrl;
import org.apache.kafka.common.config.AbstractConfig;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.common.config.ConfigDef.Importance;
import org.apache.kafka.common.config.ConfigDef.NonEmptyString;
import org.apache.kafka.common.config.ConfigDef.Type;
import org.apache.kafka.common.config.ConfigDef.Validator;
import org.apache.kafka.common.config.ConfigDef.Width;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.connect.errors.ConnectException;
import org.radarbase.connect.rest.oura.user.OuraUserRepository;
import org.radarbase.connect.rest.oura.user.OuraServiceUserRepository;

public class OuraRestSourceConnectorConfig extends AbstractConfig {
  public static final Pattern COLON_PATTERN = Pattern.compile(":");

  private static final String SOURCE_POLL_INTERVAL_CONFIG = "rest.source.poll.interval.ms";
  private static final String SOURCE_POLL_INTERVAL_DOC = "How often to poll the source URL.";
  private static final String SOURCE_POLL_INTERVAL_DISPLAY = "Polling interval";
  private static final Long SOURCE_POLL_INTERVAL_DEFAULT = 60000L;

  static final String SOURCE_URL_CONFIG = "rest.source.base.url";
  private static final String SOURCE_URL_DOC = "Base URL for REST source connector.";
  private static final String SOURCE_URL_DISPLAY = "Base URL for REST source connector.";

  public static final String OURA_USERS_CONFIG = "oura.users";
  private static final String OURA_USERS_DOC =
      "The user ID of Oura users to include in polling, separated by commas. "
          + "Non existing user names will be ignored. "
          + "If empty, all users in the user directory will be used.";
  private static final String OURA_USERS_DISPLAY = "Oura users";

  public static final String OURA_API_CLIENT_CONFIG = "oura.api.client";
  private static final String OURA_API_CLIENT_DOC =
      "Client ID for the Oura API";
  private static final String OURA_API_CLIENT_DISPLAY = "Oura API client ID";

  public static final String OURA_API_SECRET_CONFIG = "oura.api.secret";
  private static final String OURA_API_SECRET_DOC = "Secret for the Oura API client set in oura.api.client.";
  private static final String OURA_API_SECRET_DISPLAY = "Oura API client secret";

  public static final String OURA_USER_REPOSITORY_CONFIG = "oura.user.repository.class";
  private static final String OURA_USER_REPOSITORY_DOC = "Class for managing users and authentication.";
  private static final String OURA_USER_REPOSITORY_DISPLAY = "User repository class";

  public static final String OURA_USER_POLL_INTERVAL = "oura.user.poll.interval";
  private static final String OURA_USER_POLL_INTERVAL_DOC = "Polling interval per Oura user per request route in seconds.";
  // 150 requests per hour -> 2.5 per minute. There are currently 5 paths, that limits us to 1
  // call per route per 2 minutes.
  private static final int OURA_USER_POLL_INTERVAL_DEFAULT = 150;
  private static final String OURA_USER_POLL_INTERVAL_DISPLAY = "Per-user per-route polling interval.";

  public static final String OURA_USER_REPOSITORY_URL_CONFIG = "oura.user.repository.url";
  private static final String OURA_USER_REPOSITORY_URL_DOC = "URL for webservice containing user credentials. Only used if a webservice-based user repository is configured.";
  private static final String OURA_USER_REPOSITORY_URL_DISPLAY = "User repository URL";
  private static final String OURA_USER_REPOSITORY_URL_DEFAULT = "";

  public static final String OURA_USER_REPOSITORY_CLIENT_ID_CONFIG = "oura.user.repository.client.id";
  private static final String OURA_USER_REPOSITORY_CLIENT_ID_DOC = "Client ID for connecting to the service repository.";
  private static final String OURA_USER_REPOSITORY_CLIENT_ID_DISPLAY = "Client ID for user repository.";

  public static final String OURA_USER_REPOSITORY_CLIENT_SECRET_CONFIG = "oura.user.repository.client.secret";
  private static final String OURA_USER_REPOSITORY_CLIENT_SECRET_DOC = "Client secret for connecting to the service repository.";
  private static final String OURA_USER_REPOSITORY_CLIENT_SECRET_DISPLAY = "Client Secret for user repository.";

  public static final String OURA_USER_REPOSITORY_TOKEN_URL_CONFIG = "oura.user.repository.oauth2.token.url";
  private static final String OURA_USER_REPOSITORY_TOKEN_URL_DOC = "OAuth 2.0 token url for retrieving client credentials.";
  private static final String OURA_USER_REPOSITORY_TOKEN_URL_DISPLAY = "OAuth 2.0 token URL.";

// Route enabled configs strings
  private static final String OURA_DAILY_ACTIVITY_ENABLED_CONFIG = "oura.daily.activity.enabled";
  private static final String OURA_DAILY_ACTIVITY_ENABLED_DOC = "Enable or disable Oura daily activity";
  private static final String OURA_DAILY_ACTIVITY_ENABLED_DISPLAY = "Oura daily activity enabled";

  private static final String OURA_DAILY_READINESS_ENABLED_CONFIG = "oura.daily.readiness.enabled";
  private static final String OURA_DAILY_READINESS_ENABLED_DOC = "Enable or disable Oura daily readiness";
  private static final String OURA_DAILY_READINESS_ENABLED_DISPLAY = "Oura daily readiness enabled";

  private static final String OURA_DAILY_SLEEP_ENABLED_CONFIG = "oura.daily.sleep.enabled";
  private static final String OURA_DAILY_SLEEP_ENABLED_DOC = "Enable or disable Oura daily sleep";
  private static final String OURA_DAILY_SLEEP_ENABLED_DISPLAY = "Oura daily sleep enabled";

  private static final String OURA_DAILY_OXYGEN_SATURATION_ENABLED_CONFIG = "oura.daily.oxygen.saturation.enabled";
  private static final String OURA_DAILY_OXYGEN_SATURATION_ENABLED_DOC = "Enable or disable Oura daily oxygen saturation";
  private static final String OURA_DAILY_OXYGEN_SATURATION_ENABLED_DISPLAY = "Oura daily oxygen saturation enabled";

  private static final String OURA_HEART_RATE_ENABLED_CONFIG = "oura.heart.rate.enabled";
  private static final String OURA_HEART_RATE_ENABLED_DOC = "Enable or disable Oura heart rate";
  private static final String OURA_HEART_RATE_ENABLED_DISPLAY = "Oura heart rate enabled";

  private static final String OURA_PERSONAL_INFO_ENABLED_CONFIG = "oura.personal.info.enabled";
  private static final String OURA_PERSONAL_INFO_ENABLED_DOC = "Enable or disable Oura personal info";
  private static final String OURA_PERSONAL_INFO_ENABLED_DISPLAY = "Oura personal info enabled";

  private static final String OURA_SESSION_ENABLED_CONFIG = "oura.session.enabled";
  private static final String OURA_SESSION_ENABLED_DOC = "Enable or disable Oura sessions";
  private static final String OURA_SESSION_ENABLED_DISPLAY = "Oura sessions enabled";

  private static final String OURA_SLEEP_ENABLED_CONFIG = "oura.sleep.enabled";
  private static final String OURA_SLEEP_ENABLED_DOC = "Enable or disable Oura sleep";
  private static final String OURA_SLEEP_ENABLED_DISPLAY = "Oura sleep enabled";

  private static final String OURA_TAG_ENABLED_CONFIG = "oura.tag.enabled";
  private static final String OURA_TAG_ENABLED_DOC = "Enable or disable Oura tags";
  private static final String OURA_TAG_ENABLED_DISPLAY = "Oura tags enabled";

  private static final String OURA_WORKOUT_ENABLED_CONFIG = "oura.workout.enabled";
  private static final String OURA_WORKOUT_ENABLED_DOC = "Enable or disable Oura workouts";
  private static final String OURA_WORKOUT_ENABLED_DISPLAY = "Oura workouts enabled";

  private static final String OURA_RING_CONFIGURATION_ENABLED_CONFIG = "oura.ring.configuration.enabled";
  private static final String OURA_RING_CONFIGURATION_ENABLED_DOC = "Enable or disable Oura ring configuration";
  private static final String OURA_RING_CONFIGURATION_ENABLED_DISPLAY = "Oura ring configuration enabled";

  private static final String OURA_REST_MODE_PERIOD_ENABLED_CONFIG = "oura.rest.mode.period.enabled";
  private static final String OURA_REST_MODE_PERIOD_ENABLED_DOC = "Enable or disable Oura rest mode period";
  private static final String OURA_REST_MODE_PERIOD_ENABLED_DISPLAY = "Oura rest mode period enabled";

  private static final String OURA_SLEEP_TIME_RECOMMENDATION_ENABLED_CONFIG = "oura.sleep.time.recommendation.enabled";
  private static final String OURA_SLEEP_TIME_RECOMMENDATION_ENABLED_DOC = "Enable or disable Oura sleep time recommendation";
  private static final String OURA_SLEEP_TIME_RECOMMENDATION_ENABLED_DISPLAY = "Oura sleep time recommendation enabled";

  private static final String OURA_DAILY_STRESS_ENABLED_CONFIG = "oura.daily.stress.enabled";
  private static final String OURA_DAILY_STRESS_ENABLED_DOC = "Enable or disable Oura daily stress";
  private static final String OURA_DAILY_STRESS_ENABLED_DISPLAY = "Oura daily stress enabled";

  private static final String OURA_VO2MAX_ENABLED_CONFIG = "oura.vo2max.enabled";
  private static final String OURA_VO2MAX_ENABLED_DOC = "Enable or disable Oura vo2max";
  private static final String OURA_VO2MAX_ENABLED_DISPLAY = "Oura vo2max enabled";

  private static final String OURA_DAILY_RESILIENCE_ENABLED_CONFIG = "oura.daily.resilience.enabled";
  private static final String OURA_DAILY_RESILIENCE_ENABLED_DOC = "Enable or disable Oura daily resilience";
  private static final String OURA_DAILY_RESILIENCE_ENABLED_DISPLAY = "Oura daily resilience enabled";

  private static final String OURA_DAILY_CARDIOVASCULAR_AGE_ENABLED_CONFIG = "oura.daily.cardiovascular.age.enabled";
  private static final String OURA_DAILY_CARDIOVASCULAR_AGE_ENABLED_DOC = "Enable or disable Oura daily cardiovascular age";
  private static final String OURA_DAILY_CARDIOVASCULAR_AGE_ENABLED_DISPLAY = "Oura daily cardiovascular age enabled";

  private static final String OURA_ENHANCED_TAG_ENABLED_CONFIG = "oura.enhanced.tag.enabled";
  private static final String OURA_ENHANCED_TAG_ENABLED_DOC = "Enable or disable Oura enhanced tag";
  private static final String OURA_ENHANCED_TAG_ENABLED_DISPLAY = "Oura enhanced tag enabled";

  private OuraUserRepository userRepository;
  private final Headers clientCredentials;

  public OuraRestSourceConnectorConfig(ConfigDef config, Map<String, String> parsedConfig, boolean doLog) {
    super(config, parsedConfig, doLog);

    String credentialString = getOuraClient() + ":" + getOuraClientSecret();
    String credentialsBase64 = Base64.getEncoder().encodeToString(
        credentialString.getBytes(StandardCharsets.UTF_8));
    this.clientCredentials = Headers.of("Authorization", "Basic " + credentialsBase64);
  }

  public OuraRestSourceConnectorConfig(Map<String, String> parsedConfig, boolean doLog) {
    this(OuraRestSourceConnectorConfig.conf(), parsedConfig, doLog);
  }

  public static ConfigDef conf() {
    int orderInGroup = 0;
    String group = "Oura";

    Validator nonControlChar = new ConfigDef.NonEmptyStringWithoutControlChars() {
      @Override
      public String toString() {
        return "non-empty string without control characters";
      }
    };

    return new ConfigDef()
    .define(SOURCE_POLL_INTERVAL_CONFIG,
            Type.LONG,
            SOURCE_POLL_INTERVAL_DEFAULT,
            Importance.LOW,
            SOURCE_POLL_INTERVAL_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            SOURCE_POLL_INTERVAL_DISPLAY)

        .define(SOURCE_URL_CONFIG,
            Type.STRING,
            NO_DEFAULT_VALUE,
            Importance.HIGH,
            SOURCE_URL_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            SOURCE_URL_DISPLAY)

        .define(OURA_USERS_CONFIG,
            Type.LIST,
            Collections.emptyList(),
            Importance.HIGH,
            OURA_USERS_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_USERS_DISPLAY)

        .define(OURA_API_CLIENT_CONFIG,
            Type.STRING,
            NO_DEFAULT_VALUE,
            new NonEmptyString(),
            Importance.HIGH,
            OURA_API_CLIENT_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_API_CLIENT_DISPLAY)

        .define(OURA_API_SECRET_CONFIG,
            Type.PASSWORD,
            NO_DEFAULT_VALUE,
            Importance.HIGH,
            OURA_API_SECRET_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_API_SECRET_DISPLAY)

        .define(OURA_USER_POLL_INTERVAL,
            Type.INT,
            OURA_USER_POLL_INTERVAL_DEFAULT,
            Importance.MEDIUM,
            OURA_USER_POLL_INTERVAL_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_USER_POLL_INTERVAL_DISPLAY)

        .define(OURA_USER_REPOSITORY_CONFIG,
            Type.CLASS,
            OuraServiceUserRepository.class,
            Importance.MEDIUM,
            OURA_USER_REPOSITORY_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_USER_REPOSITORY_DISPLAY)

        .define(OURA_USER_REPOSITORY_URL_CONFIG,
            Type.STRING,
            OURA_USER_REPOSITORY_URL_DEFAULT,
            Importance.LOW,
            OURA_USER_REPOSITORY_URL_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_USER_REPOSITORY_URL_DISPLAY)

        .define(OURA_USER_REPOSITORY_CLIENT_ID_CONFIG,
            Type.STRING,
            "",
            Importance.MEDIUM,
            OURA_USER_REPOSITORY_CLIENT_ID_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_USER_REPOSITORY_CLIENT_ID_DISPLAY)

        .define(OURA_USER_REPOSITORY_CLIENT_SECRET_CONFIG,
            Type.PASSWORD,
            "",
            Importance.MEDIUM,
            OURA_USER_REPOSITORY_CLIENT_SECRET_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_USER_REPOSITORY_CLIENT_SECRET_DISPLAY)

        .define(OURA_USER_REPOSITORY_TOKEN_URL_CONFIG,
            Type.STRING,
            "",
            Importance.MEDIUM,
            OURA_USER_REPOSITORY_TOKEN_URL_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_USER_REPOSITORY_TOKEN_URL_DISPLAY)

        // Route enable/disable flags
        .define(OURA_DAILY_ACTIVITY_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_DAILY_ACTIVITY_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_DAILY_ACTIVITY_ENABLED_DISPLAY)

        .define(OURA_DAILY_READINESS_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_DAILY_READINESS_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_DAILY_READINESS_ENABLED_DISPLAY)

        .define(OURA_DAILY_SLEEP_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_DAILY_SLEEP_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_DAILY_SLEEP_ENABLED_DISPLAY)

        .define(OURA_DAILY_OXYGEN_SATURATION_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_DAILY_OXYGEN_SATURATION_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_DAILY_OXYGEN_SATURATION_ENABLED_DISPLAY)

        .define(OURA_HEART_RATE_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_HEART_RATE_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_HEART_RATE_ENABLED_DISPLAY)

        .define(OURA_PERSONAL_INFO_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_PERSONAL_INFO_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_PERSONAL_INFO_ENABLED_DISPLAY)

        .define(OURA_SESSION_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_SESSION_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_SESSION_ENABLED_DISPLAY)

        .define(OURA_SLEEP_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_SLEEP_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_SLEEP_ENABLED_DISPLAY)

        .define(OURA_TAG_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_TAG_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_TAG_ENABLED_DISPLAY)

        .define(OURA_WORKOUT_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_WORKOUT_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_WORKOUT_ENABLED_DISPLAY)

        .define(OURA_RING_CONFIGURATION_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_RING_CONFIGURATION_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_RING_CONFIGURATION_ENABLED_DISPLAY)

        .define(OURA_REST_MODE_PERIOD_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_REST_MODE_PERIOD_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_REST_MODE_PERIOD_ENABLED_DISPLAY)

        .define(OURA_SLEEP_TIME_RECOMMENDATION_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_SLEEP_TIME_RECOMMENDATION_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_SLEEP_TIME_RECOMMENDATION_ENABLED_DISPLAY)

        .define(OURA_DAILY_STRESS_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_DAILY_STRESS_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_DAILY_STRESS_ENABLED_DISPLAY)

        .define(OURA_VO2MAX_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_VO2MAX_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_VO2MAX_ENABLED_DISPLAY)

        .define(OURA_DAILY_RESILIENCE_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_DAILY_RESILIENCE_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_DAILY_RESILIENCE_ENABLED_DISPLAY)

        .define(OURA_DAILY_CARDIOVASCULAR_AGE_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_DAILY_CARDIOVASCULAR_AGE_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_DAILY_CARDIOVASCULAR_AGE_ENABLED_DISPLAY)

        .define(OURA_ENHANCED_TAG_ENABLED_CONFIG,
            Type.BOOLEAN,
            true,
            Importance.LOW,
            OURA_ENHANCED_TAG_ENABLED_DOC,
            group,
            ++orderInGroup,
            Width.SHORT,
            OURA_ENHANCED_TAG_ENABLED_DISPLAY)
        ;
  }

  public List<String> getOuraUsers() {
    return getList(OURA_USERS_CONFIG);
  }

  public String getOuraClient() {
    return getString(OURA_API_CLIENT_CONFIG);
  }

  public String getOuraClientSecret() {
    return getPassword(OURA_API_SECRET_CONFIG).value();
  }

  public OuraUserRepository getUserRepository(OuraUserRepository reuse) {
    if (reuse != null && reuse.getClass().equals(getClass(OURA_USER_REPOSITORY_CONFIG))) {
      userRepository = reuse;
    } else {
      userRepository = createUserRepository();
    }
    userRepository.initialize(this);
    return userRepository;
  }

  public OuraUserRepository getUserRepository() {
    userRepository.initialize(this);
    return userRepository;
  }

  @SuppressWarnings("unchecked")
  public OuraUserRepository createUserRepository() {
    try {
      return ((Class<? extends OuraUserRepository>)
          getClass(OURA_USER_REPOSITORY_CONFIG)).getDeclaredConstructor().newInstance();
    } catch (IllegalAccessException | InstantiationException
        | InvocationTargetException | NoSuchMethodException e) {
      throw new ConnectException("Invalid class. " + e);
    }
  }

  public HttpUrl getOuraUserRepositoryUrl() {
    String urlString = getString(OURA_USER_REPOSITORY_URL_CONFIG).trim();
    if (urlString.charAt(urlString.length() - 1) != '/') {
      urlString += '/';
    }
    HttpUrl url = HttpUrl.parse(urlString);
    if (url == null) {
      throw new ConfigException(OURA_USER_REPOSITORY_URL_CONFIG,
          getString(OURA_USER_REPOSITORY_URL_CONFIG),
          "User repository URL " + urlString + " cannot be parsed as URL.");
    }
    return url;
  }

  public Headers getClientCredentials() {
    return clientCredentials;
  }

  public Duration getPollIntervalPerUser() {
    return Duration.ofSeconds(getInt(OURA_USER_POLL_INTERVAL));
  }

  public Duration getTooManyRequestsCooldownInterval() {
    return Duration.ofHours(1);
  }

  public String getOuraUserRepositoryClientId() {
    return getString(OURA_USER_REPOSITORY_CLIENT_ID_CONFIG);
  }

  public String getOuraUserRepositoryClientSecret() {
    return getPassword(OURA_USER_REPOSITORY_CLIENT_SECRET_CONFIG).value();
  }

  public URL getOuraUserRepositoryTokenUrl() {
    String value = getString(OURA_USER_REPOSITORY_TOKEN_URL_CONFIG);
    if (value == null || value.isEmpty()) {
      return null;
    } else {
      try {
        return new URL(getString(OURA_USER_REPOSITORY_TOKEN_URL_CONFIG));
      } catch (MalformedURLException e) {
        throw new ConfigException("Oura user repository token URL is invalid.");
      }
    }
  }

  // Route enabled getters
  public boolean getOuraDailyActivityEnabled() { return getBoolean(OURA_DAILY_ACTIVITY_ENABLED_CONFIG); }
  public boolean getOuraDailyReadinessEnabled() { return getBoolean(OURA_DAILY_READINESS_ENABLED_CONFIG); }
  public boolean getOuraDailySleepEnabled() { return getBoolean(OURA_DAILY_SLEEP_ENABLED_CONFIG); }
  public boolean getOuraDailyOxygenSaturationEnabled() { return getBoolean(OURA_DAILY_OXYGEN_SATURATION_ENABLED_CONFIG); }
  public boolean getOuraHeartRateEnabled() { return getBoolean(OURA_HEART_RATE_ENABLED_CONFIG); }
  public boolean getOuraPersonalInfoEnabled() { return getBoolean(OURA_PERSONAL_INFO_ENABLED_CONFIG); }
  public boolean getOuraSessionEnabled() { return getBoolean(OURA_SESSION_ENABLED_CONFIG); }
  public boolean getOuraSleepEnabled() { return getBoolean(OURA_SLEEP_ENABLED_CONFIG); }
  public boolean getOuraTagEnabled() { return getBoolean(OURA_TAG_ENABLED_CONFIG); }
  public boolean getOuraWorkoutEnabled() { return getBoolean(OURA_WORKOUT_ENABLED_CONFIG); }
  public boolean getOuraRingConfigurationEnabled() { return getBoolean(OURA_RING_CONFIGURATION_ENABLED_CONFIG); }
  public boolean getOuraRestModePeriodEnabled() { return getBoolean(OURA_REST_MODE_PERIOD_ENABLED_CONFIG); }
  public boolean getOuraSleepTimeRecommendationEnabled() { return getBoolean(OURA_SLEEP_TIME_RECOMMENDATION_ENABLED_CONFIG); }
  public boolean getOuraDailyStressEnabled() { return getBoolean(OURA_DAILY_STRESS_ENABLED_CONFIG); }
  public boolean getOuraVo2maxEnabled() { return getBoolean(OURA_VO2MAX_ENABLED_CONFIG); }
  public boolean getOuraDailyResilienceEnabled() { return getBoolean(OURA_DAILY_RESILIENCE_ENABLED_CONFIG); }
  public boolean getOuraDailyCardiovascularAgeEnabled() { return getBoolean(OURA_DAILY_CARDIOVASCULAR_AGE_ENABLED_CONFIG); }
  public boolean getOuraEnhancedTagEnabled() { return getBoolean(OURA_ENHANCED_TAG_ENABLED_CONFIG); }

}