import numpy as np
from numpy.testing import assert_allclose

from cloudnetpy.categorize.itu import (
    calc_liquid_specific_attenuation,
    calc_gas_specific_attenuation,
)


def test_calc_liquid_specific_attenuation():
    T = np.array([273.15])
    # Table 1 from Ellis, S. M., and J. Vivekanandan (2011), Liquid water
    # content estimates using simultaneous S and Ka band radar measurements,
    # Radio Sci., 46, RS2021, https://doi.org/10.1029/2010RS004361
    F = [3.0, 6.0, 10.0, 33.5, 92.5]
    desired = np.array([[0.005], [0.015], [0.058], [0.70], [4.09]])
    actual = np.array([calc_liquid_specific_attenuation(T, f) for f in F])
    assert_allclose(actual, desired, rtol=1.25)


def test_calc_gas_specific_attenuation():
    F = np.arange(1.0, 351.0)
    p = np.array([101325.0])
    e = np.array([997.288879])
    T = np.array([288.15])
    # P.676-13 SpAtt in Validation examples for Study Group 3 Earth-space
    # propagation prediction methods, version 8.1.0
    desired = np.array(
        [
            0.00543956278523152,
            0.00692041985305887,
            0.00753875298921938,
            0.0080893032007799,
            0.0087138851774633,
            0.00945878388736656,
            0.0103510016576237,
            0.0114159172990049,
            0.0126844231824067,
            0.0141985419481866,
            0.0160185121242455,
            0.0182336522890195,
            0.0209803221017014,
            0.0244728471310642,
            0.0290583460595937,
            0.035315808093998,
            0.0442344420823219,
            0.0575129248802385,
            0.0779176981867389,
            0.108930855292919,
            0.150432227805704,
            0.187337256302312,
            0.194288975955127,
            0.173161144145986,
            0.146234738793519,
            0.125028120867969,
            0.110735270258498,
            0.101755960206737,
            0.0964937186754364,
            0.0938245472647051,
            0.0930203461858608,
            0.0936194272430361,
            0.0953316414621242,
            0.0979785311930885,
            0.101457329488921,
            0.105719957546205,
            0.110761525716111,
            0.116615248834556,
            0.12335225576667,
            0.131085868720279,
            0.1399808647322,
            0.150269299668818,
            0.162276043004897,
            0.176459879874357,
            0.193481228618503,
            0.214318396186458,
            0.240479522516308,
            0.274425135491184,
            0.320551412199418,
            0.388427185801063,
            0.504720834393432,
            0.737478758828536,
            1.24976526258345,
            2.3389036991214,
            4.32495608469197,
            7.19113553695808,
            10.2058515033922,
            12.4983909037159,
            13.7852796921134,
            14.7783166371223,
            15.1669859823089,
            14.1611666944363,
            11.0013423529745,
            7.02035328306923,
            3.99033835912072,
            2.1541957189258,
            1.22711547353348,
            0.805021005816876,
            0.611818852454113,
            0.513993027511561,
            0.456679762364262,
            0.419049017688118,
            0.393231981169629,
            0.375189614411591,
            0.362554411918302,
            0.3538348995822,
            0.348043514266983,
            0.34450101647835,
            0.34272577911896,
            0.342367616902642,
            0.343166504139791,
            0.344925898112575,
            0.347494984409563,
            0.350756559882441,
            0.354618583493521,
            0.359008175785182,
            0.36386729148043,
            0.369149560401202,
            0.374817961670356,
            0.380843105494605,
            0.38720196911992,
            0.393876982788148,
            0.400855396337025,
            0.408128883038975,
            0.415693358189639,
            0.423549008752367,
            0.431700549648348,
            0.440157744885374,
            0.448936261310874,
            0.45805896475064,
            0.467557831227051,
            0.477476744164629,
            0.487875606814217,
            0.498836461990835,
            0.510472760683938,
            0.522943713831019,
            0.536477107604495,
            0.551406700241543,
            0.568235720026429,
            0.58774914374735,
            0.611221699515333,
            0.640824320197898,
            0.68046764482887,
            0.737669244986783,
            0.827930320462051,
            0.984979824514719,
            1.27706930472684,
            1.74078815446473,
            1.92437252073128,
            1.51596945491459,
            1.15039394993829,
            0.960113467317079,
            0.865748411002288,
            0.817871199418132,
            0.793758165817817,
            0.782753130128758,
            0.779514523011241,
            0.781178360871077,
            0.786115860242935,
            0.793353787245652,
            0.802286601756331,
            0.812525242940002,
            0.823813686148605,
            0.835980892422666,
            0.848912108439287,
            0.862531187339247,
            0.876789421920628,
            0.891658358792947,
            0.907125125942756,
            0.92318939935073,
            0.939861476121271,
            0.957161124852027,
            0.975117008750941,
            0.993766556337142,
            1.01315620719004,
            1.03334199714535,
            1.05439047530375,
            1.07637996856192,
            1.09940223114302,
            1.12356453919825,
            1.14899231617052,
            1.17583240558721,
            1.20425714706218,
            1.23446946212951,
            1.26670922397357,
            1.30126127596722,
            1.33846558788869,
            1.37873020982056,
            1.42254792269842,
            1.47051782201565,
            1.5233735536673,
            1.58202061920162,
            1.64758619171735,
            1.72148640618343,
            1.80551838531848,
            1.90198778294956,
            2.01388811162291,
            2.14515681386993,
            2.3010470606983,
            2.48867730108997,
            2.71785912145068,
            3.00236949098822,
            3.36194627535183,
            3.82548091088421,
            4.43621401491621,
            5.26026865040405,
            6.40052137044415,
            8.01778578387714,
            10.3561737202111,
            13.7410832580785,
            18.4117613559481,
            23.8431542842444,
            27.6777422230024,
            27.0376132539242,
            22.6165322327108,
            17.4937312559993,
            13.3529216847872,
            10.3855261314407,
            8.31933230831711,
            6.87121497588628,
            5.83700012836909,
            5.08208851047049,
            4.51925648245828,
            4.0914968603038,
            3.7608723084514,
            3.50159749002171,
            3.29576733316483,
            3.13069091700555,
            2.99719889503999,
            2.88854803905173,
            2.79969817076414,
            2.72682567145921,
            2.66699015128375,
            2.61790211812598,
            2.57775843464587,
            2.54512403818112,
            2.51884573223143,
            2.49798854037201,
            2.48178815238731,
            2.46961499617093,
            2.46094681004401,
            2.45534750052801,
            2.45245069691071,
            2.45194685027063,
            2.45357303223183,
            2.45710480802237,
            2.46234971641689,
            2.46914200412984,
            2.47733834669684,
            2.48681435049274,
            2.49746167733587,
            2.5091856683935,
            2.52190337087895,
            2.53554189150791,
            2.55003701644998,
            2.56533204973298,
            2.58137683159043,
            2.5981269057218,
            2.6155428103367,
            2.63358947253681,
            2.65223568932367,
            2.67145368151379,
            2.69121870925324,
            2.71150873977454,
            2.73230415962432,
            2.75358752488412,
            2.77534334396776,
            2.79755788845174,
            2.82021902811701,
            2.8433160869789,
            2.866839717581,
            2.89078179124578,
            2.91513530232511,
            2.93989428478941,
            2.96505373974424,
            2.99060957267649,
            3.01655853941455,
            3.04289819994399,
            3.06962687935649,
            3.09674363532853,
            3.12424823163161,
            3.15214111726904,
            3.18042341091896,
            3.20909689044049,
            3.23816398727194,
            3.26762778561838,
            3.29749202639154,
            3.32776111593068,
            3.35844013959842,
            3.38953488041347,
            3.42105184295342,
            3.45299828283686,
            3.48538224217738,
            3.51821259149365,
            3.5514990786627,
            3.58525238561998,
            3.61948419364278,
            3.65420725820725,
            3.68943549458727,
            3.72518407557149,
            3.76146954291939,
            3.79830993446581,
            3.83572492912531,
            3.87373601245586,
            3.91236666592847,
            3.95164258363549,
            3.99159192087608,
            4.03224557991312,
            4.07363753923489,
            4.1158052339247,
            4.15878999629742,
            4.20263756787847,
            4.24739869617106,
            4.2931298326038,
            4.33989395172949,
            4.38776151636407,
            4.43681161918154,
            4.48713333867363,
            4.53882735681706,
            4.59200789790316,
            4.64680506363225,
            4.70336765992607,
            4.76186663756503,
            4.82249930392163,
            4.88549450981745,
            4.95111907820999,
            5.01968582617174,
            5.09156364728988,
            5.16719028100981,
            5.24708861738692,
            5.33188769817017,
            5.42235002024051,
            5.51940738949785,
            5.62420851205511,
            5.73818290212594,
            5.86312778392836,
            6.00132787882699,
            6.15572298171657,
            6.33014620176975,
            6.52966868322059,
            6.76110809653969,
            7.03379464880959,
            7.36075166112684,
            7.76055975531429,
            8.26037325557967,
            8.90090282847922,
            9.74468830173956,
            10.8891849832653,
            12.4827796304828,
            14.7235758828824,
            17.8092420843371,
            21.9879225168979,
            27.6140127915806,
            34.0107349762257,
            37.8922094897409,
            35.9350505931588,
            29.9784111747925,
            23.9031332666525,
            19.2954462516199,
            16.115428329438,
            13.9607931570346,
            12.4889338959205,
            11.4685017298088,
            10.7517003608714,
            10.243974004127,
            9.88269807042919,
            9.62563418586282,
            9.4465976907526,
            9.32925773294452,
            9.26153394742424,
            9.23420684769905,
            9.24052101969451,
            9.27565757437168,
            9.33625037782753,
            9.42002976229003,
            9.52557879600113,
            9.65217200153906,
            9.79967300351,
            9.96847559933702,
            10.1594787665631,
        ]
    )
    desired = desired.reshape(len(desired), 1, 1)
    actual = np.array([calc_gas_specific_attenuation(p + e, e, T, f) for f in F])
    assert_allclose(actual, desired)
