module mod_fourier
implicit none

private
public FFT_init,FFT_close,forwDFT_r2c,backDFT_c2r,forwDFT_c2c,backDFT_c2c,DCT1_2d_c2c

integer, parameter :: dp=selected_real_kind(15,30) 

real(dp), parameter :: pi= 3.14159265358979323846_dp
complex(dp), parameter :: im=(0.0_dp,1.0_dp)

integer Nk,Nw,Nwlog2,Nklog2

integer, allocatable :: btrev(:),btrev_RFT(:),btrev_DCT(:)
complex(dp), allocatable :: expf(:),expf_RFT(:),expf_DCT(:)
real(dp), allocatable :: smf(:),spf(:),c1f(:)

contains

 subroutine FFT_init(Nk_val,Nw_val)
  integer Nk_val,Nw_val

   Nw=Nw_val
   Nwlog2=nint(log10(real(Nw,dp))/log10(2.0_dp))

   Nk=Nk_val
   Nklog2=nint(log10(real(Nk,dp))/log10(2.0_dp))

   allocate(btrev(0:Nw-1),expf(2*Nw),btrev_RFT(0:Nw/2-1),expf_RFT(Nw), &
    btrev_DCT(0:Nk/2-1),expf_DCT(Nk/2),smf(0:Nk/2-1),spf(0:Nk/2-1),c1f(0:Nk/2))

   call prepare_btrev4(Nw,Nwlog2,btrev)
   call prepare_btrev4(Nw/2,Nwlog2-1,btrev_RFT)
   call prepare_btrev4(Nk/2,Nklog2-1,btrev_DCT)

   call prepare_expf(Nw,Nwlog2,expf,flag=1)
   call prepare_expf(Nw/2,Nwlog2-1,expf_RFT,flag=1)
   call prepare_expf(Nk/2,Nklog2-1,expf_DCT,flag=0)

   call prepare_scf(Nk,smf,spf,c1f)

 end subroutine

 subroutine FFT_close

   Nk=0; Nw=0; Nwlog2=0; Nklog2=0

   deallocate(btrev,expf,btrev_RFT,expf_RFT,btrev_DCT,expf_DCT,smf,spf,c1f)

 end subroutine

 subroutine prepare_btrev4(N,Nlog2,btrev)
  integer N,Nlog2,btrev(0:N-1), j,pow2(Nlog2),btpos(Nlog2)

   do j=1,Nlog2
    pow2(j)=2**(j-1)
    btpos(j)=Nlog2-j
   end do

   btpos(Nlog2-1)=0
   btpos(Nlog2)=1

   do j=0,N-1
    btrev(j)=sum(pow2,mask=btest(j,btpos))
   end do

 end subroutine

 subroutine prepare_expf(N,Nlog2,expf,flag)
  integer N,Nlog2,flag,lev,Nlev,k
  complex(dp) expf(1:)

   do lev=1,Nlog2
    Nlev=2**lev
    do k=0,Nlev/2-1
     expf(Nlev/2+k)=exp(-2*pi*im/Nlev *k)
     if (flag==1) expf(N+Nlev/2+k)=exp(+2*pi*im/Nlev *k)
    end do
   end do

 end subroutine

 subroutine prepare_scf(N,smf,spf,c1f)
  integer N,j
  real(dp) smf(0:N/2-1),spf(0:N/2-1),c1f(0:N/2)

   do j=0,N/2-1
    smf(j)=0.5_dp-sin(j*2*pi/N)
    spf(j)=0.5_dp+sin(j*2*pi/N)
   end do

   c1f(0)=+1.0_dp
   do j=1,N/2-1
    c1f(j)=2*cos(j*2*pi/N)
   end do
   c1f(N/2)=-1.0_dp

 end subroutine

 subroutine FFT1d(sgn,NFFT,NFFTlog2,f,c,btrev,expf)
  integer sgn,NFFT,NFFTlog2,btrev(0:NFFT-1)
  complex(dp) f(0:NFFT-1),c(0:NFFT-1),expf(:)
  integer i,lev,Nlev,M,k,indx1,indx2,offs
  complex(dp) p01,m01,p02,p13,m02,im13,A,B

   c=f(btrev)        

   do i=0,NFFT-1,4   
    p02=c(i)+c(i+2)
    m02=c(i)-c(i+2)
    p13=c(i+1)+c(i+3)
    im13=sgn*im*(c(i+1)-c(i+3))
    c(i)=p02+p13
    c(i+1)=m02+im13
    c(i+2)=p02-p13
    c(i+3)=m02-im13
   end do

   Nlev=4
   do lev=3,NFFTlog2 
    M=Nlev 
    Nlev=2*M
    if (sgn==-1) offs=M
    if (sgn==+1) offs=M+NFFT
    do i=0,NFFT-1,Nlev
     do k=0,M-1
      indx1=i+k
      indx2=indx1+M
      A=c(indx1)
      B=c(indx2)*expf(offs+k) 
      c(indx1)=A+B
      c(indx2)=A-B
     end do
    end do
   end do

 end subroutine

 subroutine forwDFT_c2c(f,c)
  complex(dp), dimension(0:Nw-1) :: f,c

   call FFT1d(-1,Nw,Nwlog2,f,c,btrev,expf)

 end subroutine

 subroutine backDFT_c2c(c,f)
  complex(dp), dimension(0:Nw-1) :: c,f

   call FFT1d(+1,Nw,Nwlog2,c,f,btrev,expf)

 end subroutine

 subroutine forwDFT_r2c(f,c)
  real(dp), dimension(0:Nw-1) :: f
  complex(dp), dimension(0:Nw/2) :: c
  complex(dp) aux(0:Nw/2-1),auxFT(0:Nw/2-1),A,B
  integer k

   aux=cmplx(f(0::2),f(1::2),dp)

   call FFT1d(-1,Nw/2,Nwlog2-1,aux,auxFT,btrev_RFT,expf_RFT)

   c(0)=real(auxFT(0),dp)+aimag(auxFT(0))

   do k=1,Nw/2-1
    A=auxFT(k)
    B=conjg(auxFT(Nw/2-k))
    c(k)=( (A+B)-im*expf(Nw/2+k)*(A-B) )/2
   end do

   c(Nw/2)=real(auxFT(0),dp)-aimag(auxFT(0))

 end subroutine

 subroutine backDFT_c2r(c,f)
  complex(dp), dimension(0:Nw/2) :: c
  real(dp), dimension(0:Nw-1) :: f
  complex(dp) aux(0:Nw/2-1),auxFT(0:Nw/2-1),A,B
  integer k

   auxFT(0)=(1,1)*c(0)+(1,-1)*c(Nw/2)

   do k=1,Nw/2-1
    A=c(k)
    B=conjg(c(Nw/2-k))
    auxFT(k)= (A+B)+im*expf(Nw+Nw/2+k)*(A-B) 
   end do

   call FFT1d(+1,Nw/2,Nwlog2-1,auxFT,aux,btrev_RFT,expf_RFT)

   f(0::2)=real(aux,dp)
   f(1::2)=aimag(aux)

 end subroutine

 subroutine DCT1_1d_c2c(f,c)
  complex(dp), dimension(0:Nk/2) :: f,c
  complex(dp) aux(0:Nk/2-1),auxFT(0:Nk/2-1),A,B
  integer k

   aux=smf*f(0:Nk/2-1)+spf*f(Nk/2:1:-1)

   call FFT1d(-1,Nk/2,Nklog2-1,aux,auxFT,btrev_DCT,expf_DCT)

   c(0)=2*auxFT(0)
   c(1)=sum(f*c1f)

   do k=1,Nk/4-1
    A=auxFT(k)
    B=auxFT(Nk/2-k)
    c(2*k)=A+B
    c(2*k+1)=c(2*k-1)+im*(A-B)
   end do

   c(Nk/2)=2*auxFT(Nk/4)

 end subroutine

 subroutine DCT1_2d_c2c(f,c)
  complex(dp), dimension(0:Nk/2,0:Nk/2) :: f,c,aux2
  complex(dp) aux1(0:Nk/2)
  integer j

   do j=0,Nk/2
    aux1=f(j,:)
    call DCT1_1d_c2c(aux1,aux2(:,j))
   end do

   do j=0,Nk/2
    aux1=aux2(j,:)
    call DCT1_1d_c2c(aux1,c(:,j))
   end do

 end subroutine

end module
module mod_stopwatch
implicit none      

integer, parameter :: dp=selected_real_kind(15,30), & 
                      long=selected_int_kind(15)      

integer(long) rtime_vals(10)
real(dp) ctime_vals(10)
integer :: tindx=0

private dp,rtime_vals,ctime_vals,tindx

contains

 subroutine stopwatch(flag,msg,rtime,ctime)
  integer flag,ndigr,ndigc
  character(len=*) msg
  real(dp) rtime,ctime,drtime,dctime
  optional msg,rtime,ctime
  integer(long) rtime2,rate
  character(len=80) timestr

   select case(flag)
   case(1)

    tindx=tindx+1

    call system_clock(rtime_vals(tindx))
    call CPU_time(ctime_vals(tindx))

    if (present(msg)) write(*,fmt="(a)") trim(msg)

   case(2:4)

    call system_clock(rtime2,rate)
    call CPU_time(dctime)

    drtime=(rtime2-rtime_vals(tindx))/real(rate,dp)
    dctime=dctime-ctime_vals(tindx)

    ndigr=max(1,ceiling(log10(drtime)))+6
    ndigc=max(1,ceiling(log10(dctime)))+6

    if (flag==2) &
     write(timestr,fmt="(a,"// &
       "f"//char(48+ndigr/10)//char(48+mod(ndigr,10))//".3,a,"// &
       "f"//char(48+ndigc/10)//char(48+mod(ndigc,10))//".3,a,f7.2)") &
       "   time:",drtime,"s   CPUtime:",dctime,"s   ratio:",dctime/drtime

    if (flag==3) &
     write(timestr,fmt="(3x,f"//char(48+ndigr/10)//char(48+mod(ndigr,10))//".3,a)") drtime,"s"

    if (flag/=4) then
     if (present(msg)) then
      write(*,fmt="(a)") trim(msg)//trim(timestr)
     else
      write(*,fmt="(a)") trim(timestr)
     end if
    end if

    if (present(rtime)) rtime=drtime
    if (present(ctime)) ctime=dctime

    tindx=tindx-1

   end select

 end subroutine

end module

module mod_SCBA
use mod_fourier
use mod_stopwatch
implicit none

integer, parameter :: dp=selected_real_kind(15,30) 

real(dp), parameter :: pi=3.14159265358979323846_dp

complex(dp), parameter :: im=(0,1)

real(dp) :: ET,kappa,Apar,C0par, doping, beta,chempot

integer NBZ,Nw

real(dp), allocatable, dimension(:,:) :: wght,epsk, uq,vq,wq, etaq

real(dp), allocatable :: Earr(:),warr(:),nFarr(:)
real(dp) Emin,Emax,wmax

real(dp) FWHM
integer flag_peak 

real(dp), allocatable :: AkE(:,:,:)
complex(dp), allocatable :: selfE(:,:,:)

contains

 subroutine init_SCBA1
  integer m,n,i1,i2,j
  real(dp) dE

   call FFT_init(NBZ,Nw)

   if (allocated(wght)) deallocate(wght,epsk,etaq,uq,vq,wq,Earr,warr,nFarr,AkE,selfE)

   m=NBZ/2

   allocate(wght(0:m,0:m),epsk(0:m,0:m),etaq(0:m,0:m),uq(0:m,0:m),vq(0:m,0:m),wq(0:m,0:m))

   n=Nw-1

   allocate(Earr(0:n),warr(0:n),nFarr(0:n),AkE(0:m,0:m,0:n),selfE(0:m,0:m,0:n))

   do i1=0,NBZ/2
    do i2=0,NBZ/2
     wght(i1,i2)=4.0
     if (i1==0 .or. i1==NBZ/2) wght(i1,i2)=wght(i1,i2)/2
     if (i2==0 .or. i2==NBZ/2) wght(i1,i2)=wght(i1,i2)/2
    end do
   end do
   wght=wght/real(NBZ,dp)**2

   wmax=(Emax-Emin)/2
   do j=0,Nw-1
    Earr(j)=(Emin*(Nw-j)+Emax*j)/Nw
    warr(j)=j*2*wmax/Nw
    if (j>Nw/2) warr(j)=(j-Nw)*2*wmax/Nw
   end do

   dE=(Emax-Emin)/Nw
   beta=1/dE 

   nFarr=(1-tanh(beta*Earr/2))/2

 end subroutine

 subroutine init_SCBA2
  real(dp) kx,ky,gammak,Aq,Bq,delta
  integer i1,i2,j

   do i1=0,NBZ/2
    kx=2*pi*real(i1)/NBZ
    do i2=0,NBZ/2
     ky=2*pi*real(i2)/NBZ

     gammak=(cos(kx)+cos(ky))/2

     epsk(i1,i2)=C0par*4*gammak 

     Aq=ET+4*kappa*gammak
     Bq=4*kappa*gammak
     wq(i1,i2)=sqrt(max(Aq**2-Bq**2,1e-12_dp))
     uq(i1,i2)=sqrt((Aq/wq(i1,i2)+1)/2)
     vq(i1,i2)=sqrt((Aq/wq(i1,i2)-1)/2)*sign(1.0_dp,Bq)

     etaq(i1,i2)=(cos(kx)-cos(ky))/2
    end do
   end do

   selfE=0

   call adjust_chempot(flag=1)

   do j=0,Nw-1
    select case(flag_peak)
    case(1)
     delta=FWHM/2/sqrt(2*log(2.0))
     AkE(:,:,j)=exp(-(Earr(j)-(epsk-chempot))**2/(2*delta**2))/(sqrt(2*pi)*delta)
    case(2)
     delta=FWHM/2
     AkE(:,:,j)=delta/((Earr(j)-(epsk-chempot))**2+delta**2)/pi
    end select
   end do

 end subroutine

 subroutine close_SCBA

   if (allocated(wght)) deallocate(wght,epsk,etaq,uq,vq,wq,Earr,warr,nFarr,AkE,selfE)

   call FFT_close         

 end subroutine

 subroutine comp_selfE_direct
  integer k1,k2,jE,iq1,iq2,q1,q2,jw,i1,i2,j
  real(dp) delta,val,Ma2,Mb2,Bqw(2)
  complex(dp) selfE1(0:Nw-1),selfE2(0:Nw-1)

   if (flag_peak==1) delta=FWHM/2/sqrt(2*log(2.0_dp))
   if (flag_peak==2) delta=FWHM/2

   do jE=0,Nw-1
    do k1=0,NBZ/2
     do k2=0,NBZ/2

      val=0

      do q1=0,NBZ-1
       do q2=0,NBZ-1

        i1=mod(k1-q1+NBZ,NBZ); i1=min(i1,NBZ-i1)
        i2=mod(k2-q2+NBZ,NBZ); i2=min(i2,NBZ-i2)
        iq1=min(q1,NBZ-q1)
        iq2=min(q2,NBZ-q2)

        Ma2=(etaq(i1,i2)*uq(iq1,iq2)-etaq(k1,k2)*vq(iq1,iq2))**2

        Mb2=(etaq(k1,k2)*uq(iq1,iq2)-etaq(i1,i2)*vq(iq1,iq2))**2

        do jw=0,Nw-1

         j=mod(jE-jw+Nw,Nw)

         if (flag_peak==1) &
          Bqw=exp(-([+1,-1]*warr(jw)-wq(iq1,iq2))**2/(2*delta**2))/(sqrt(2*pi)*delta)
         if (flag_peak==2) &
          Bqw=delta/(([+1,-1]*warr(jw)-wq(iq1,iq2))**2+delta**2)/pi

         val=val + Ma2*Bqw(1)*(1-nFarr(j))*AkE(i1,i2,j) + Mb2*Bqw(2)*nFarr(j)*AkE(i1,i2,j)

        end do
       end do
      end do

      selfE(k1,k2,jE)= -pi*im* 3*(4*Apar)**2 *val* 2*wmax/(real(Nw)*real(NBZ)**2)

     end do
    end do
   end do

   do k1=0,NBZ/2
    do k2=0,NBZ/2
     selfE2=selfE(k1,k2,:)
     selfE1=0
     do jE=0,Nw-1
      do j=0,Nw-1
       if (j/=jE) selfE1(jE)=selfE1(jE) + selfE2(j)/(Earr(jE)-Earr(j))
      end do
     end do
     selfE1=im/pi * 2*wmax/Nw * selfE1
     selfE(k1,k2,:)=selfE1+selfE2
    end do
   end do

 end subroutine

 subroutine comp_selfE_FFT
  real(dp) t,delta,prefac
  integer m,n,i1,i2,jt,jw
  real(dp), allocatable :: auxR(:)
  complex(dp), allocatable :: Akt1(:,:,:),Akt2(:,:,:),Bqt(:,:),P1wKK(:), &
   auxRFT(:),aux1(:,:),aux2(:,:),aux3(:,:),aux(:),auxFT(:)

   if (flag_peak==1) delta=FWHM/2/sqrt(2*log(2.0_dp))
   if (flag_peak==2) delta=FWHM/2

   m=NBZ/2
   n=Nw/2
   allocate(Akt1(0:m,0:m,0:n),Akt2(0:m,0:m,0:n), &
    Bqt(0:m,0:m),P1wKK(0:Nw-1),aux(0:Nw-1),auxFT(0:Nw-1), &
    auxR(0:Nw-1),auxRFT(0:n),aux1(0:m,0:m),aux2(0:m,0:m),aux3(0:m,0:m))

   call stopwatch(1,"real DFT: A(k,E) -> t domain")
   do i2=0,NBZ/2 
    do i1=0,NBZ/2
     auxR=AkE(i1,i2,:)*(1-nFarr)
     call forwDFT_r2c(auxR,auxRFT)
     Akt1(i1,i2,:)=auxRFT

     auxR=AkE(i1,i2,:)*nFarr
     call forwDFT_r2c(auxR,auxRFT)
     Akt2(i1,i2,:)=auxRFT
    end do
   end do
   call stopwatch(2)

   call stopwatch(1,"DCT1 2D: k -> R domain , combine into ImΣ, R -> k domain")
   do jt=0,Nw/2

    t=pi/wmax*jt

    Bqt=cmplx(cos(wq*t),-sin(wq*t),dp) * Nw/(2*wmax)

    call DCT1_2d_c2c(uq**2*Bqt,aux1)
    call DCT1_2d_c2c(etaq**2*Akt1(:,:,jt),aux2)
    call DCT1_2d_c2c(aux1*aux2,aux3)
    selfE(:,:,jt)=aux3

    call DCT1_2d_c2c(uq*vq*Bqt,aux1)
    call DCT1_2d_c2c(etaq*Akt1(:,:,jt),aux2)
    call DCT1_2d_c2c(aux1*aux2,aux3)
    selfE(:,:,jt)=selfE(:,:,jt)-2*etaq*aux3

    call DCT1_2d_c2c(vq**2*Bqt,aux1)
    call DCT1_2d_c2c(Akt1(:,:,jt),aux2)
    call DCT1_2d_c2c(aux1*aux2,aux3)
    selfE(:,:,jt)=selfE(:,:,jt)+etaq**2*aux3

    Bqt=conjg(Bqt)

    call DCT1_2d_c2c(vq**2*Bqt,aux1)
    call DCT1_2d_c2c(etaq**2*Akt2(:,:,jt),aux2)
    call DCT1_2d_c2c(aux1*aux2,aux3)
    selfE(:,:,jt)=selfE(:,:,jt)+aux3

    call DCT1_2d_c2c(uq*vq*Bqt,aux1)
    call DCT1_2d_c2c(etaq*Akt2(:,:,jt),aux2)
    call DCT1_2d_c2c(aux1*aux2,aux3)
    selfE(:,:,jt)=selfE(:,:,jt)-2*etaq*aux3

    call DCT1_2d_c2c(uq**2*Bqt,aux1)
    call DCT1_2d_c2c(Akt2(:,:,jt),aux2)
    call DCT1_2d_c2c(aux1*aux2,aux3)
    selfE(:,:,jt)=selfE(:,:,jt)+etaq**2*aux3

    prefac=-pi* 3*(4*Apar)**2* (2*wmax)/real(Nw,dp)**2/real(NBZ,dp)**4

    if (flag_peak==1) prefac=prefac*exp(-(delta*t)**2/2)
    if (flag_peak==2) prefac=prefac*exp(-delta*t)

    selfE(:,:,jt)=prefac*selfE(:,:,jt)

   end do
   call stopwatch(2)

   auxR(0)=0
   auxR(Nw/2)=0
   do jw=1,Nw/2-1
    auxR(jw)=-(1/pi)/warr(jw)
    auxR(Nw-jw)=-(1/pi)/warr(Nw-jw)
   end do
   aux=auxR*2*wmax/Nw
   call forwDFT_c2c(aux,P1wKK)

   call stopwatch(1,"complex DFT: ImΣ(k,t) + KK -> E domain")
   do i2=0,NBZ/2 
    do i1=0,NBZ/2
     auxFT(0:Nw/2)=selfE(i1,i2,0:Nw/2)
     auxFT(Nw/2+1:Nw-1)=conjg(auxFT(Nw/2-1:1:-1))
     auxFT=(im+P1wKK)*auxFT
     call backDFT_c2c(auxFT,aux)
     selfE(i1,i2,:)=aux
    end do
   end do
   call stopwatch(2)

 end subroutine

 subroutine comp_AkE
  integer jE

   call stopwatch(1,"spectral function construction")
   do jE=0,Nw-1
    AkE(:,:,jE)=-aimag(1/(Earr(jE)+im*FWHM/2-(epsk-chempot)-selfE(:,:,jE)))/pi
   end do
   call stopwatch(2)

 end subroutine

 subroutine adjust_chempot(flag)
  integer flag,i
  real(dp) chp_min,chp_max,chp,nf,dnfdchp

   select case(flag)
   case(1)
    chp_min=2*minval(epsk)
    chp_max=2*maxval(epsk)
    write(*,fmt="(a,2f12.6,a,g15.6)") "adjusting chempot in the range",chp_min,chp_max," beta:",beta
    call solve_Ridders(fun,chp_min,chp_max,chp,1e-8_dp,100)
    chempot=chp
   case(2)
    call comp_occ(chempot,beta,nf,dnfdchp)
    chp=chempot+(doping-nf)/dnfdchp
    chempot=chp
   end select

 contains

  subroutine comp_occ(chempot,beta,nf,dnfdchp)
   real(dp) chempot,beta,nf,dnfdchp,fac
   optional dnfdchp
   real(dp), allocatable :: aux1(:),aux2(:),denR(:,:),denI(:,:)
   integer jE

    allocate(aux1(0:Nw-1),aux2(0:Nw-1),denR(0:NBZ/2,0:NBZ/2),denI(0:NBZ/2,0:NBZ/2))

    do jE=0,Nw-1
     denR=Earr(jE)-(epsk-chempot)-real(selfE(:,:,jE),dp)
     denI=FWHM/10-aimag(selfE(:,:,jE))
     fac=(1-tanh(beta*Earr(jE)/2))/pi
     aux1(jE)=fac*sum(wght*denI/(denR**2+denI**2))
     if (present(dnfdchp)) aux2(jE)=fac*sum(wght*denR*denI/(denR**2+denI**2)**2)
    end do

    nf=sum(aux1)*(Emax-Emin)/Nw
    if (present(dnfdchp)) dnfdchp=-2*sum(aux2)*(Emax-Emin)/Nw

  end subroutine

  function fun(x)
   real(dp) x,fun,nf

    call comp_occ(x,beta,nf)

    fun=nf-doping

  end function

  subroutine solve_Ridders(f,a,b,x,xacc,maxiter)
   real(dp) f,a,b,x,xacc,x1,x2,x3,f1,f2,f3,fx,xold
   integer maxiter,iter

    x1=a
    f1=f(a)
    x2=b
    f2=f(b)
    xold=1e25_dp
    do iter=1,maxiter
      x3=(x1+x2)/2
      f3=f(x3)
      x=x3+(x3-x1)*sign(1.0_dp,f1-f2)*f3/sqrt(f3**2-f1*f2)
      fx=f(x)
      if (fx==0.0_dp .or. abs(x-xold)<=xacc) exit
      if (f3*fx<0) then
        x1=x3
        f1=f3
        x2=x
        f2=fx
      else if (fx*f1<0) then
        x2=x
        f2=fx
      else
        x1=x
        f1=fx
      end if
      xold=x
    end do

  end subroutine

 end subroutine

end module

program prg_SCBA
use mod_SCBA
implicit none
integer task

  write(*,fmt="(a)") "task: 1 - SCBA calculation for a given parameter point"
  write(*,fmt="(a)") "      2 - compare direct and FFT evaluation"
  write(*,fmt="(a)") "      3 - sample parameter point for testing"
  read(*,*) task

  select case(task)
  case(1)
   call comp_SCBA_input
  end select

contains

 subroutine init_BZpath(NBZ,Npt,kpath_len,kpath_indx)
  integer NBZ,Npt,j
  real(dp), allocatable :: kpath_len(:)
  integer, allocatable :: kpath_indx(:,:)

   if (allocated(kpath_len)) deallocate(kpath_len,kpath_indx)

   Npt=3*NBZ/2+1
   allocate(kpath_len(Npt),kpath_indx(2,Npt))

   do j=0,NBZ/2-1
    kpath_indx(:,1+j)=[j,0]
    kpath_indx(:,1+NBZ/2+j)=[NBZ/2,j]
    kpath_indx(:,1+NBZ+j)=NBZ/2-j
   end do
   kpath_indx(:,1+3*NBZ/2)=0

   kpath_len(1)=0
   do j=2,Npt
    kpath_len(j)=kpath_len(j-1) &
     +sqrt( (2*pi/NBZ)**2 * sum( (kpath_indx(:,j)-kpath_indx(:,j-1))**2 ) )
   end do

 end subroutine

 subroutine save_BZquant(fname)
  character(len=*) fname
  integer i1,i2,Npt,pt
  real(dp) kx,ky
  real(dp), allocatable :: klen(:)
  integer, allocatable :: indx(:,:)

   call init_BZpath(NBZ,Npt,klen,indx)

   open(1,file=fname//"_map")
   write(1,fmt="(a)") "# 1:kx 2:ky 3:u 4:v 5:w 6:eps"
   do i1=0,NBZ/2
    kx=2*pi*real(i1)/NBZ
    do i2=0,NBZ/2
     ky=2*pi*real(i2)/NBZ
     write(1,*) kx,ky,uq(i1,i2),vq(i1,i2),wq(i1,i2),epsk(i1,i2)
    end do
   end do
   close(1)

   open(1,file=fname//"_path")
   write(1,fmt="(a)") "# 1:kpath 2:u 3:v 4:w 5:eps"
   do pt=1,Npt
    i1=indx(1,pt)
    i2=indx(2,pt)
    write(1,*) klen(pt),uq(i1,i2),vq(i1,i2),wq(i1,i2),epsk(i1,i2)
   end do
   close(1)

 end subroutine

 subroutine save_cuts(fname,Emin_save,Emax_save,step)
  character(len=*) fname
  real(dp), optional :: Emin_save,Emax_save
  integer, optional :: step
  integer jEmin,jEmax,jstep,j,jk,k1,k2
  real(dp) kpath

   jEmin=0
   if (present(Emin_save)) jEmin=sum(minloc(abs(Earr-Emin_save)))-1

   jEmax=Nw-1
   if (present(Emax_save)) jEmax=sum(minloc(abs(Earr-Emax_save)))-1

   jstep=1
   if (present(step)) jstep=step

   open(11,file=fname//"_G")  
   open(12,file=fname//"_X1") 
   open(13,file=fname//"_X2") 
   open(14,file=fname//"_M")  
   open(15,file=fname//"_D1") 
   open(16,file=fname//"_D2") 
   open(17,file=fname//"_Z1") 
   open(18,file=fname//"_Z2") 
   open(19,file=fname//"_S")  
   do jk=11,19
    write(jk,fmt="(a)") "# 1:kpath 2:E 3:A 4:ReΣ 5:ImΣ"
   end do

   do j=jEmin,jEmax,jstep
    do jk=1,9
     select case(jk)
     case(1); k1=0    ; k2=0    ; kpath=0                   
     case(2); k1=NBZ/2; k2=0    ; kpath=pi                  
     case(3); k1=0    ; k2=NBZ/2; kpath=pi                  
     case(4); k1=NBZ/2; k2=NBZ/2; kpath=2*pi                
     case(5); k1=NBZ/4; k2=0    ; kpath=pi/2                
     case(6); k1=0    ; k2=NBZ/4; kpath=pi/2                
     case(7); k1=NBZ/2; k2=NBZ/4; kpath=3*pi/2              
     case(8); k1=NBZ/4; k2=NBZ/2; kpath=3*pi/2              
     case(9); k1=NBZ/4; k2=NBZ/4; kpath=(2+sqrt(0.5_dp))*pi 
     end select

     write(10+jk,fmt="(5g15.6)") kpath, Earr(j), AkE(k1,k2,j), selfE(k1,k2,j)
    end do
   end do

   do jk=11,19
    close(jk)
   end do

 end subroutine

 subroutine save_maps(fname,Emin_save,Emax_save,stepE,stepk)
  character(len=*) fname       
  real(dp), optional :: Emin_save,Emax_save
  integer, optional :: stepE,stepk
  integer jEmin,jEmax,jEstep,jkstep,part,jk,jE,Npt,pt,i1,i2
  real(dp), allocatable :: klen(:)
  integer, allocatable :: indx(:,:)

   jEmin=0
   if (present(Emin_save)) jEmin=sum(minloc(abs(Earr-Emin_save)))-1

   jEmax=Nw-1
   if (present(Emax_save)) jEmax=sum(minloc(abs(Earr-Emax_save)))-1

   jEstep=1
   if (present(stepE)) jEstep=stepE

   jkstep=1
   if (present(stepk)) jkstep=stepk

   call init_BZpath(NBZ,Npt,klen,indx)

   do part=1,3
    open(1,file=fname//"_p"//char(48+part))
    write(1,fmt="(a)") "# 1:kpath 2:E 3:A 4:ReΣ 5:ImΣ"
    do jk=0,NBZ/2,jkstep
     pt=1+(part-1)*NBZ/2+jk
     i1=indx(1,pt)
     i2=indx(2,pt)
     do jE=jEmin,jEmax,jEstep
      write(1,fmt="(5g15.6)") klen(pt),Earr(jE),AkE(i1,i2,jE),selfE(i1,i2,jE)
     end do
     write(1,*)
     write(1,*)
    end do
    close(1)
   end do

 end subroutine

 subroutine save_mapFS(fname)
  character(len=*) fname
  integer i1,i2,jE
  real(dp) kx,ky

   jE=sum(minloc(abs(Earr)))

   open(1,file=fname)
   write(1,fmt="(a)") "# 1:kx 2:ky 3:A(k,E=0)"
   write(1,fmt="(a,f15.8)") "# actual E value = ",Earr(jE)
   do i1=0,NBZ/2
    kx=2*pi*real(i1)/NBZ
    do i2=0,NBZ/2
     ky=2*pi*real(i2)/NBZ
     write(1,*) kx,ky,AkE(i1,i2,jE)
    end do
   end do
   close(1)

 end subroutine

 subroutine save_DOS(fname,Emin_save,Emax_save,step)
  character(len=*) fname
  real(dp), optional :: Emin_save,Emax_save
  integer, optional :: step
  integer jEmin,jEmax,jstep,j

   jEmin=0
   if (present(Emin_save)) jEmin=sum(minloc(abs(Earr-Emin_save)))-1

   jEmax=Nw-1
   if (present(Emax_save)) jEmax=sum(minloc(abs(Earr-Emax_save)))-1

   jstep=1
   if (present(step)) jstep=step

   open(1,file=fname)
   write(1,fmt="(a)") "# 1:E 2:DOS"
   do j=jEmin,jEmax,jstep
    write(1,fmt="(2g15.6)") Earr(j),sum(wght*AkE(:,:,j))
   end do
   close(1)

 end subroutine

 subroutine comp_SCBA_input
  integer Niter,iter,flag_pars,flag_out,jEstep_cuts,jEstep_maps,jkstep_maps
  real(dp) Ud,JH,Delta,thop,the,E1,E2,E3,FWHM_iter,FWHM_final,Emin_save,Emax_save
  character(len=80) fname

   write(*,fmt="(a)") "model parameters (1 - set by hand, 2 - given by microscopic):"
   read(*,*) flag_pars

   select case(flag_pars)
   case(1)
    write(*,fmt="(a)") "ET, kappa, A, C0:"
    read(*,*) ET,kappa,Apar,C0par

   case(2)
    write(*,fmt="(a)") "U,JH,Δ,t:"
    read(*,*) Ud,JH,Delta,thop

    ET=sqrt(Delta**2+JH**2)-3*JH

    the=atan(JH/Delta)/2

    E1=Ud+JH-Delta+2*ET
    E2=Ud+JH      +2*ET
    E3=Ud+JH+Delta+2*ET

    kappa=thop**2*( (1+cos(2*the))/E1 + 2*sin(2*the)/E2 + (1-cos(2*the))/E3 )/3

    C0par=thop/3*cos(the)**2

    Apar=thop/sqrt(6.0_dp)*cos(the)

    write(*,fmt="(a,2f15.8)") "ET,κ:",ET,kappa
    write(*,fmt="(a,2f15.8)") "A,C0:",Apar,C0par

    if (ET<8*kappa) stop
   end select

   write(*,fmt="(a)") "doping:"
   read(*,*) doping

   write(*,fmt="(a)") "NBZ, Nw:"
   read(*,*) NBZ,Nw

   write(*,fmt="(a)") "Emin, Emax, FWHM (iterations), FWHM (final), peak type (1-gaussian,2-lorentzian):"
   read(*,*) Emin,Emax,FWHM_iter,FWHM_final,flag_peak

   write(*,fmt="(a)") "number of iterations:"
   read(*,*) Niter

   write(*,fmt="(a)") "what to save:"
   write(*,fmt="(a)") "  +1 cuts at symmetry k-points"
   write(*,fmt="(a)") "  +2 (k,E) maps"
   write(*,fmt="(a)") "  +4 (kx,ky) FS map"
   write(*,fmt="(a)") "  +8 density of states"
   write(*,fmt="(a)") " +16 T dispersions and Bogoliubov factors"
   read(*,*) flag_out

   write(*,fmt="(a)") "Emin_save, Emax_save, Estep for cuts, Estep for maps, kstep for maps:"
   read(*,*) Emin_save,Emax_save,jEstep_cuts,jEstep_maps,jkstep_maps

   write(*,fmt="(a)") "file name:"
   read(*,*) fname

   FWHM=FWHM_iter
   call init_SCBA1
   call init_SCBA2

   do iter=1,Niter
    write(*,fmt="(/a,i2)") "iteration ",iter
    call stopwatch(1)

    call comp_selfE_fft

    if (iter<=2) call adjust_chempot(1)
    if (iter>=3) call adjust_chempot(2)

    call comp_AkE

    call stopwatch(2)
   end do

   FWHM=FWHM_final
   call comp_AkE

   if (mod(flag_out/1,2)==1) call save_cuts(trim(fname)//"_cut",Emin_save,Emax_save,jEstep_cuts)
   if (mod(flag_out/2,2)==1) call save_maps(trim(fname)//"_map",Emin_save,Emax_save,jEstep_maps)
   if (mod(flag_out/4,2)==1) call save_mapFS(trim(fname)//"_mapFS")
   if (mod(flag_out/8,2)==1) call save_DOS(trim(fname)//"_dos",Emin_save,Emax_save,jEstep_cuts)
   if (mod(flag_out/16,2)==1) call save_BZquant(trim(fname)//"_BZquant")

   call close_SCBA

 end subroutine

end program
