module mod_BCSeq
implicit none        
integer, parameter :: dp=selected_real_kind(15,30)

private
public BCSeq_bW_to_lambda, BCSeq_lambda_to_bW

contains

 function quad_Romberg(f,a,b,levmin,levmax,kmax,acc,N) result(val)
  integer levmin,levmax,kmax,lev,N,j,k
  real(dp) f,a,b,acc,sum1,sum2,h,T(0:levmax,0:kmax),val,prev

   sum1=(f(a)+f(b))/2
   sum2=0.0_dp
   do lev=0,levmax

     N=2**lev
     h=(b-a)/N
     do j=1,N-1,2
       sum2=sum2+f(a+h*j)
     end do
     T(lev,0)=h*(sum1+sum2)

     if (lev>0) then
      do k=1,min(lev,kmax)
        T(lev,k)=(4**k*T(lev,k-1)-T(lev-1,k-1))/(4**k-1)
      end do
     end if
     val=T(lev,min(lev,kmax))

     if (lev>=levmin) then
       if (abs(val-prev)<acc) exit
     end if
     prev=val
   end do

 end function

 subroutine solve_Ridders(f,a,b,x,xacc,maxiter)
  real(dp) f,a,b,x,xacc,x1,x2,x3,f1,f2,f3,fx,xold
  integer maxiter,iter

   x1=a
   f1=f(a)
   x2=b
   f2=f(b)
   xold=1e100_dp
   do iter=1,maxiter
     x3=(x1+x2)/2
     f3=f(x3)
     x=x3+(x3-x1)*sign(1.0_dp,f1-f2)*f3/sqrt(f3**2-f1*f2)
     fx=f(x)
     if (fx==0.0_dp .or. abs(x-xold)<=xacc) exit
     if (f3*fx<0) then
       x1=x3
       f1=f3
       x2=x
       f2=fx
     else if (fx*f1<0) then
       x2=x
       f2=fx
     else
       x1=x
       f1=fx
     end if
     xold=x
   end do

 end subroutine

 function eval_int_tanhxx(bW2)
  real(dp) bW2,eval_int_tanhxx
  integer N

   if (bW2<1) then
    eval_int_tanhxx=bW2-quad_Romberg(faux1,0.0_dp,bW2,3,12,5,1e-12_dp,N)
   else
    eval_int_tanhxx= 1-quad_Romberg(faux1,0.0_dp,1.0_dp,3,12,5,1e-12_dp,N) &
     + log(bW2) - quad_Romberg(faux2,0.0_dp,min(log(bW2),4.0_dp),3,12,5,1e-12_dp,N)
   end if

 contains

  function faux1(x)
   real(dp) x,faux1,x2

    if (x>1e-3) then
     faux1=1-tanh(x)/x
    else
     x2=x**2           
     faux1=((17*x2/315-2.0_dp/15)*x2+1.0_dp/3)*x2
    end if

  end function

  function faux2(y) 
   real(dp) y,faux2

    faux2=2/(exp(2*exp(y))+1)

  end function

 end function

 function BCSeq_bW_to_lambda(bW) result(lambda)
  real(dp) bW,lambda

   lambda=1/eval_int_tanhxx(bW/2)

 end function

 function BCSeq_lambda_to_bW(lambda) result(bW)
  real(dp) lambda,bW,lnbW

   if (lambda>1.0_dp) then

    call solve_Ridders(faux1,0.0_dp,2.5_dp,bW,1e-12_dp,maxiter=50)

   else if (lambda>0.05_dp) then

    call solve_Ridders(faux2,log(1.0_dp),log(1e9_dp),lnbW,1e-12_dp,maxiter=50)
    bW=exp(lnbW)

   else

    bW=2*exp(1/lambda-0.8187801401720234_dp)

   end if

 contains

  function faux1(bW)
   real(dp) bW,faux1

    faux1=1/lambda-eval_int_tanhxx(bW/2)

  end function

  function faux2(lnbW)
   real(dp) lnbW,faux2

    faux2=lambda-1/eval_int_tanhxx(exp(lnbW)/2)

  end function

 end function

end module
module mod_BCS_int
implicit none

integer, parameter :: dp=selected_real_kind(15,30)

real(dp), parameter :: pi=3.141592653589793238_dp

integer, parameter :: N=3

type exp_table
 real(dp), dimension( -2*N:+2*N,-2*N:+2*N, -2*N:+2*N,-2*N:+2*N ) :: coef
end type

type cos_table
 real(dp), dimension( 0:+2*N,0:+2*N, 0:+2*N,0:+2*N ) :: coef
end type

type(exp_table) :: expfun(1:4),expfun34,expfun234 

type(cos_table) :: cosfun(1:4),cosfun34,cosfun234 

contains

 subroutine collect_exp_sums
  integer ix,iy,jx,jy,kx,ky,lx,ly, i,j,k,l, sgnij,sgnkl, t, cnt(4)
  logical term(4)

   write(*,fmt="(a)") "collecting exponentials..."

   cnt=0

   do t=1,4
    expfun(t)%coef=0
   end do

   ix=0
   iy=0
   i=(2*N+1)*(ix+N)+iy+N+1

   do jx=-N,+N
    do jy=-N,+N

     j=(2*N+1)*(jx+N)+jy+N+1

     do kx=-N,+N
      do ky=-N,+N

       k=(2*N+1)*(kx+N)+ky+N+1

       do lx=-N,+N
        do ly=-N,+N

         l=(2*N+1)*(lx+N)+ly+N+1

         if (i==k .or. j==l) cycle 

         if ( (i==j .and. k==l) .or. (i==l .and. k==j) ) cycle

         sgnij=0

         if (abs(jx-ix)==1 .and. abs(jy-iy)==0) sgnij=+1 
         if (abs(jx-ix)==0 .and. abs(jy-iy)==1) sgnij=-1 

         sgnkl=0

         if (abs(kx-lx)==1 .and. abs(ky-ly)==0) sgnkl=+1 
         if (abs(kx-lx)==0 .and. abs(ky-ly)==1) sgnkl=-1 

         if (sgnij*sgnkl==0) cycle

         term=.false.

         if (i==l) term(1)=.true.

         if (abs(ix-lx)+abs(iy-ly)==1) term(2)=.true.

         if (abs(ix-kx)+abs(iy-ky)==1 .and. j/=k .and. l/=i) term(3)=.true.

         if (abs(jx-lx)+abs(jy-ly)==1 .and. j/=k .and. l/=i) term(4)=.true.

         do t=1,4
          if (term(t)) then
           cnt(t)=cnt(t)+1
           expfun(t)%coef(ix-kx,iy-ky,jx-lx,jy-ly)=expfun(t)%coef(ix-kx,iy-ky,jx-lx,jy-ly)+sgnij*sgnkl
          end if
         end do

        end do
       end do
      end do
     end do
    end do
   end do

   write(*,fmt="(a,i3,a)") "δ_il    :",cnt(1)," configurations"
   write(*,fmt="(a,i3,a)") "δ^NN_il :",cnt(2)," configurations"
   write(*,fmt="(a,i3,a)") "δ^NN_ik :",cnt(3)," configurations"
   write(*,fmt="(a,i3,a)") "δ^NN_jl :",cnt(4)," configurations"
   write(*,*)

 end subroutine

 subroutine symmetrize
  integer t,m1,m2,m3,m4,sgn1,sgn2,sgn3,sgn4
  type(exp_table) expfun_sym(1:4)
  real(dp) evals(0:1,0:1,0:1,0:1)

   do t=1,4

    do m1=-2*N,+2*N
     do m2=-2*N,+2*N
      do m3=-2*N,+2*N
       do m4=-2*N,+2*N
        expfun_sym(t)%coef(m1,m2,m3,m4)=    &
         ( expfun(t)%coef(+m1,+m2,+m3,+m4)+ &
           expfun(t)%coef(+m1,+m2,-m3,-m4)+ &
           expfun(t)%coef(-m1,-m2,+m3,+m4)+ &
           expfun(t)%coef(-m1,-m2,-m3,-m4)  )/2
       end do
      end do
     end do
    end do

   end do

   expfun=expfun_sym

   expfun34%coef=expfun(3)%coef+expfun(4)%coef
   expfun234%coef=expfun(2)%coef+expfun(3)%coef+expfun(4)%coef

   do t=1,4

    do m1=0,+2*N
     do m2=0,+2*N
      do m3=0,+2*N
       do m4=0,+2*N

        do sgn1=0,1
         do sgn2=0,1
          do sgn3=0,1
           do sgn4=0,1
            evals(sgn1,sgn2,sgn3,sgn4)= &
             expfun_sym(t)%coef( (2*sgn1-1)*m1, (2*sgn2-1)*m2, (2*sgn3-1)*m3, (2*sgn4-1)*m4 )
           end do
          end do
         end do
        end do

        if (any(abs( evals-sum(evals)/16 )>1e-12)) then
         write(*,*) "ERROR - asymmetry among coefficients"
         stop
        end if

        cosfun(t)%coef(m1,m2,m3,m4) = 2**count([m1,m2,m3,m4]/=0) * evals(0,0,0,0)

       end do
      end do
     end do
    end do

   end do

   cosfun34%coef=cosfun(3)%coef+cosfun(4)%coef
   cosfun234%coef=cosfun(2)%coef+cosfun(3)%coef+cosfun(4)%coef

 end subroutine

 subroutine print_nonzero_exp(expfun)
  type(exp_table) expfun,expfun_aux
  integer m1,m2,m3,m4,mvec(4)
  real(dp) val

   expfun_aux=expfun

   do
    mvec=maxloc(abs(expfun_aux%coef)+1e-6*expfun_aux%coef)-1-2*N
    val=expfun_aux%coef(mvec(1),mvec(2),mvec(3),mvec(4))
    if (abs(val)<1e-6) exit
    write(*,fmt="(4i3,3x,f5.1)") mvec,val
    expfun_aux%coef(mvec(1),mvec(2),mvec(3),mvec(4))=0
   end do

 end subroutine

 subroutine print_nonzero_cos(cosfun)
  type(cos_table) cosfun,cosfun_aux
  integer m1,m2,m3,m4,mvec(4)
  real(dp) val

   cosfun_aux=cosfun

   do
    mvec=maxloc(abs(cosfun_aux%coef)+1e-6*cosfun_aux%coef)-1
    val=cosfun_aux%coef(mvec(1),mvec(2),mvec(3),mvec(4))
    if (abs(val)<1e-6) exit
    write(*,fmt="(4i3,3x,f5.1)") mvec,val
    cosfun_aux%coef(mvec(1),mvec(2),mvec(3),mvec(4))=0
   end do

 end subroutine

 function eval_expfun(expfun,Kx,Ky,Kxp,Kyp) result(val)
  type(exp_table) expfun
  real(dp) Kx,Ky,Kxp,Kyp,val,coef
  complex(dp) cval
  integer m1,m2,m3,m4

   cval=0

   do m1=-2*N,+2*N
    do m2=-2*N,+2*N
     do m3=-2*N,+2*N
      do m4=-2*N,+2*N

       coef=expfun%coef(m1,m2,m3,m4)

       if (coef/=0) cval=cval+coef*exp((0,1)*(Kx*m1+Ky*m2+Kxp*m3+Kyp*m4))

      end do
     end do
    end do
   end do

   val=cval

 end function         

 function eval_cosfun(cosfun,Kx,Ky,Kxp,Kyp) result(val)
  type(cos_table) cosfun
  real(dp) Kx,Ky,Kxp,Kyp,val,coef
  integer m1,m2,m3,m4

   val=0

   do m1=0,+2*N
    do m2=0,+2*N
     do m3=0,+2*N
      do m4=0,+2*N

       coef=cosfun%coef(m1,m2,m3,m4)

       if (coef/=0) val=val+coef*cos(Kx*m1)*cos(Ky*m2)*cos(Kxp*m3)*cos(Kyp*m4)

      end do
     end do
    end do
   end do

 end function         

 subroutine eval_pairing_formfac(Kx,Ky,Kxp,Kyp,F1,F2,F3,F1s,F2s,F3s,F1d,F2d,F3d)
  real(dp) Kx,Ky,Kxp,Kyp,F1,F2,F3, &
   cx,c2x,c3x,cy,c2y,c3y,cxp,c2xp,c3xp,cyp,c2yp,c3yp, &
   s1,s1p,s2,s2p,s3,s3p,s4,s4p,s5,s5p,d1,d1p,d2,d2p,d3,d3p,d4,d4p
  real(dp), optional :: F1s,F2s,F3s,F1d,F2d,F3d
  real(dp), parameter :: sqrt41=sqrt(41.0_dp)

   cx=cos(Kx)
   c2x=cos(2*Kx)
   c3x=cos(3*Kx)

   cy=cos(Ky)
   c2y=cos(2*Ky)
   c3y=cos(3*Ky)

   cxp=cos(Kxp)
   c2xp=cos(2*Kxp)
   c3xp=cos(3*Kxp)

   cyp=cos(Kyp)
   c2yp=cos(2*Kyp)
   c3yp=cos(3*Kyp)

   s1=cx+cy
   s1p=cxp+cyp

   s2=c2x+c2y
   s2p=c2xp+c2yp

   s3=c3x+c3y
   s3p=c3xp+c3yp

   s4=cx*c2y+c2x*cy
   s4p=cxp*c2yp+c2xp*cyp

   s5=cx*cy
   s5p=cxp*cyp

   d1=cx-cy
   d1p=cxp-cyp

   d2=c2x-c2y
   d2p=c2xp-c2yp

   d3=c3x-c3y
   d3p=c3xp-c3yp

   d4=cx*c2y-c2x*cy
   d4p=cxp*c2yp-c2xp*cyp

   F1=6*d1*d1p-2*s1*s1p

   F2=32*s5*s5p-8*(s2*s5p+s5*s2p)+2*(s2*s2p+d2*d2p)

   F3=8*(d1*d1p+s1*s1p)-4*(s1*s4p+s4*s1p)+12*(d1*d4p+d4*d1p)+2*(s1*s3p+s3*s1p+d1*d3p+d3*d1p)

   if (present(F1s)) F1s=-2*s1*s1p
   if (present(F1d)) F1d=+6*d1*d1p

   if (present(F2s)) F2s=32*s5*s5p-8*(s2*s5p+s5*s2p)+2*s2*s2p
   if (present(F2d)) F2d=2*d2*d2p

   if (present(F3s)) F3s=8*s1*s1p-4*(s1*s4p+s4*s1p)+2*(s1*s3p+s3*s1p)
   if (present(F3d)) F3d=8*d1*d1p+12*(d1*d4p+d4*d1p)+2*(d1*d3p+d3*d1p)

 end subroutine

 subroutine VKK_avg_simple(doping,Nphi,V0,V1a,V1b)
  real(dp) doping,V0,V1a,V1b, kF,phi,phip,Kx,Ky,Kxp,Kyp,F1,F2,F3,F1s,F2s,F3s,F1s_avg,F2s_avg,F3s_avg
  integer Nphi,i,j

   kF=sqrt(2*pi*doping)

   F1s_avg=0
   F2s_avg=0
   F3s_avg=0

   do i=1,Nphi
    do j=1,Nphi

     phi=real(i-1,dp)*2*pi/Nphi
     phip=real(j-1,dp)*2*pi/Nphi

     Kx=pi+kF*cos(phi)
     Ky=pi+kF*sin(phi)
     Kxp=pi+kF*cos(phip)
     Kyp=pi+kF*sin(phip)

     call eval_pairing_formfac(Kx,Ky,Kxp,Kyp,F1,F2,F3,F1s,F2s,F3s)

     F1s_avg=F1s_avg+F1s
     F2s_avg=F2s_avg+F2s
     F3s_avg=F3s_avg+F3s
    end do
   end do

   F1s_avg=F1s_avg/Nphi**2
   F2s_avg=F2s_avg/Nphi**2
   F3s_avg=F3s_avg/Nphi**2

   V0=+F1s_avg    
   V1a=-F2s_avg   
   V1b=-F3s_avg/2 

 end subroutine

end module

program prg_BCS_int
use mod_BCS_int        
use mod_BCSeq
implicit none

 call task_map_BCS_lambda(0.10_dp, 1.7_dp,2.2_dp, 200, 0.01_dp,0.6_dp, 200, 5.0_dp, 0.6_dp, "xBCS_nf0.10_map")
 call task_map_BCS_lambda(0.15_dp, 1.7_dp,2.2_dp, 200, 0.01_dp,0.6_dp, 200, 5.0_dp, 0.6_dp, "xBCS_nf0.15_map")
 call task_map_BCS_lambda(0.20_dp, 1.7_dp,2.2_dp, 200, 0.01_dp,0.6_dp, 200, 5.0_dp, 0.6_dp, "xBCS_nf0.20_map")

 call task_map_BCS_lambda(0.10_dp, 1.7_dp,2.7_dp, 500, 0.3_dp,0.3_dp, 1, 5.0_dp, 0.6_dp, "xBCS_nf0.10_t0.3cut")
 call task_map_BCS_lambda(0.15_dp, 1.7_dp,2.7_dp, 500, 0.3_dp,0.3_dp, 1, 5.0_dp, 0.6_dp, "xBCS_nf0.15_t0.3cut")
 call task_map_BCS_lambda(0.20_dp, 1.7_dp,2.7_dp, 500, 0.3_dp,0.3_dp, 1, 5.0_dp, 0.6_dp, "xBCS_nf0.20_t0.3cut")
 call task_map_BCS_lambda(0.25_dp, 1.7_dp,2.7_dp, 500, 0.3_dp,0.3_dp, 1, 5.0_dp, 0.6_dp, "xBCS_nf0.25_t0.3cut")
 call task_map_BCS_lambda(0.30_dp, 1.7_dp,2.7_dp, 500, 0.3_dp,0.3_dp, 1, 5.0_dp, 0.6_dp, "xBCS_nf0.30_t0.3cut")
 call task_map_BCS_lambda(0.35_dp, 1.7_dp,2.7_dp, 500, 0.3_dp,0.3_dp, 1, 5.0_dp, 0.6_dp, "xBCS_nf0.35_t0.3cut")
 call task_map_BCS_lambda(0.40_dp, 1.7_dp,2.7_dp, 500, 0.3_dp,0.3_dp, 1, 5.0_dp, 0.6_dp, "xBCS_nf0.40_t0.3cut")

contains

 subroutine task_print_exp

  write(*,fmt="(a)") "δ_il term:"
  call print_nonzero_exp(expfun(1))

  write(*,fmt="(/a)") "δ^NN_il term:"
  call print_nonzero_exp(expfun(2))

  write(*,fmt="(/a)") "δ^NN_ik term:"
  call print_nonzero_exp(expfun(3))

  write(*,fmt="(/a)") "δ^NN_jl term:"
  call print_nonzero_exp(expfun(4))

  write(*,fmt="(/a)") "δ^NN_ik+δ^NN_jl terms together:"
  call print_nonzero_exp(expfun34)

 end subroutine

 subroutine task_print_cos

  write(*,fmt="(a)") "δ_il term:"
  call print_nonzero_cos(cosfun(1))

  write(*,fmt="(/a)") "δ^NN_il term:"
  call print_nonzero_cos(cosfun(2))

  write(*,fmt="(/a)") "δ^NN_ik term:"
  call print_nonzero_cos(cosfun(3))

  write(*,fmt="(/a)") "δ^NN_jl term:"
  call print_nonzero_cos(cosfun(4))

  write(*,fmt="(/a)") "δ^NN_ik+δ^NN_jl terms together:"
  call print_nonzero_cos(cosfun34)

 end subroutine

 subroutine check_pairing_formfac
  real(dp) Kx,Ky,Kxp,Kyp,F1ref,F2ref,F3ref,F1,F2,F3,F1s,F2s,F3s,F1d,F2d,F3d
  integer i

   do i=1,12
    call random_number(Kx); Kx=pi*(2*Kx-1)
    call random_number(Ky); Ky=pi*(2*Ky-1)
    call random_number(Kxp); Kxp=pi*(2*Kxp-1)
    call random_number(Kyp); Kyp=pi*(2*Kyp-1)

    F1ref=eval_expfun(expfun(1),Kx,Ky,Kxp,Kyp)
    F2ref=eval_expfun(expfun(2),Kx,Ky,Kxp,Kyp)
    F3ref=eval_expfun(expfun34,Kx,Ky,Kxp,Kyp)

    call eval_pairing_formfac(Kx,Ky,Kxp,Kyp,F1,F2,F3,F1s,F2s,F3s,F1d,F2d,F3d)

    write(*,fmt="(i3,4f15.12)") i,abs(F1-F1ref),abs(F2-F2ref),abs(F3-F3ref)
    write(*,fmt="(3x,4f15.12)") abs(F1s+F1d-F1),abs(F2s+F2d-F2),abs(F3s+F3d-F3)
   end do

 end subroutine

 subroutine SVD_decompose_cosfun(cosfun)
  type(cos_table) cosfun
  integer m1,m2,m3,m4,jA,jB,i,M
  real(dp), allocatable :: aux(:,:),singval(:),U(:,:),Vtrans(:,:),singvec(:)

   M=(2*N+1)**2

   allocate(aux(M,M),singval(M),U(M,M),Vtrans(M,M),singvec(M))

   do m1=0,2*N
    do m2=0,2*N
     jA=(2*N+1)*m1+m2+1

     do m3=0,2*N
      do m4=0,2*N
       jB=(2*N+1)*m3+m4+1

       aux(jA,jB)=cosfun%coef(m1,m2,m3,m4)

      end do
     end do
    end do
   end do

   call svd_real(M,M,aux,singval,U,Vtrans)

   do i=1,M
    if (abs(singval(i))>1e-12) then

     if (dot_product(Vtrans(i,:),U(:,i))<0) then
      singval(i)=-singval(i)
     end if

     singvec=U(:,i)

     if (sum(singvec)<0) singvec=-singvec

     write(*,fmt="(a,i3,f20.15)") "singular value ",i,singval(i)
     do m1=0,2*N
      do m2=0,2*N
       jA=(2*N+1)*m1+m2+1
       if (abs(U(jA,i))>1e-12) write(*,fmt="(20x,2i4,2f15.10)") m1,m2,singvec(jA) 
      end do
     end do
    end if
   end do

 end subroutine

 subroutine svd_real(M,N,mtx,singval,U,Vtrans)
  integer M,N,info,lwork
  real(dp) mtx(M,N),U(M,M),Vtrans(N,N),singval(min(M,N)),rwork(5*min(M,N))
  real(dp), allocatable :: aux(:,:),work(:)

   lwork=-1
   allocate(aux(M,N),work(1))
   call DGESVD('A','A', M,N, aux,M, singval, U,M, Vtrans,N, work,lwork, info)
   lwork=int(work(1))
   deallocate(work)

   allocate(work(lwork))
   aux=mtx
   call DGESVD('A','A', M,N, aux,M, singval, U,M, Vtrans,N, work,lwork, info)
   deallocate(work,aux)

   if (info/=0) write(*,*) 'DGESVD returned INFO=',info

 end subroutine

 subroutine svd_cmplx(M,N,mtx,singval,U,Vconjg)
  integer M,N,info,lwork
  complex(dp) mtx(M,N),U(M,M),Vconjg(N,N)
  real(dp) singval(min(M,N)),rwork(5*min(M,N))
  complex(dp), allocatable :: aux(:,:),work(:)

   lwork=-1
   allocate(aux(M,N),work(1))
   call ZGESVD('A','A', M,N, aux,M, singval, U,M, Vconjg,N, work,lwork, rwork, info)
   lwork=int(work(1))
   deallocate(work)

   allocate(work(lwork))
   aux=mtx
   call ZGESVD('A','A', M,N, aux,M, singval, U,M, Vconjg,N, work,lwork, rwork, info)
   deallocate(work,aux)

   if (info/=0) write(*,*) 'ZGESVD returned INFO=',info

 end subroutine

 subroutine task_VKK_avg(Ndop,dop_min,dop_max,Nphi,fname)
  integer Ndop,Nphi,jd
  real(dp) dop_min,dop_max,dop,kF,V0,V1a,V1b
  character(len=*) fname

   open(1,file=fname)
   write(1,fmt="(a)") "# 1:doping  2:kF  3:V0  4:V1a (due to T-disp)  5:V1b (due to <TT>,<T†T†>)"
   write(1,fmt="(a)") "# BCS pairing <VKK'> is (3A^2/E_T)*[V0+κ/E_T*(V1a+V1b)]"
   do jd=1,Ndop
    write(*,*) jd,"/",Ndop

    dop=dop_min
    if (Ndop>1) dop=(dop_min*(Ndop-jd)+dop_max*(jd-1))/(Ndop-1)

    kF=sqrt(2*pi*dop)

    call VKK_avg_simple(dop,Nphi,V0,V1a,V1b)

    write(1,*) dop,kF,V0,V1a,V1b
   end do
   close(1)

 end subroutine

 subroutine task_map_BCS_lambda(doping, Dmin,Dmax,ND,tmin,tmax,Nt, Ud,JH, fname)
  real(dp) doping, Dmin,Dmax,tmin,tmax, Ud,JH, &
   V0,V1a,V1b, Delta,thop, ET,the, E1,E2,E3,kappa_fac,kappa,gapM,tcrit, &
   C0,A, DOS, lambda,lam0,lam1a,lam1b, bW, Tc, bW0,Tc0,bW01a,Tc01a,bW1b,Tc1b
  integer ND,Nt,i,j
  character(len=*) fname

   call VKK_avg_simple(doping,1000,V0,V1a,V1b)

   write(*,*) "V0,V1a,V1b:",V0,V1a,V1b

   open(1,file=fname)
   write(1,fmt="(a)") "# 1:Δ 2:t 3:ET 4:κ 5:ωM 6:λ 7:λ0 8:λ1a 9:λ1b 10:βcΩ 11:Tc 12-14:Tc0,Tc01a,Tc1b"
   open(2,file=fname//"PB")
   write(2,fmt="(a)") "# 1:Δ 2:ET 3:t_crit"
   do i=1,ND
    Delta=Dmin
    if (ND>1) Delta=(Dmin*(ND-i)+Dmax*(i-1))/(ND-1)

    ET=sqrt(Delta**2+JH**2)-3*JH
    the=atan(JH/Delta)/2

    E1=Ud+JH-Delta+2*ET
    E2=Ud+JH      +2*ET
    E3=Ud+JH+Delta+2*ET

    kappa_fac=( (1+cos(2*the))/E1 + 2*sin(2*the)/E2 + (1-cos(2*the))/E3 )/3

    tcrit=sqrt( ET/8 / kappa_fac )

    write(2,*) Delta,ET,tcrit

    do j=1,Nt
     thop=tmin
     if (Nt>1) thop=(tmin*(Nt-j)+tmax*(j-1))/(Nt-1)

     kappa=thop**2*kappa_fac

     gapM=sqrt(max( ET*(ET-8*kappa), 0.0_dp))

     C0=thop/3*cos(the)**2
     DOS=1/(4*pi*C0)

     A=thop/sqrt(6.0_dp)*cos(the)

     if (ET>8*kappa) then
      lam0=  -(3*A**2/ET)* V0 * DOS
      lam1a= -(3*A**2/ET)* kappa/ET*V1a * DOS
      lam1b= -(3*A**2/ET)* kappa/ET*V1b * DOS

      lambda=lam0+lam1a+lam1b

      bW=BCSeq_lambda_to_bW(lambda)
      Tc=ET/bW

      bW0=BCSeq_lambda_to_bW(lam0)
      Tc0=ET/bW0

      bW01a=BCSeq_lambda_to_bW(lam0+lam1a)
      Tc01a=ET/bW01a

      bW1b=BCSeq_lambda_to_bW(lam1b)
      Tc1b=ET/bW1b
     else
      lam0=0
      lam1a=0
      lam1b=0
      lambda=0
      bW=0
      Tc=0
      bW0=0
      Tc0=0
      bW01a=0
      Tc01a=0
      bW1b=0
      Tc1b=0
     end if

     write(1,*) Delta,thop,ET,kappa,gapM,lambda,lam0,lam1a,lam1b,bW,Tc,Tc0,Tc01a,Tc1b
    end do
    if (Nt>1) write(1,*)
   end do
   close(1)
   close(2)

 end subroutine

end program
