/*
 * Copyright 2025 IKERLAN and BArcelona SUpercomputing Center - Centro Nacional de Supercomputacion (BSC-CNS)
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
#include "smw_contention_guard/non_critical_application.hpp"

#include <chrono>

#include "smw_contention_guard/contention_function.hpp"

namespace smw_contention_guard
{
NonCriticalApplication::NonCriticalApplication(const std::string & name)
: smw_base_application::BaseApplication{name}
{
  declare_parameter("pmu_topic", std::string(this->get_namespace()) + "/contention_guard/pmu");
  declare_parameter(
    "feedback_topic", std::string(this->get_namespace()) + "/contention_guard/feedback");
  declare_parameter("pmu_timer_ms", 1);

  pmu_cb_group_ = create_callback_group(rclcpp::CallbackGroupType::MutuallyExclusive);
  throttle_cb_group_ = create_callback_group(rclcpp::CallbackGroupType::MutuallyExclusive);
  feedback_cb_group_ = create_callback_group(rclcpp::CallbackGroupType::MutuallyExclusive);
  alive_cb_group_ = create_callback_group(rclcpp::CallbackGroupType::MutuallyExclusive);

  configurePmu();
}

NonCriticalApplication::CallbackReturn NonCriticalApplication::on_shutdown(const State & state)
{
  NonCriticalApplication::CallbackReturn cb_return = BaseApplication::on_shutdown(state);
  pmu_pub_sk_->stopOfferService();

  return cb_return;
}

NonCriticalApplication::CallbackReturn NonCriticalApplication::on_configure(const State & state)
{
  NonCriticalApplication::CallbackReturn cb_return = BaseApplication::on_configure(state);

  std::string pmu_topic = get_parameter("pmu_topic").as_string();
  std::string feedback_topic = get_parameter("feedback_topic").as_string();
  int pmu_timer_ms = get_parameter("pmu_timer_ms").as_int();
  int reference_cycle_ms = get_parameter("reference_cycle_ms").as_int();

  pmu_pub_sk_ = std::make_unique<smw_comm::ServiceSkeleton<PMUCounters>>(this, pmu_topic);
  pmu_pub_sk_->offerService();

  rclcpp::SubscriptionOptions options;
  options.callback_group = feedback_cb_group_;
  feedback_sub_ = this->create_subscription<CGFeedback>(
    feedback_topic, 10,
    std::bind(&NonCriticalApplication::feedbackCallback, this, std::placeholders::_1), options);

  pmu_timer_ = this->create_wall_timer(
    std::chrono::milliseconds(pmu_timer_ms), std::bind(&NonCriticalApplication::pmuCallback, this),
    pmu_cb_group_);
  pmu_timer_->cancel();

  throttle_timer_ = this->create_wall_timer(
    std::chrono::milliseconds::zero(), std::bind(&NonCriticalApplication::throttleCallback, this),
    throttle_cb_group_);
  throttle_timer_->cancel();

  if (health_client_ != nullptr) {
    alive_timer_ = this->create_wall_timer(
      std::chrono::milliseconds(reference_cycle_ms),
      std::bind(&NonCriticalApplication::aliveCallback, this), throttle_cb_group_);
    alive_timer_->cancel();
  }

  return cb_return;
}

void NonCriticalApplication::feedbackCallback(std::shared_ptr<CGFeedback> msg)
{
  if (msg->state == CGFeedback::THROTTLE) {
    is_throttling = true;
    pmu_timer_->cancel();
    throttle_timer_->reset();
    if ((health_client_ != nullptr) & alive_timer_->is_canceled()) {
      // RCLCPP_INFO(this->get_logger(), "ALIVE RESET.");  //DEBUG
      alive_timer_->reset();
    }
  } else if (msg->state == CGFeedback::ON) {
    is_throttling = false;
    pmu_->stop();
    pmu_->reset_and_start();
    pmu_timer_->reset();
    if ((health_client_ != nullptr) & !alive_timer_->is_canceled()) {
      alive_timer_->cancel();
      // RCLCPP_INFO(this->get_logger(), "ALIVE CANCEL.");  //DEBUG
    }
  } else if (msg->state == CGFeedback::OFF) {
    is_throttling = false;
    pmu_timer_->cancel();
    pmu_->stop();
    if ((health_client_ != nullptr) & !alive_timer_->is_canceled()) {
      alive_timer_->cancel();
      // RCLCPP_INFO(this->get_logger(), "ALIVE CANCEL.");  //DEBUG
    }
  } else {
    RCLCPP_INFO(this->get_logger(), "ERROR: unknown signal received from control node.");  //DEBUG
  }
}

void NonCriticalApplication::pmuCallback()
{
  std::vector<unsigned int> counters = pmu_->stop_and_read();
  pmu_->start();

  auto msg = PMUCounters();
  std::copy(counters.begin(), counters.end(), std::back_inserter(msg.counters));

  // time_pmc_sent = std::chrono::high_resolution_clock::now();
  pmu_pub_sk_->send(msg);
  // RCLCPP_INFO(this->get_logger(), "Non-critical PMU counters [%d]", counters[0]);
}

void NonCriticalApplication::throttleCallback()
{
  throttle_timer_->cancel();
  time_throttle_enter = std::chrono::high_resolution_clock::now();
  RCLCPP_INFO(this->get_logger(), "Non-critical. Throttle starts");
  while (is_throttling) { /* idle */
  }
  RCLCPP_INFO(
    this->get_logger(), "Non-critical. Throttle stopped. Duration %.3f us",
    (float)std::chrono::duration_cast<std::chrono::microseconds>(
      std::chrono::high_resolution_clock::now() - time_throttle_enter)
      .count());
}

void NonCriticalApplication::aliveCallback() { health_client_->reportCheckpoint(); }

void NonCriticalApplication::beforeRun()
{
  before_run_time_ = std::chrono::high_resolution_clock::now();
}

void NonCriticalApplication::afterRun()
{
  after_run_time_ = std::chrono::high_resolution_clock::now();
  auto duration_time =
    std::chrono::duration_cast<std::chrono::microseconds>(after_run_time_ - before_run_time_)
      .count();
  RCLCPP_INFO(
    this->get_logger(), "Non-critical. Task finished. Duration: %s us",
    std::to_string(duration_time).c_str());
}

bool NonCriticalApplication::configurePmu()
{
  if (get_parameter("enable_performance_monitoring").as_bool())
    RCLCPP_INFO(
      get_logger(),
      "Non-critical applications ignore the \"enable_performance_monitoring\" and "
      "\"performance_monitoring_mode\" parameters.");

  pmu_ = std::make_shared<PmuLibrary>();
  std::vector<unsigned int> events(3);
  events[ContentionFunctionL3::L2D_CACHE_LMISS_RD] = PMU_A78AE_L2D_CACHE_LMISS_RD;
  events[ContentionFunctionL3::L2D_CACHE_WB] = PMU_A78AE_L2D_CACHE_WB;
  events[ContentionFunctionL3::CPU_CYCLES] = PMU_A78AE_CPU_CYCLES;
  int config_result = pmu_->configure(
    events,
    PMU_MODE_PER_THREAD | PMU_MODE_SCOPE_ALL_PROCESSES | PMU_MODE_CORE_FIXED | getCpuSet()[0]);

  if (config_result == 0) {
    RCLCPP_INFO(get_logger(), "PMU successfully configured for a non-critical application.");
    return true;
  }
  RCLCPP_ERROR(get_logger(), "Error in PMU configuration: %d.", config_result);
  return false;
}
}  // namespace smw_contention_guard
