/*
  * Copyright 2025 IKERLAN and BArcelona SUpercomputing Center - Centro Nacional de Supercomputacion (BSC-CNS)
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include "smw_contention_guard/critical_application.hpp"

namespace smw_contention_guard
{
CriticalApplication::CriticalApplication(const std::string & name)
: smw_base_application::BaseApplication(name)
{
  declare_parameter(
    "enable_topic", std::string(this->get_namespace()) + "/contention_guard/enable");
  cguard_msg_.enable = false;
}

CriticalApplication::CallbackReturn CriticalApplication::on_configure(const State & state)
{
  CriticalApplication::CallbackReturn cb_return = BaseApplication::on_configure(state);

  std::string enable_topic = get_parameter("enable_topic").as_string();
  cguard_pub_ = this->create_publisher<CGEnable>(enable_topic, 10);

  return cb_return;
}

void CriticalApplication::beforeRun()
{
  cguard_msg_.enable = true;
  cguard_pub_->publish(cguard_msg_);
  before_run_time_ = std::chrono::high_resolution_clock::now();
  RCLCPP_INFO(get_logger(), "Critical. Task starts.");
  BaseApplication::beforeRun();
}

void CriticalApplication::afterRun()
{
  BaseApplication::afterRun();
  after_run_time_ = std::chrono::high_resolution_clock::now();
  cguard_msg_.enable = false;
  cguard_pub_->publish(cguard_msg_);
  auto duration_time =
    std::chrono::duration_cast<std::chrono::microseconds>(after_run_time_ - before_run_time_)
      .count();
  RCLCPP_INFO(
    get_logger(), "Critical. Task finished. Duration: %s us",
    std::to_string(duration_time).c_str());
}
}  // namespace smw_contention_guard
