/*
  * Copyright 2025 IKERLAN and BArcelona SUpercomputing Center - Centro Nacional de Supercomputacion (BSC-CNS)
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include "smw_contention_guard/contention_guard.hpp"

#include <chrono>
#include <cstdio>
#include <memory>

#include "rclcpp/rclcpp.hpp"

namespace smw_contention_guard
{
ContentionGuard::ContentionGuard() : smw_base_application::BaseApplication{"contention_guard"}
{
  declare_parameter(
    "enable_topic", std::string(this->get_namespace()) + "/contention_guard/enable");
  declare_parameter("pmu_topic", std::string(this->get_namespace()) + "/contention_guard/pmu");
  declare_parameter(
    "feedback_topic", std::string(this->get_namespace()) + "/contention_guard/feedback");
  declare_parameter("contention_threshold", 18500.0);

  enable_cb_group_ = this->create_callback_group(rclcpp::CallbackGroupType::MutuallyExclusive);
  last_feedback_.state = CGFeedback::OFF;
}

bool ContentionGuard::initialize()
{
  std::string enable_topic = get_parameter("enable_topic").as_string();
  std::string pmu_topic = get_parameter("pmu_topic").as_string();
  std::string feedback_topic = get_parameter("feedback_topic").as_string();

  contention_function_ =
    std::make_unique<ContentionFunctionL3>(get_parameter("contention_threshold").as_double());

  rclcpp::SubscriptionOptions enable_options;
  enable_options.callback_group = enable_cb_group_;
  enable_sub_ = this->create_subscription<CGEnable>(
    enable_topic, 10, std::bind(&ContentionGuard::enableCallback, this, std::placeholders::_1),
    enable_options);

  pmu_proxy_ = std::make_shared<smw_comm::ServiceProxy<PMUCounters>>(this, pmu_topic, 10);

  feedback_pub_ = this->create_publisher<CGFeedback>(feedback_topic, 10);
  pmu_proxy_->findService();

  return true;
}

bool ContentionGuard::run()
{
  std::vector<std::unique_ptr<PMUCounters>> msg_list = pmu_proxy_->getOldestSample();
  if (!msg_list.empty() & (last_feedback_.state == CGFeedback::ON)) {
    // RCLCPP_INFO(this->get_logger(), "ControlGuard. Received pmu_counter: %d", msg_list[0]->counters[0]);
    contention_detected_ = contention_function_->check(msg_list[0]);
    decideState();
  }
  return true;
}

bool ContentionGuard::terminate() { return true; }

void ContentionGuard::enableCallback(std::shared_ptr<CGEnable> msg)
{
  cg_enabled_ = msg->enable;
  decideState();
}

void ContentionGuard::decideState()
{
  // RCLCPP_INFO(this->get_logger(), "control fcn");
  CGFeedback feedback;
  if (cg_enabled_ & contention_detected_) {  // ENABLED & CONTENTION
    feedback.state = CGFeedback::THROTTLE;
    RCLCPP_INFO(
      this->get_logger(), "Contention guard. Last bandwidth: %0.3f",
      contention_function_->getLastBandwidth());
    contention_detected_ = false;
  } else if (cg_enabled_ & !contention_detected_) {  // ENABLED & NO CONTENTION
    feedback.state = CGFeedback::ON;
  } else {  // ANY OTHER CASE
    feedback.state = CGFeedback::OFF;
  }

  if (feedback.state != last_feedback_.state) {  // IF CHANGE SINCE LAST
    feedback_pub_->publish(feedback);
    last_feedback_ = feedback;
    RCLCPP_INFO(get_logger(), "Contention guard. Feedback state: %d", last_feedback_.state);
  }
}
}  // namespace smw_contention_guard