/*
 * Copyright 2025 IKERLAN and BArcelona SUpercomputing Center - Centro Nacional de Supercomputacion (BSC-CNS)
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


 #include "smw_contention_guard/contention_function.hpp"

#include <limits>

namespace smw_contention_guard
{
ContentionFunctionL3::ContentionFunctionL3(const uint32_t & threshold)
: threshold_(threshold),
  l2d_misses_(std::numeric_limits<uint32_t>::max()),
  l2d_write_backs_(std::numeric_limits<uint32_t>::max()),
  cpu_cycles_(std::numeric_limits<uint32_t>::max())
{
}

bool ContentionFunctionL3::pmuHasBeenReset(const std::unique_ptr<PMUCounters> & msg)
{
  return (l2d_misses_ >= msg->counters[L2D_CACHE_LMISS_RD]) &
         (l2d_write_backs_ >= msg->counters[L2D_CACHE_WB]) &
         (cpu_cycles_ >= msg->counters[CPU_CYCLES]);
}

bool ContentionFunctionL3::check(const std::unique_ptr<PMUCounters> & msg)
{
  bool contention = false;
  if (!pmuHasBeenReset(msg)) {
    unsigned int l3_accesses = (msg->counters[L2D_CACHE_LMISS_RD] - l2d_misses_) +
                               (msg->counters[L2D_CACHE_WB] - l2d_write_backs_);
    unsigned cycles = msg->counters[CPU_CYCLES] - cpu_cycles_;

    l3_bwidth_ = (double)64 * cpu_clock_freq_ * l3_accesses / cycles / 1024 / 1024;
    contention = l3_bwidth_ >= threshold_;
  }

  l2d_misses_ = msg->counters[L2D_CACHE_LMISS_RD];
  l2d_write_backs_ = msg->counters[L2D_CACHE_WB];
  cpu_cycles_ = msg->counters[CPU_CYCLES];

  return contention;
}

double ContentionFunctionL3::getLastBandwidth() { return l3_bwidth_; }

}  // namespace smw_contention_guard