/*
 * Copyright 2025 IKERLAN and BArcelona SUpercomputing Center - Centro Nacional de Supercomputacion (BSC-CNS)
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

 #ifndef SMW_CONTENTION_GUARD__RUN_APPLICATION_HPP
#define SMW_CONTENTION_GUARD__RUN_APPLICATION_HPP

#include <cstdlib>
#include <iostream>
#include <type_traits>

#include "rclcpp/rclcpp.hpp"
#include "smw_contention_guard/critical_application.hpp"
#include "smw_contention_guard/non_critical_application.hpp"
#include "smw_util/thread_utilities.hpp"

namespace smw_contention_guard
{
template <
  typename Application,
  std::enable_if_t<std::is_base_of_v<CriticalApplication, Application>, bool> = true>
int runCriticalApplication(int argc, char ** argv)
{
  rclcpp::init(argc, argv);
  const auto node = std::make_shared<Application>();
  rclcpp::executors::SingleThreadedExecutor default_executor, timer_executor;

  default_executor.add_callback_group(
    node->getDefaultCallbackGroup(), node->get_node_base_interface());
  timer_executor.add_callback_group(node->getTimerCallbackGroup(), node->get_node_base_interface());

  auto default_thread = std::thread([&]() { default_executor.spin(); });
  auto timer_thread = std::thread([&]() { timer_executor.spin(); });

  std::vector<int64_t> cpu_set = node->getCpuSet();
  smw_util::configure_thread(default_thread, smw_util::ThreadPriority::HIGHEST, cpu_set);
  smw_util::configure_thread(timer_thread, smw_util::ThreadPriority::HIGHEST, cpu_set);

  default_thread.join();
  timer_thread.join();

  rclcpp::shutdown();

  return EXIT_SUCCESS;
}

template <
  typename Application,
  std::enable_if_t<std::is_base_of_v<NonCriticalApplication, Application>, bool> = true>
int runNonCriticalApplication(int argc, char ** argv)
{
  rclcpp::init(argc, argv);
  const auto node = std::make_shared<Application>();
  rclcpp::executors::SingleThreadedExecutor default_executor;
  rclcpp::executors::SingleThreadedExecutor non_critical_executor;
  rclcpp::executors::SingleThreadedExecutor feedback_executor;
  rclcpp::executors::SingleThreadedExecutor throttle_executor;
  rclcpp::executors::SingleThreadedExecutor pmu_executor;
  rclcpp::executors::SingleThreadedExecutor alive_executor;

  default_executor.add_callback_group(
    node->getDefaultCallbackGroup(), node->get_node_base_interface());
  non_critical_executor.add_callback_group(
    node->getTimerCallbackGroup(), node->get_node_base_interface());
  feedback_executor.add_callback_group(
    node->getFeedbackCallbackGroup(), node->get_node_base_interface());
  throttle_executor.add_callback_group(
    node->getThrottleCallbackGroup(), node->get_node_base_interface());
  pmu_executor.add_callback_group(node->getPmuCallbackGroup(), node->get_node_base_interface());
  alive_executor.add_callback_group(node->getAliveCallbackGroup(), node->get_node_base_interface());

  auto default_thread = std::thread([&]() { default_executor.spin(); });

  auto non_critical_thread = std::thread([&]() { non_critical_executor.spin(); });

  auto feedback_thread = std::thread([&]() { feedback_executor.spin(); });

  auto throttle_thread = std::thread([&]() { throttle_executor.spin(); });

  auto pmu_thread = std::thread([&]() { pmu_executor.spin(); });

  auto alive_thread = std::thread([&]() { alive_executor.spin(); });

  std::vector<int64_t> cpu_set = node->getCpuSet();
  smw_util::configure_thread(default_thread, smw_util::ThreadPriority::HIGH, cpu_set);
  smw_util::configure_thread(non_critical_thread, smw_util::ThreadPriority::LOWEST, cpu_set);
  smw_util::configure_thread(feedback_thread, smw_util::ThreadPriority::HIGHEST, cpu_set);
  smw_util::configure_thread(throttle_thread, smw_util::ThreadPriority::HIGH, cpu_set);
  smw_util::configure_thread(pmu_thread, smw_util::ThreadPriority::LOW, cpu_set);
  smw_util::configure_thread(alive_thread, smw_util::ThreadPriority::HIGHEST, cpu_set);

  non_critical_thread.join();
  feedback_thread.join();
  throttle_thread.join();
  pmu_thread.join();
  alive_thread.join();

  rclcpp::shutdown();

  return EXIT_SUCCESS;
}

}  // namespace smw_contention_guard

#endif  // SMW_CONTENTION_GUARD__RUN_APPLICATION_HPP
