 # **Contention Guard**

The Contention Guard (CGuard) is a safety mechanism designed to control and mitigate memory contention situations by stopping non-critical tasks’ memory usage if necessary. We extended the C++ *BaseApplication* class to support the definition of critical and non-critical applications (as the *CriticalApplication* and *NonCriticalApplication* classes), and we developed the C++ *ContentionGuard* class to monitor these applications and apply the mechanism (i.e., decide whether to stop the *NonCriticalApplication*).

There is an example of how to set up the contention guard mechanism in `safexplain_middleware/smw_examples/include/contention_guard/` (header files),`safexplain_middleware/smw_examples/src/contention_guard/` (source files) and its corresponding launch file `safexplain_middleware/smw_examples/launch/contention_guard.launch.py`.

## Define a *CriticalApplication*

A *CriticalApplication* must be implemented in the same way as a *BaseApplication* , i.e., by defining the `initialize()`, `run()`, and `terminate()` functions, but inheriting from `smw_contention_guard::CriticalApplication` instead.

```cpp
#include "smw_contention_guard/critical_application.hpp"
#include "smw_util/kernel_wrapper.hpp"

namespace smw_examples
{
class ExampleCriticalApplication final : public smw_contention_guard::CriticalApplication
{
public:
  ExampleCriticalApplication();
  ~ExampleCriticalApplication() override;

  bool initialize() override;
  bool run() override;
  bool terminate() override;
  // ...
};
}  // namespace smw_examples
```

![Screenshot from the header file of an example CriticalApplication](images/cguard_critical_header.png)

Additionally, when defining the C++ `main` function, the application must be executed with `smw_contention_guard::runCriticalApplication`.

```cpp
#include "smw_contention_guard/run_application.hpp"
#include "smw_examples/contention_guard/example_critical_application.hpp"

int main(int argc, char ** argv)
{
    return smw_contention_guard::runCriticalApplication<smw_examples::ExampleCriticalApplication>(
    argc, argv);
}
```

![Screenshot from the source file of an example CriticalApplication](images/cguard_critical_main.png)

## Define a Non*CriticalApplication*

The same procedure applies to defining a *NonCriticalApplication*, but it must inherit from `smw_contention_guard::NonCriticalApplication` and be executed with `smw_contention_guard::runNonCriticalApplication`.

```cpp
#include "smw_contention_guard/non_critical_application.hpp"
#include "smw_util/kernel_wrapper.hpp"

namespace smw_examples
{
class ExampleNonCriticalApplication final : public smw_contention_guard::NonCriticalApplication
{
public:
  ExampleNonCriticalApplication();
  ~ExampleNonCriticalApplication() override;

  bool initialize() override;
  bool run() override;
  bool terminate() override;
// ...
};
}  // namespace smw_examples
```

![Screenshot from the header file of an example NonCriticalApplication](images/cguard_noncritical_header.png)

```cpp
#include "smw_contention_guard/run_application.hpp"
#include "smw_examples/contention_guard/example_non_critical_application.hpp"

int main(int argc, char ** argv)
{
  return smw_contention_guard::runNonCriticalApplication<
    smw_examples::ExampleNonCriticalApplication>(argc, argv);
}
```

![Screenshot from the source file of an example NonCriticalApplication](images/cguard_noncritical_main.png)

## Configure the *ContentionGuard* mechanism

After defining both the critical and non-critical applications, they must be properly configured along with the *ContentionGuard*. The following parameters need to be specified:

* **CriticalApplication**:
  * `reference_cycle_ms`: period of the *CriticalApplication*'s `run()` function in milliseconds.
  * `cpu_set`: core on which the application will run. Multiple cores can be selected by adding them to the list.
* **NonCriticalApplication**:
  * `reference_cycle_ms`: period of the *NonCriticalApplication*'s `run()` function in milliseconds.
  * `pmu_timer_ms`: interval at which contention is measured. This controls the period of the callback that sends messages to the  *ContentionGuard* .
  * `cpu_set`: core on which the application will run. Multiple cores can be selected by adding them to the list.
* **ContentionGuard**:
  * `reference_cycle_ms`: period of the *ContentionGuard*'s `run()` function in milliseconds. This must match the `pmu_timer_ms` parameter from the *NonCriticalApplication* .
  * `contention_threshold`: threshold value used to determine whether contention exists.
  * `cpu_set`: core on which the application will run. Multiple cores can be selected by adding them to the list.

Note that each node should run on a different core, which means the `cpu_set` values must be different. In this case, the `taskset` command is not used (as explained earlier for mapping nodes to cores), because the `cpu_set` value determines the core from which contention counters are collected. It is recommended to use cores from different clusters to avoid contention between the three nodes.

```python
return LaunchDescription(
    [
        # ...
        Node(
            package="smw_lifecycle_manager",
            executable="lifecycle_manager",
            namespace="smw_examples",
            parameters=[
                {
                    "node_names": [
                        "example_critical_application",
                        "example_non_critical_application",
                        "contention_guard",
                    ],
                    "startup_timeout_s": 10,
                },
            ],
        ),
        Node(
            package="smw_examples",
            executable="example_critical_application",
            namespace="smw_examples",
            parameters=[
                {
                    "reference_cycle_ms": 1500,   		#
                    "enable_alive_monitoring": True,
                    "enable_health_monitoring": False,
                    "cpu_set": [1],               		#
                },
            ],
        ),
        Node(
            package="smw_examples",
            executable="example_non_critical_application",
            namespace="smw_examples",
            parameters=[
                {
                    "reference_cycle_ms": 50,     		#
                    "pmu_timer_ms": pmu_timer_ms,   		#
                    "enable_alive_monitoring": True,
                    "enable_health_monitoring": False,
                    "cpu_set": [3],               		#
                },
            ],
        ),
        Node(
            package="smw_contention_guard",
            executable="contention_guard",
            namespace="smw_examples",
            parameters=[
                {
                    "reference_cycle_ms": pmu_timer_ms,		#
                    "enable_alive_monitoring": True,
                    "enable_health_monitoring": False,
                    "contention_threshold": 3000.0,		#
                    "cpu_set": [2],      			#
                },
            ],
        ),
        # ...
    ])
```

![Screenshot from the contention_guard.launch.py</code> file](images/cguard_launch.png)
