import {defineStore} from "pinia";
import {PropType} from "vue";
import {Coordinate, Polygon} from "@/types";
import axios from "axios";
import {process_url} from "@/components/map/wms/processes/process.ts";
import proj4 from "proj4";
import {
    DrawingToolConfiguration,
    DrawnObject,
    ObjectConfiguration,
    ObjectGroup,
    ObjectProcess,
    Properties
} from "@/types/drawing";


export const useDrawingStore = defineStore('drawing', {

    state: (): any => ({
        drawingTool: null as DrawingToolConfiguration|null,
        dialogObject: null as ObjectConfiguration|null,
        groupObjects: [] as PropType<DrawnObject[]>,
        groups: [] as PropType<ObjectGroup[]>,
        groupId: 1 as number,
        groupName: '' as string,
        clickedCoord: null as PropType<Coordinate>|null,
        clickedPolygon: null as PropType<Polygon>|null,
        loadingCount: 0,
        isLoading: false as boolean
    }),
    getters: {
        isPointMode() {
            return this.dialogObject?.type == 'Point'
        },
        objCount() {
            return this.groupObjects.length + 1
        },
        isReportMode() {
            for (const obj of this.drawingTool.objects) {
                if (obj.processes.length > 0) {
                    return true
                }
            }
            return false
        }
    },
    actions: {
        addObject(fields: Properties) {
            this.groupObjects.push({
                "index": this.objCount,
                "dialogObjId": this.dialogObject.id,
                "coord": this.clickedCoord,
                "polygon": this.clickedPolygon,
                "properties": fields
          })
        },
        removeObject(obj: DrawnObject) {
            const index = this.groupObjects.findIndex((groupObj: DrawnObject) => groupObj.index === obj.index)
            this.groupObjects.splice(index, 1)

            if (this.groupObjects.length === 0) {
                this.dialogObject = null
            }
        },
        isGroupSaved() {
            return (this.getGroupIfExists(this.groupId) !== false)
        },
        getGroupIfExists(id: number) {
            for (const group of this.groups) {
              if (id === group.id) {
                  return group
              }
            }
            return false
        },
        getNewGroupId(id: number = 1) {
            if (this.getGroupIfExists(id) === false) {
                return id
            } else {
                return this.getNewGroupId(id + 1)
            }
        },
        getPreviousGroup(id: number) {
            if (id === 0) {
                return false
            }
            const group = this.getGroupIfExists(id-1)
            if (group !== false) {
                return group
            } else {
                return this.getSucceedingGroup(id-1)
            }
        },
        getSucceedingGroup(id: number) {
            const group = this.getGroupIfExists(id+1)
            if (group !== false) {
                return group
            } else {
                return this.getSucceedingGroup(id + 1)
            }
        },
        saveGroup() {
            if (!this.isGroupSaved()) {
                this.groups.push({
                    "id": this.groupId,
                    "name": this.groupName,
                    "objects": this.groupObjects
                })
            } else {
                for (const group of this.groups) {
                  if (this.groupId == group.id) {
                      group.name = this.groupName
                      group.objects = this.groupObjects
                  }
                }
            }
        },
        createGroup() {
            this.dialogObject = null
            this.groupObjects = []
            this.groupId = this.getNewGroupId()
            this.groupName = ''
            this.saveGroup()
        },
        setGroup(group: ObjectGroup) {
            if (group.objects.length > 0) {
                for (const obj of this.drawingTool?.objects) {
                    if (obj.id == group.objects[0].dialogObjId) {
                        this.dialogObject = obj
                    }
                }
            } else {
                this.dialogObject = null
            }
            this.groupId = group.id
            this.groupName = group.name
            this.groupObjects = group.objects
        },
        deleteActiveGroup() {
            const index = this.groups.findIndex((group: ObjectGroup) => group.id === this.groupId)
            this.groups.splice(index, 1)
            this.dialogObject = null
            this.groupName = ''
            this.groupObjects = []

            if (this.groups.length > 0) {
                let nextGroup = this.getPreviousGroup(this.groupId)
                if (!nextGroup) {
                    nextGroup = this.getSucceedingGroup(this.groupId)
                }
                if (nextGroup) {
                    this.setGroup(nextGroup)
                }
            }
        },
        requestReport() {
            this.saveGroup()

            this.loadingCount = 0
            this.isLoading = true

            for (const group of this.groups) {
                group['report'] = []
                for (const process of this.dialogObject.processes) {
                    this.requestValuesByProcess(process, group)
                }
            }
        },
        getGeometry(group: ObjectGroup|undefined = undefined) {
            this.saveGroup()

            if (!group) {
                group = this.groups[0]
            }

            return [
                    "geometry",
                    [{
                        "type": "Polygon",
                        "crs": "EPSG:4326",
                        "coordinates": this.transformGroup(group)
                    }]]
        },
        transformGroup(group: ObjectGroup) {
            const result = []

            for (const obj of group['objects']) {
                for (const polygon of obj['polygon']) {
                    const transformed_polygon = []
                    for (let coord of polygon) {
                        const transformed_coord = proj4('EPSG:3857', 'WGS84', coord)
                        transformed_polygon.push(transformed_coord)
                    }
                    result.push(transformed_polygon)
                }
            }
            return result
        },
        requestValuesByProcess(process: ObjectProcess, group: ObjectGroup) {

            const data = {
                "layer_id": process.area_layer,
                "process_type": process.type,
                "process_sub_type": process.sub_type,
                "time_steps": [],
                "polygon": this.getGeometry(group)
            }

            axios({
                method: 'post',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                url: process_url,
                data: data
            })
            .then((response) => {
                let content = []
                for (const [i] of response.data['values'].entries()) {
                    content.push(response.data['time_steps'][i] + " " + response.data['values'][i])
                }
                group['report'].push({
                    'label': process.label,
                    'values': content
                })
            })
            .finally(() => {
                this.loadingCount++
                if (this.loadingCount === (this.dialogObject.processes.length * this.groups.length)) {
                    this.isLoading = false
                }
            })
        }
    }
})