from django.test import TestCase    # noqa
from unittest import mock

from django.contrib.auth.models import Group    # noqa

from main.management.commands._lib import File  # noqa
from main.management.commands._api import GeoServerApi  # noqa
from main.models import Bucket  # noqa

class S3FileMock:
    def __init__(self, file_name):
        self._object_name = file_name


class TestFile(TestCase):

    @classmethod
    def setUpClass(cls):
        # call super
        super(TestFile, cls).setUpClass()

        # get test bucket for all testing
        cls.super_test_bucket = Bucket.objects.create(
            name="super-test-bucket",
            group_id=Group.objects.create(name="test-group").id,
        )

        # wms file suffix
        cls.super_test_bucket.wms_file_suffix = "_wms"

        # wfs file suffix
        cls.super_test_bucket.wfs_file_suffix = "_wfs"


    def test_is_netcdf(self):
        netcdf_file = File(TestFile.super_test_bucket, "netcdf.nc")
        
        wrong_format = File(TestFile.super_test_bucket, "wrong.format")
        wrong_format2 = File(TestFile.super_test_bucket, "drought.NC")
        wrong_format3 = File(TestFile.super_test_bucket, "netcdf.netcdf")
        wrong_format4 = File(TestFile.super_test_bucket, "oli.Netcdf")
        no_format = File(TestFile.super_test_bucket, "no_extension")
        
        self.assertTrue(netcdf_file.is_netcdf())
        self.assertFalse(wrong_format.is_netcdf())
        self.assertFalse(wrong_format2.is_netcdf())
        self.assertFalse(wrong_format3.is_netcdf())
        self.assertFalse(wrong_format4.is_netcdf())
        self.assertFalse(no_format.is_netcdf())


    def test_is_geotiff(self):
        geotiff_file = File(TestFile.super_test_bucket, "file.tif")
        geotiff_multi_file = File(TestFile.super_test_bucket, "*.tif")
        wrong_format = File(TestFile.super_test_bucket, "wrong.format")
        nc_file = File(TestFile.super_test_bucket, "pp.nc")
        tiff_file = File(TestFile.super_test_bucket, "temp.tiff")
        tiff_file2 = File(TestFile.super_test_bucket, "temp.TIF")
        no_tif_file = File(TestFile.super_test_bucket, "")
        no_format = File(TestFile.super_test_bucket, "no_extension")
        
        self.assertTrue(geotiff_file.is_geotiff())
        self.assertFalse(geotiff_multi_file.is_geotiff())
        self.assertFalse(wrong_format.is_geotiff())
        self.assertFalse(nc_file.is_geotiff())
        self.assertFalse(tiff_file.is_geotiff())
        self.assertFalse(tiff_file2.is_geotiff())
        self.assertFalse(no_tif_file.is_geotiff())
        self.assertFalse(no_format.is_geotiff())

    def test_is_csv(self):
        csv_file = File(TestFile.super_test_bucket, "file.csv")
        wrong_format = File(TestFile.super_test_bucket, "drought.nc")
        no_file = File(TestFile.super_test_bucket, "")
        any_format = File(TestFile.super_test_bucket, "pp.jpg")
        no_format = File(TestFile.super_test_bucket, "no_extension")

        self.assertTrue(csv_file.is_csv())
        self.assertFalse(wrong_format.is_csv())
        self.assertFalse(no_file.is_csv())
        self.assertFalse(any_format.is_csv())
        self.assertFalse(no_format.is_csv())


    def test_is_wms_file(self):
        netcdf = File(TestFile.super_test_bucket, "test_wms.nc")
        geotiff = File(TestFile.super_test_bucket, "test_wms.tif",)
        no_suffix = File(TestFile.super_test_bucket, "wrong.nc")
        test1 = File(TestFile.super_test_bucket, "test_wms.csv")
        test2 = File(TestFile.super_test_bucket, "test_wms")

        self.assertTrue(File.is_wms_file(netcdf) or File.is_wms_file(geotiff), "File is wms")
        self.assertFalse(File.is_wms_file(no_suffix), "File is not wms")
        self.assertFalse(File.is_wms_file(test1))
        self.assertFalse(File.is_wms_file(test2)) # error no extension


    def test_is_zip_file_for_import(self):
        zip_file = File(TestFile.super_test_bucket, "test.test_wfs.zip")
        self.assertTrue(zip_file.is_shapefile_for_import(), "File is zip")

        wrong_format = File(TestFile.super_test_bucket, "test.wrong.format")
        self.assertFalse(File.is_shapefile_for_import(wrong_format), "File is no zip")
        
        wrong_format = File(TestFile.super_test_bucket, "hola.zip")
        self.assertTrue(File.is_shapefile_for_import(wrong_format), "File is zip")
        
        shape = File(TestFile.super_test_bucket, "shapes_wfs.zip")
        self.assertTrue(shape.is_shapefile_for_import(), "File is zip")

        rar = File(TestFile.super_test_bucket, "shapes_wfs.rar")
        self.assertFalse(rar.is_shapefile_for_import(), "File is not zip") # Rar is not considered zip

        uppercase_zip = File(TestFile.super_test_bucket, "zipped_layers_wfs.ZIP") # ZIP in upper letters is considered a zip
        self.assertTrue(uppercase_zip.is_shapefile_for_import(), "File is zip")

        no_extension = File(TestFile.super_test_bucket, "no_extension")
        self.assertFalse(no_extension.is_shapefile_for_import(), "File is not zip")

        csv_zip = File(TestFile.super_test_bucket, "layer_wfs.csv.zip")
        self.assertTrue(csv_zip.is_shapefile_for_import(), "File is zip")

        txt = File(TestFile.super_test_bucket, "info_wfs.txt")
        self.assertFalse(txt.is_shapefile_for_import(), "File is not zip")


    def test_get_unique_store_name(self):

        api_mock = mock.Mock(spec=GeoServerApi)
        api_mock.is_coveragestore_existing.side_effect = [True, False]

        nc_file = File(TestFile.super_test_bucket, "drought.nc")
        self.assertEqual(nc_file.get_unique_store_name(api_mock), "drought_B")
        
        api_mock.is_coveragestore_existing.side_effect = [False]
        cog_file = File(TestFile.super_test_bucket, "pp.tif")
        self.assertEqual(cog_file.get_unique_store_name(api_mock), "pp")

        api_mock.is_coveragestore_existing.side_effect = [True, False]
        long_nc = File(TestFile.super_test_bucket, "Qrouted/daily_sum/year/rel_change/rcp45/ensmedian/germany/facc100/Qrouted@hjahdiwnakslala.nc")
        self.assertEqual(long_nc.get_unique_store_name(api_mock), "QroutedDailySumYearRelChangeRcp45EnsmediaGermanyFacc100Qroute_B")
        # 63 characters
        
        api_mock.is_coveragestore_existing.side_effect = [False]
        funny_tif = File(TestFile.super_test_bucket, "Hola_me-llamo/N@t@li@_un_gusto.tif")
        self.assertEqual(funny_tif.get_unique_store_name(api_mock), "hola_me_llamo_n@t@li@_un_gusto")
        # Replaced "-" and "/" with "_", and everything lower case

        api_mock.is_coveragestore_existing.side_effect = [False]
        long_tif = File(TestFile.super_test_bucket, "Hola_me-llamo/N@t@li@_un_gusto.78jdhdjskalnd_kj.ksn.980.plj0.yaquiero.986hacabar!!!!.tif")
        self.assertEqual(long_tif.get_unique_store_name(api_mock), "HolaLlamoN@t@li@Gusto78jdhdjsKsn980Plj0Yaquiero986hacab")
        # It returned a store name with 63 characters

        api_mock.is_coveragestore_existing.side_effect = [True, True, True, True, False]
        nc_file = File(TestFile.super_test_bucket, "drought.nc")
        self.assertEqual(nc_file.get_unique_store_name(api_mock), "drought_E")

        # Try with file name that have dates, cause the unique_store_name function takes out dates from store names
        names = [
            "DSWI_201602_Germany_test_cog.tif",
            "DSWI-14-07-2021_Germany_test_cog.tif",
            "DSWI_20240930_Germany_test_cog.tif",
            "DSWI_2024-09-30_Germany_test_cog.tif"
        ]
        for name in names:
            api_mock.is_coveragestore_existing.side_effect = [False]
            date_tif2 = File(TestFile.super_test_bucket, name)
            self.assertEqual(date_tif2.get_unique_store_name(api_mock), "dswi_germany_test")

        api_mock.is_coveragestore_existing.side_effect = [False]
        date_tif5 = File(TestFile.super_test_bucket, "waldzustandsmonitor/data/test/Forest_condition_v0004-0005_Germany_032016_full_bmeanw_R20m_cog_032016.tif")
        self.assertEqual(date_tif5.get_unique_store_name(api_mock), "WaldzustDataTestForestConditioV00040005GermanyFullBmeanwR20m")
