import secrets
import string

from datetime import datetime

from django import forms # noqa
from django.urls import reverse # noqa
from django.utils.html import format_html  # noqa

from django.contrib import admin, messages # noqa
from django_tabbed_changeform_admin.admin import DjangoTabbedChangeformAdmin # noqa
from django.core.exceptions import ValidationError # noqa

from main.models import Bucket # noqa
from main.lib.backend import get_user_groups # noqa

from ..models import MetadataRecord, MetadataRecordAuthor


class AuthorInlineFormset(forms.BaseInlineFormSet):
    model = MetadataRecordAuthor

    def clean(self):
        count_authors = 0
        for form in self.forms:
            cleaned_data = form.cleaned_data

            if cleaned_data == {}:
                continue    # form is empty and will not be saved. Skip validation.

            if not cleaned_data.get('DELETE', False):
                count_authors += 1

            if count_authors == 0:
                raise ValidationError("A Metadata-Record should have at least one assigned author.")


class AuthorInline(admin.StackedInline):
    model = MetadataRecordAuthor
    formset = AuthorInlineFormset
    min_num = 1
    extra = 0
    classes = ['tab-author-inline']
    autocomplete_fields = ['author']


class MetadataRecordAdmin(DjangoTabbedChangeformAdmin, admin.ModelAdmin):
    model = MetadataRecord
    inlines = [AuthorInline]
    list_display = ('title', 'bucket', 'file_path', 'download', 'is_published')


    def get_form(self, request, obj=None, **kwargs):
        form = super().get_form(request, obj, **kwargs)
        # if form for creating new MetadataRecord, set placeholder, otherwise make it readonly (get_readonly_fields)

        placeholders = (
            ('version', 'e.g. 1.0'),
            ('keywords', 'e.g. Drought, Precipitation, Campaign-name'),
        )

        for item in placeholders:
            if not obj:
                form.base_fields[item[0]].widget.attrs['placeholder'] = item[1]
                form.base_fields['file_path'].widget.attrs['placeholder'] = 'e.g. subfolder/metadata.jsonld'
            if obj:
                if not obj.version:
                    form.base_fields[item[0]].widget.attrs['placeholder'] = item[1]

        if "keywords" in form.base_fields:
            form.base_fields["keywords"].widget.attrs.update(size="60")
        return form

    fieldsets = [
        (None, {
            'fields': ('bucket', 'file_path', 'dataset_path', 'geonetwork_uuid', 'record_id'),
            'classes': ('tab-basic',),
        }),
        (None, {
            'fields': ('title', 'abstract', 'language', 'format', 'keywords'),
            'classes': ('tab-abstract',),
        }),
        (None, {
            'fields': ('publisher', 'license', 'version'),
            'classes': ('tab-legal',),
        }),
        (None, {
            'fields': ('created_at', 'updated_at', 'published_at'),
            'classes': ('tab-dates',),
        }),
        (None, {
            'fields': ('is_published', 'is_doi_requested', ('is_doi_check_passed', 'logs'), 'is_doi_approved', 'doi'),
            'classes': ('tab-publication',),
        }),
    ]

    tabs = [
        ("Basic Information", ["tab-basic"]),
        ("Data Description", ["tab-abstract"]),
        ("Author Information", ["tab-author-inline"]),
        ("Legal Information", ["tab-legal"]),
        ("Publication", ["tab-publication"]),
        ("Dates", ["tab-dates"]),
    ]


    def has_change_permission(self, request, obj=None):
        if obj:
            if obj.doi or obj.is_delete_scheduled:
                return False
        return True


    def has_delete_permission(self, request, obj=None):
        if obj:
            if obj.doi or obj.is_delete_scheduled:
                return False
        return True


    def get_readonly_fields(self, request, obj):
        fields = ['dataset_path', 'geonetwork_uuid', 'record_id', 'doi', 'created_at', 'updated_at', 'published_at',
                  'is_doi_check_passed', 'logs']

        if not request.user.is_superuser:
            fields.append('is_doi_approved', )

        if not obj:
            fields.extend(['is_doi_requested', 'is_doi_approved'])

        if obj:
            fields.extend(['bucket', 'file_path'])

            if obj.published_at and (obj.is_doi_requested or obj.is_doi_approved):
                fields.append('is_published')

            if not obj.is_doi_requested or not obj.is_doi_check_passed:
                fields.append('is_doi_approved')

            if not obj.is_published or obj.is_doi_requested:
                fields.append('is_doi_requested')

        return fields


    def formfield_for_foreignkey(self, db_field, request, **kwargs):

        if not request.user.is_superuser:
            # if not superuser, a user can only select his buckets
            if db_field.name == "bucket":
                kwargs["queryset"] = Bucket.objects.filter(group__in=get_user_groups(request))

        return super().formfield_for_foreignkey(db_field, request, **kwargs)


    def get_queryset(self, request):
        # if not superuser, a user can see only see MetadataRecords from his buckets
        qs = super().get_queryset(request)

        if request.user.is_superuser:
            return qs
        return qs.filter(bucket__group__in=get_user_groups(request))


    def save_model(self, request, obj: Bucket, form, change):
        if obj.is_published:
            obj.published_at = datetime.now()
        super().save_model(request, obj, form, change)


    def save_related(self, request, form, formset, change):

        super(MetadataRecordAdmin, self).save_related(request, form, formset, change)

        obj = form.instance
        obj.dataset_path = get_folder_name(obj.file_path)

        if not obj.record_id:
            obj.record_id = create_file_identifier()

        if not obj.dataset_path:
            obj.dataset_path = get_folder_name(obj.file_path)

        if not obj.is_published and obj.doi:
            obj.is_published = True

        if obj.is_published and not obj.published_at:
            obj.published_at = datetime.now()

        obj.is_sync_scheduled = True

        super().save_model(request, obj, form, change)


    def delete_model(self, request, obj):
        obj.is_delete_scheduled = True
        obj.save()


    def download(self, obj):
        link = reverse('adminDownloadOverview', kwargs={'record_id': obj.record_id})
        return format_html('<a target="_blank" href="{}">link &#x2197;</a>'.format(link))


def get_folder_name(path: str):
    path = path.strip('/')
    pieces = path.split('/')

    if len(pieces) > 1:
        del pieces[-1]
        return '/'.join(pieces)
    else:
        return ''


def create_file_identifier():
    chars = string.ascii_letters + string.digits
    result = ''

    for _ in range(32):
        result += secrets.choice(chars)

    return result
