import os
import requests

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry  # noqa

from jinja2 import Template
from django.shortcuts import get_object_or_404  # noqa
from django.http import HttpResponse, JsonResponse    # noqa
from main.models import WfsLayer    # noqa


def get_time_records(wfs_layer: WfsLayer, timeseries_id: int):

    url = """
        {url}/{workspace}/ows?service=WFS
        &version=2.0.0
        &request=GetFeature
        &typeName={workspace}%3A{data_store_name}_timerecord
        &outputFormat=application%2Fjson
        &CQL_FILTER=timeseries_id%20=%20{timeseries_id}
        &propertyName=value_number,date
        &sortBy=date+A
    """.format(url=wfs_layer.url, workspace=wfs_layer.workspace, data_store_name=wfs_layer.data_store_name, timeseries_id=timeseries_id)

    return requests.get(url, timeout=(5, 15))


def wfs_features(request, layer_id):

    wfs_layer = get_object_or_404(WfsLayer, id=layer_id)

    data = _create_request_body(request, wfs_layer.workspace, wfs_layer.data_store_name)

    url = '{}/{}/ows?service=WFS&version=2.0.0&outputFormat=application%2Fjson&count=1'.format(wfs_layer.url, wfs_layer.workspace)

    retry_strategy = Retry(
        total=3,
        backoff_factor=2,
        allowed_methods=['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PUT', 'POST']
    )

    try:
        adapter = HTTPAdapter(max_retries=retry_strategy)

        session = requests.Session()
        session.mount("https://", adapter)
        session.mount("http://", adapter)

        response = session.post(url, data=data, timeout=30)

        return HttpResponse(response, content_type='application/json')
    except Exception:
        return HttpResponse("Cannot request WFS {}".format(url), status=500)


def _create_request_body(request, workspace, data_store_name):

    count = request.GET.get('count')
    # bottomLeftLon, bottomLeftLat, topRightLon, topRightLat
    bbox = request.GET.get('bbox')
    coords = bbox.split(',')

    property_id = request.GET.get('property')
    begin_date = request.GET.get('beginDate')
    end_date = request.GET.get('endDate')
    threshold_value = request.GET.get('threshold')

    if property_id:
        filename = '/wfs/filter_request.xml'
    else:
        filename = '/wfs/standard_request.xml'

    with open(os.path.dirname(__file__) + filename, encoding='utf-8') as xml_file:

        template = Template(xml_file.read())

        return template.render(
            workspace=workspace,
            data_store_name=data_store_name,
            count=count if count else 1000,
            lowerCornerLon=float(coords[0]),
            lowerCornerLat=float(coords[1]),
            upperCornerLon=float(coords[2]),
            upperCornerLat=float(coords[3]),
            property_id=property_id,
            begin_date=begin_date,
            end_date=end_date,
            threshold_value=threshold_value
        )
