
from main.models import StaLayer, StaObservedPropertyFilter, StaThingProperty  # noqa


class RequestBuilder:

    def __init__(self, request):
        self.request = request
        self.filters = []

        self.obs_property_id = self.request.GET.get("property")
        self.threshold = request.GET.get("threshold")
        self.begin_date = request.GET.get("beginDate")
        self.end_date = request.GET.get("endDate")
        self.max_results = request.GET.get("count")
        self.bbox = request.GET.get("bbox")

        self.basic_url = ""
        self.bbox_url = None



    def build_thing_queries(self, layer: StaLayer) -> None:

        if self.obs_property_id:
            url = self._locations_by_property()

            self.filters.append("Datastreams/ObservedProperty/id eq {}".format(self.obs_property_id))
        else:
            url = self._locations_by_properties()

            self._add_property_filters(layer, "Datastreams/ObservedProperty/")

        self._add_thing_filters(layer, "")

        if self.max_results is not None:
            url += "&$top={}".format(self.max_results)

        if self.threshold:
            self.filters.append("Datastreams/Observations/result gt {}".format(self.threshold))
        if self.begin_date:
            self.filters.append("Datastreams/Observations/resultTime gt {}T00:00:00%2B01:00".format(self.begin_date))
        if self.end_date:
            self.filters.append("Datastreams/Observations/resultTime lt {}T00:00:00%2B01:00".format(self.end_date))

        if len(self.filters) > 0:
            self.basic_url = url + "&$filter=" + " and ".join(self.filters)
        else:
            self.basic_url = url

        if self.bbox:
            bbox_filter = _get_bbox_filter(self.bbox)
            if filter != "":
                self.filters.append(bbox_filter)
                self.bbox_url = url + "&$filter=" + " and ".join(self.filters)



    def _locations_by_property(self) -> str:
        return "".join([
            "v1.1/Things?",
            "$expand=",
            "Locations($select=location)",
            ",Datastreams(",
                "$filter=ObservedProperty/id eq {};",
                "$top=1;",
                "$orderBy=resultTime desc;",
                "$expand=Observations($select=result;$top=1;$orderBy=resultTime desc)",
            ")",
            "&$count=true",
            "&$select=id",
            "&$resultFormat=GeoJSON"
        ]).format(self.obs_property_id)



    def _locations_by_properties(self) -> str:
        """
        do not request a single observed property, but filter by a category in ObservedProperty-properties
        """

        return "v1.1/Things?$expand=Locations($select=location)&$count=true&$select=id&$resultFormat=GeoJSON"



    def get_property_query(self, layer: StaLayer) -> str:
        url = "v1.1/ObservedProperties?$select=id,name,properties&$orderBy=name"

        self._add_thing_filters(layer, "Datastreams/Thing/")
        self._add_property_filters(layer, "")

        if len(self.filters) > 0:
            url += "&$filter=" + " and ".join(self.filters)

        return url



    def get_property_query_by_thing(self, layer: StaLayer, thing_id: int) -> str:
        url = "v1.1/ObservedProperties?$orderBy=name"

        self.filters.append("Datastreams/Thing/id eq {}".format(thing_id))

        self._add_property_filters(layer, "")

        url += "&$filter=" + " and ".join(self.filters)

        return url



    def _add_thing_filters(self, layer: StaLayer, query_path: str) -> None:
        for thing_filter in StaThingProperty.objects.filter(endpoint=layer.endpoint, apply_as_filter=True):

            self.filters.append("substringof('{}',{}properties/{})".format(
                thing_filter.property_value,
                query_path,
                thing_filter.property_key)
            )



    def _add_property_filters(self, layer: StaLayer, query_path: str) -> None:
        for obs_filter in StaObservedPropertyFilter.objects.filter(layer=layer):
            self.filters.append("substringof('{}',{}{})".format(
                obs_filter.property_value,
                query_path,
                obs_filter.property_key)
            )



def _get_bbox_filter(bbox: str) -> str:
    """
    get Bounding-Box-FilterParameter for STA-Query

    the param 'bbox' needs to contain following values in this order:

        bottomLeftLon, bottomLeftLat, topRightLon, topRightLat
    """

    params = bbox.split(",")

    top_lon = params[0]
    top_lat = params[3]
    bottom_lon = params[2]
    bottom_lat = params[1]

    if top_lon and top_lat and bottom_lon and bottom_lat:
        # define polygon in WKT-representation
        b_r_corner = "{} {}".format(bottom_lon, bottom_lat)
        t_r_corner = "{} {}".format(bottom_lon, top_lat)
        t_l_corner = "{} {}".format(top_lon, top_lat)
        b_l_corner = "{} {}".format(top_lon, bottom_lat)

        # define counter-clock-wise with coords as lon-lat
        polygon = "{}, {}, {}, {}, {}".format(b_r_corner, t_r_corner, t_l_corner, b_l_corner, b_r_corner)

        return "st_within(Locations/location, geography'POLYGON (({}))')".format(polygon)

    return ""