import time
import datetime

from django.core import management # noqa
from django.core.management.base import BaseCommand # noqa

from data_import.models import CsvImportJob, STA, WFS # noqa

from main.lib.s3_utils import get_s3_filesystem, get_minio_client # noqa

from ...models import Bucket, BucketLog, GeojsonLayer, WmsLayer
from ._lib import File, FileCollection, GeotiffHandler, ImagemosaicHandler, NetcdfHandler, ShapefileHandler
from ._api import GeoServerApi


class Command(BaseCommand):

    def handle(self, *args, **options): # noqa

        minio_client = get_minio_client()
        fs = get_s3_filesystem()
        gs_api = GeoServerApi()

        while True:

            for bucket in Bucket.objects.all():

                # check if bucket is scheduled for updates
                start_time = datetime.datetime.now()

                if bucket.is_update_running or not bucket.is_time_for_automated_update(start_time):
                    continue

                # instantiate all necessary service-classes (and reset their logs)

                nc_handler = NetcdfHandler(gs_api, fs, minio_client)
                tif_handler = GeotiffHandler(gs_api, fs, minio_client)
                shp_handler = ShapefileHandler(gs_api, fs, minio_client)
                mosaic_handler = ImagemosaicHandler(gs_api, fs, minio_client)

                log = BucketLog(
                    bucket=bucket,
                    start_time=start_time
                )

                bucket.is_update_running = True
                bucket.save()

                # get list of files
                bucket_files = FileCollection(bucket.name)

                # create Layers and Datastores in GeoServer

                for s3_file in bucket_files.collection:

                    file = File.create_by_s3_obj(bucket, s3_file)

                    if bucket.wms_is_integration_active and file.is_wms_file():
                        if file.is_netcdf():
                            nc_handler.create_layer(file)
                        elif file.is_geotiff():

                            files_in_directory = FileCollection(file.bucket.name, file.get_path())
                            date_list = files_in_directory.extract_date_list(bucket)

                            if files_in_directory.are_all_with_extension(".tif") and date_list:
                                if mosaic_handler.is_mosaic_created(file):
                                    mosaic_handler.create_granule(file, date_list)
                                else:
                                    mosaic_handler.create_imagemosaic_layer(file)
                            else:
                                tif_handler.create_layer(file)

                    if bucket.shp_is_integration_active:
                        if file.is_shapefile_for_import():
                            shp_handler.create_layer(file)

                    # create CsvImportJobs (for parsing + importing later)

                    if file.is_csv():
                        if bucket.wfs_is_integration_active:
                            is_created = create_import_job_if_not_exists(file, WFS)
                            if is_created:
                                log.add("Create " + WFS + "-ImportJob: " + file.relative_path)
                        if bucket.sta_is_integration_active:
                            is_created = create_import_job_if_not_exists(file, STA)
                            if is_created:
                                log.add("Create " + STA + "-ImportJob: " + file.relative_path)

                    del file

                log.add_list(nc_handler.logs)
                log.add_list(tif_handler.logs)
                log.add_list(shp_handler.logs)

                # Delete Layers and Datastores

                if bucket.wms_delete_integrated_layers_if_file_not_longer_exists:
                    for layer in WmsLayer.objects.filter(bucket=bucket):
                        file = File.create_by_absolute_path(bucket, layer.file_path)

                        if not bucket_files.contains(file):
                            if file.is_netcdf():
                                nc_handler.delete_layer(file)
                            if file.is_geotiff():
                                tif_handler.delete_layer(file)
                            layer.delete()
                            log.add("Deleted Layer and Datastore: " + file.relative_path)

                if bucket.shp_delete_integrated_layers_if_file_not_longer_exists:
                    for layer in GeojsonLayer.objects.filter(bucket=bucket):
                        file = File.create_by_absolute_path(bucket, layer.file_path)
                        if not bucket_files.contains(file):
                            shp_handler.delete_layer(file)
                            layer.delete()
                            log.add("Layer and Datastore deleted for: " + file.relative_path)

                # finish worker run

                if bucket.update_interval:
                    bucket.next_update = start_time + datetime.timedelta(minutes=bucket.update_interval)
                else:
                    bucket.next_update = None

                bucket.is_update_running = False

                log.time_finished = datetime.datetime.now()

                log.save()
                bucket.save()

                print("finished worker-run")

            # parse one CSV-file and persist data to PostGIS database (and create PostGIS-Datastore)
            management.call_command("parsecsv")

            management.call_command("createlegends")

            management.call_command("createmetadata")

            # wait until next run
            time.sleep(30)


def create_import_job_if_not_exists(file: File, target: str):
    try:
        CsvImportJob.objects.get(bucket=file.bucket, s3_file=file.relative_path, target=target)
        return False
    except CsvImportJob.DoesNotExist:
        job = CsvImportJob(
            bucket=file.bucket,
            s3_file=file.relative_path,
            target=target,
            file_size=file.size
        )
        job.save()
        return True
