import os
import datetime
from jinja2 import Template

from django.core.management.base import BaseCommand # noqa

from ...models import WmsColor, WmsLayerLegend, WmsLegend
from ._api import GeoServerApi


class Command(BaseCommand):

    def handle(self, *args, **options):

        legends = WmsLegend.objects.filter(is_sync_scheduled=True)

        if len(legends) == 0:
            return

        api = GeoServerApi()

        for legend in legends:
            try:
                sld = create_sld_file(legend, "de")
                api.create_style(legend.name_de, sld)

                sld = create_sld_file(legend, "en")
                api.create_style(legend.name_en, sld)

                legend.sync_log = 'Legend successfully synced ({})'.format(datetime.datetime.now().strftime("%d.%m.%Y %H:%M:%S"))

            except Exception as e:
                legend.sync_log = 'Error: ' + str(e)

            legend.is_sync_scheduled = False
            legend.save()

            try:
                layer_legends = WmsLayerLegend.objects.filter(legend=legend).select_related('wms_layer').order_by('position')
                for index, layer_legend in enumerate(layer_legends):
                    layer_name = layer_legend.wms_layer.layer_name

                    if index == 0:
                        api.add_style(layer_legends[0].legend.name_de, layer_name, True)
                        api.add_style(layer_legends[0].legend.name_en, layer_name, True)
                    else:
                        api.add_style(layer_legend.legend.name_de, layer_name)
                        api.add_style(layer_legend.legend.name_en, layer_name)

                    if layer_legend.wms_layer.store_in_geowebcache:
                        api.modify_gwc(layer_name, layer_legend.legend.name_de, "seed")
                        api.modify_gwc(layer_name, layer_legend.legend.name_en, "seed")

            except Exception as e:
                legend.sync_log = legend.sync_log + '\nError: ' + str(e)
                legend.save()


def create_sld_file(legend: WmsLegend, locale: str):

    with open(os.path.dirname(__file__) + '/_api/templates/legend_template.xml', encoding='utf-8') as template:
        template = Template(template.read())
    colors = []

    for color in WmsColor.objects.filter(legend=legend).order_by('position'):

        color_name = color.name_en if (locale == 'en') else color.name_de
        colors.append({
            'color': color.color,
            'upper_limit': color.upper_limit,
            'name': color_name,
            'is_invisible': color.is_invisible
        })

    legend_name = legend.name_en if (locale == 'en') else legend.name_de

    return template.render(
        name=legend_name,
        title=legend_name,
        legend_type=legend.type,
        colors=colors
    )
