import json
from datetime import datetime

from metadata.models import MetadataRecord, MetadataRecordAuthor, Organization # noqa


def create_jsonld_content(record: MetadataRecord):
    jsonld_formatted = {
        "@context": {
            "@vocab": "https://schema.org/",
        },
        "@type": "Dataset",
        "version": record.version if record.version else "0.1",
        "inLanguage": record.language,
        "encodingFormat": record.format.name,
        "license": record.license.name,
        "name": record.title,
        "identifier": _get_identifier_as_jsonld(record),
        "dateCreated": record.created_at.strftime("%Y-%m-%d"),
        "dateModified": record.updated_at.strftime("%Y-%m-%d"),
        "datePublished": _get_date_published(record),
        "author": _get_authors_as_jsonld(record),
        "publisher": _get_organization_as_jsonld(record.publisher),
        "abstract": record.abstract,
        "keywords": record.keywords if record.keywords else [],
    }
    return json.dumps(jsonld_formatted, indent=4)


def _get_date_published(record: MetadataRecord):
    if not record.published_at and record.is_published:
        return datetime.today().strftime("%Y-%m-%d")
    if record.published_at:
        return record.published_at.strftime("%Y-%m-%d")
    else:
        return None


def _get_identifier_as_jsonld(record: MetadataRecord):
    identifier = [
        {
            "@type": "PropertyValue",
            "propertyID": "DOI",
            "value": record.doi if record.doi else ""
        },
        {
            "@type": "PropertyValue",
            "propertyID": "UUID",
            "value": str(record.geonetwork_uuid) if record.geonetwork_uuid else ""
        }
    ]
    return identifier


def _get_authors_as_jsonld(record: MetadataRecord):
    authors = []
    record_authors = MetadataRecordAuthor.objects.filter(record=record)
    for author in record_authors:
        author = {
            "givenName": author.author.given_name,
            "familyName": author.author.family_name,
            "email": author.author.email,
            "identifier": {
                "@type": "PropertyValue",
                "name": "ORCID",
                "value": author.author.orcid
            },
            "affiliation": _get_organization_as_jsonld(author.author.affiliation)
        }
        authors.append(author)
    return authors


def _get_organization_as_jsonld(publisher):
    if publisher:
        return {
            "@type": "Organization",
            "name": publisher.name if publisher.name else "",
            "identifier": {
                "@type": "PropertyValue",
                "name": "ROR",
                "value": publisher.ror if publisher.ror else ""
            }
        }
    return {
        "@type": "Organization",
        "name": "",
        "identifier": {
            "@type": "PropertyValue",
            "name": "ROR",
            "value": ""
        }
    }
