import os
import tempfile
import uuid
import zipfile

from main.models import GeojsonLayer  # noqa
from main.lib.s3_utils import upload_file  # noqa

from . import File, Handler

MAX_FILE_SIZE = 40000000  # 400MB


class ShapefileHandler(Handler):

    def create_layer(self, file: File):

        if self.is_file_too_large(file, MAX_FILE_SIZE):
            return

        is_file_new = True
        for layer in GeojsonLayer.objects.filter(file_path=file.relative_path):
            if layer.last_upload and not file.is_newer_than(layer.last_upload):
                is_file_new = False

        if is_file_new:
            self._process_shapefile(file)
        else:
            self.log('No update for ' + file.absolute_path)


    def delete_layer(self, file: File):
        for layer in GeojsonLayer.objects.filter(bucket=file.bucket):
            self.api.delete_datastore(file.bucket.name, layer.get_store_name(), 'datastores')


    def _process_shapefile(self, file: File):

        with tempfile.TemporaryDirectory() as tmpdir:
            print(f"⏬ Downloading from: {file.absolute_path}")

            zip_path = os.path.join(tmpdir, "temp.zip")

            self.fs.get_file(file.absolute_path, zip_path)

            print("📦 Extracting ZIP...")
            with zipfile.ZipFile(zip_path, "r") as zip_ref:
                zip_ref.extractall(tmpdir)

            shp_files = [
                os.path.join(tmpdir, f)
                for f in os.listdir(tmpdir)
                if f.endswith(".shp")
            ]

            if not shp_files:
                print("❌ No .shp found in the ZIP.")

            for shp_path in shp_files:

                shp_name = get_shp_name(shp_path)

                workspace = file.bucket.name

                try:
                    layer = GeojsonLayer.objects.get(file_path=file.relative_path, shp_filename=shp_name)

                except GeojsonLayer.DoesNotExist:

                    store_name = file.get_unique_store_name(self.api, shp_name)
                    layer_name = f"{workspace}:{store_name}"

                    url = '{}/ows?service=WFS&version=2.0.0&request=GetFeature&typename={}&outputFormat=application/json&srsname=EPSG:3857'.format(
                        self.get_geoserver_url(workspace),
                        layer_name
                    )

                    layer = GeojsonLayer(
                        bucket=file.bucket,
                        file_path=file.relative_path,
                        url=url,
                        shp_filename=shp_name,
                        store_uuid=str(uuid.uuid4()),
                    )
                    self.log('Create Layer: ' + layer_name)

                fgb_path = os.path.join(tmpdir, f"shapefile.fgb")

                print(f"🛠️ Converting extracted shapefile...")
                self.create_flatgeobuf(shp_path, fgb_path)

                destination_key = f"public/{layer.store_uuid}/shapefile.fgb"

                print(f"☁️ Uploading to: {self.get_cog_bucket_name()}/{destination_key}")
                is_success = upload_file(self.get_cog_bucket_name(), destination_key, fgb_path)

                if is_success:
                    upload_url = os.environ.get("MINIO_ENDPOINT") + f"/{self.get_cog_bucket_name()}/{destination_key}"
                    print(f"✅ Upload successful: {upload_url}")

                    self.api.create_flatgeobuf_store(workspace, layer.get_store_name(), upload_url)

                    layer.last_upload = file.last_modified
                    layer.save()
                else:
                    print(f"❌ Upload failed!")


def get_shp_name(shp_path: str):
    result = shp_path.removesuffix(".shp").split('/')[-1]
    for char in [".", "-"]:
        result = result.replace(char, "_")
    return result.lower()
