import uuid

from main.models import WmsLayer    # noqa

from . import File, FileCollection, GeotiffHandler


class ImagemosaicHandler(GeotiffHandler):

    def __init__(self, gs_api, fs, minio_client):
        super().__init__(gs_api, fs, minio_client)

        self.mosaic_list = []


    def is_mosaic_created(self, file: File):
        if file.get_path() in self.mosaic_list:  # TODO
            return True
        return False


    def create_imagemosaic_layer(self, file: File):

        self.mosaic_list.append(file.get_path())

        variable = self.get_or_create_variable("band_data", file.bucket)

        workspace = file.bucket.name

        try:
            layer = self._get_layer(file)
            if layer.last_upload and file.is_newer_than(layer.last_upload):
                # self.log('No update for ' + file.absolute_path)
                return
            self.log('Add/Update Layer: ' + layer.layer_name)

        except WmsLayer.DoesNotExist:

            store_name = file.get_unique_store_name(self.api)
            layer_name = workspace + ':' + store_name

            layer = WmsLayer(
                wms_url=self.get_geoserver_url(workspace),
                scale_factor=1,
                layer_name=layer_name,
                variable=variable,
                file_path=file.get_imagemosaic_path(),
                bucket=file.bucket,
                store_uuid=uuid.uuid4()
            )
            self.log('Create Layer: ' + layer_name)

        if not self.api.is_coveragestore_existing(workspace, layer.get_store_name()):

            upload_url = self._process_geotiff(file, self._get_destination_key(layer, file.find_date(True)))

            self.api.create_imagemosaic(workspace, layer.get_store_name(), upload_url)
            self.log(f"[POST] Create store '{layer.get_store_name()}'.")

            layer.last_upload = file.last_modified
            layer.time_steps = self.api.get_wms_timesteps(layer)
            layer.save()


    def create_granule(self, file: File, mosaic_dates: list[str]):

        workspace = file.bucket.name

        layer = self._get_layer(file)
        store_name = layer.get_store_name()

        uploaded_files = FileCollection("geoserver", "public/" + str(layer.store_uuid) + "/")

        date = file.find_date(True)

        if date in uploaded_files.get_missing_dates(mosaic_dates):
            url = self._process_geotiff(file, self._get_destination_key(layer, date))
            self.api.add_granule(workspace, store_name, url)

        layer.last_upload = file.last_modified
        layer.time_steps = self.api.get_wms_timesteps(layer)
        layer.save()


    @staticmethod
    def _get_layer(file: File):
        return WmsLayer.objects.select_related('variable').get(file_path=file.get_imagemosaic_path())
