import os

from abc import ABC, abstractmethod
from osgeo import gdal  # noqa
from main.lib.backend import get_geoserver_url_for_frontend  # noqa
from main.models import Bucket, DataVariable, WmsLayer    # noqa
from main.lib.s3_utils import remove_file # noqa

from . import File

GEOSERVER_COG_BUCKET = "geoserver"


class Handler(ABC):

    def __init__(self, geoserver_api, fs, minio_client):
        self.api = geoserver_api
        self.fs = fs
        self.minio_client = minio_client
        self.logs = []

    @abstractmethod
    def create_layer(self, file: File):
        pass

    @staticmethod
    def get_cog_bucket_name():
        return GEOSERVER_COG_BUCKET


    @abstractmethod
    def delete_layer(self, file: File):
        pass


    def log(self, message: str):
        self.logs.append(message)

    def is_file_too_large(self, file: File, max_size: int):
        if file.size > max_size:
            self.log('Error - File is too large ({}): {}'.format(file.size, file.absolute_path))
            return True
        return False


    @staticmethod
    def reproject_file(input_file_path: str, output_file_path: str):
        os.environ["GDAL_NUM_THREADS"] = "ALL_CPUS"
        gdal.Warp(output_file_path, input_file_path, options=gdal.WarpOptions(dstSRS='EPSG:4326'))


    @staticmethod
    def create_cog(input_file_path: str, output_file_path: str):
        os.environ["GDAL_NUM_THREADS"] = "ALL_CPUS"

        gdal_ds = gdal.Open(input_file_path, gdal.GA_ReadOnly)
        gdal_ds.BuildOverviews("NEAREST", [2, 4, 8, 16, 32])

        gdal.Translate(
            output_file_path,
            gdal_ds,
            format="COG",
            creationOptions=[
                "COMPRESS=DEFLATE",
                "BLOCKSIZE=512",
                "BIGTIFF=YES",
                "NUM_THREADS=ALL_CPUS"
            ],
        )

        del gdal_ds


    @staticmethod
    def create_flatgeobuf(input_file_path: str, output_file_path: str):
        gdal.UseExceptions()
        gdal.VectorTranslate(output_file_path, input_file_path, format="FlatGeobuf")


    @staticmethod
    def get_geoserver_url(workspace):
        return get_geoserver_url_for_frontend() + '/' + workspace


    @staticmethod
    def get_or_create_variable(variable_name: str, bucket: Bucket):
        try:
            data_variable = DataVariable.objects.get(name=variable_name, group=bucket.group)
        except DataVariable.DoesNotExist:
            print("Create new Variable")
            data_variable = DataVariable(
                name=variable_name,
                group=bucket.group
            )
            data_variable.save()

        return data_variable


    @staticmethod
    def _get_destination_key(layer: WmsLayer, date: str=None):
        filename = "cog-" + date if date else "cog"
        return "public/" + str(layer.store_uuid) + "/" + filename + ".tif"
