import os
import tempfile
import uuid

import rasterio # noqa

from main.models import WmsLayer    # noqa
from main.lib.s3_utils import upload_file # noqa

from . import File, Handler

MAX_FILE_SIZE = 18000000000  # 18GB


class GeotiffHandler(Handler):

    def create_layer(self, file: File):

        if self.is_file_too_large(file, MAX_FILE_SIZE):
            return

        variable = self.get_or_create_variable("band_data", file.bucket)

        workspace = file.bucket.name

        try:
            layer = WmsLayer.objects.select_related('variable').get(file_path=file.absolute_path)
            if layer.last_upload and file.is_newer_than(layer.last_upload):
                self.log('Add/Update Layer: ' + layer.layer_name)
            else:
                # self.log('No update for ' + file.absolute_path)
                layer = None

        except WmsLayer.DoesNotExist:
            store_name = file.get_unique_store_name(self.api)
            layer_name = workspace + ':' + store_name

            layer = WmsLayer(
                wms_url=self.get_geoserver_url(workspace),
                scale_factor=1,
                layer_name=layer_name,
                variable=variable,
                file_path=file.absolute_path,
                bucket=file.bucket,
                store_uuid=uuid.uuid4()
            )
            self.log('Create Layer: ' + layer_name)

        if layer:

            upload_url = self._process_geotiff(file, self._get_destination_key(layer))
            if upload_url:
                self.api.create_cog_coveragestore(workspace, layer.get_store_name(), upload_url)

            with self.fs.open(file.absolute_path, 'rb') as s3_handle:
                src = rasterio.open(s3_handle)
                layer.no_data_value = self.get_no_data_value(src)

            layer.last_upload = file.last_modified
            layer.save()


    def delete_layer(self, file: File):
        for layer in WmsLayer.objects.filter(file_path=file.absolute_path):
            self.api.delete_datastore(file.bucket.name, layer.get_store_name())
            layer.delete()


    @staticmethod
    def get_no_data_value(src):
        try:
            if len(src.nodatavals) == 1:
                result = int(src.nodatavals[0])
                if isinstance(result, int):
                    return result
        except (TypeError, ValueError):
            pass
        return -9999


    def _process_geotiff(self, file: File, destination_key: str):
        # Create temp dir
        with tempfile.TemporaryDirectory() as tmp_dir:
            tif_path = os.path.join(tmp_dir, "temp.tif")
            proj_tif_path = os.path.join(tmp_dir, "projected_tif.tif")
            cog_path = os.path.join(tmp_dir, "final_cog.tif")

            self.fs.get_file(file.absolute_path, tif_path)

            with rasterio.open(tif_path) as src:
                if not self.is_valid_cog(src):
                    self.reproject_file(tif_path, proj_tif_path)
                    print("Reproject file with " + str(src.crs))
                else:
                    proj_tif_path = tif_path

            # Convert geotiff to COG
            self.create_cog(proj_tif_path, cog_path)

            is_success = upload_file(self.get_cog_bucket_name(), destination_key, cog_path)

            if is_success:
                upload_url = os.environ.get("MINIO_ENDPOINT") + f"/{self.get_cog_bucket_name()}/{destination_key}"
                print(f"✅ Upload successful: {upload_url}")

                return upload_url
            else:
                print(f"❌ Upload failed!")

        return None


    @staticmethod
    def is_valid_cog(src):
        driver = src.driver
        block_shapes = src.block_shapes
        compression = src.compression.value
        overviews = src.overviews(1)

        return (
            src.crs == 'EPSG:4326' and
            driver == "GTiff" and
            block_shapes == [(512, 512)] and
            compression == "DEFLATE" and
            overviews == [2, 4, 8, 16, 32]
        )
